/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  I have retained the CVS log from
--   this file before it was moved to the attic.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.3  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.7  2006/09/16 13:27:12  jsedwards
-- Added code to export the "class definition class" itself and to add it's
-- reference to it's own reference list.
--
-- Revision 1.1.2.6  2006/09/16 02:56:01  jsedwards
-- Add code to put reference list id in file.
--
-- Revision 1.1.2.5  2006/09/14 11:57:20  jsedwards
-- Changed type of variable from d to u in format string for time stamp.
--
-- Revision 1.1.2.4  2006/09/13 12:30:09  jsedwards
-- Added an else clause to catch unprocessed classes.
--
-- Revision 1.1.2.3  2006/09/11 11:27:03  jsedwards
-- Removed unused, ifdef'd out code.
--
-- Revision 1.1.2.2  2006/09/10 22:38:08  jsedwards
-- Added double quotes around area_code, phone_number and month values.
--
-- Revision 1.1.2.1  2006/09/10 20:48:51  jsedwards
-- Routines to export an old Alpha 0005 (a separate file for each object)
-- database into comma separated variable files (unencrypted).
--
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "old_multifile_objectify.h"

char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;

/* this is a kludge because I didn't originally put a reference to the class definition class in it's own reference list */
ObjRef old_class_definition_ref_list;


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_year_object_size(void* year_obj)
{
    assert(((C_struct_Year*)year_obj)->count > 0);
    return sizeof(C_struct_Year) + ((C_struct_Year*)year_obj)->count;
}


void fput_char(char c, FILE* fp)
{
    if (fputc(c, fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void fput_comma(FILE* fp)
{
    fput_char(',', fp);
}


void fput_uint8(uint8 n, FILE* fp)
{
    uint8 m;

    if (n > 99) fput_char('0' + (n / 100), fp);

    m = n % 100;

    if (m > 9) fput_char('0' + (m / 10), fp);

    fput_char('0' + (m % 10), fp);
}


void fput_newline(FILE* fp)
{
    fput_char('\n', fp);
}


void fput_hex4(uint8 hex[4], FILE* fp)
{
    int i;
    char hex_digit[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    if (fputs("0x", fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }

    for (i = 0; i < 4; i++)
    {
	if (fputc(hex_digit[hex[i] >> 4], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}

	if (fputc(hex_digit[hex[i] & 0xF], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}
    }
}


void fput_objref(ObjRef* ref, FILE* fp)
{
    fput_hex4(ref->id, fp);
}


void fput_timestamp(TimeStamp ts, FILE* fp)
{
    if (fprintf(fp, "\"%04u-%02u-%02u (%u) %02u:%02u:%02u.%06u\"",
		nwos_extract_year_from_time_stamp(ts),
		nwos_extract_month_from_time_stamp(ts),
		nwos_extract_day_of_month_from_time_stamp(ts),

		nwos_extract_day_of_week_from_time_stamp(ts),

		nwos_extract_hour_from_time_stamp(ts),
		nwos_extract_minute_from_time_stamp(ts),
		nwos_extract_second_from_time_stamp(ts),
		nwos_extract_microsecond_from_time_stamp(ts)) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void export_common_header(CommonHeader* ch, FILE* fp)
{
    int i;

    for (i = 0; i < 4; i++) fputc(ch->version[i], fp);

    fput_comma(fp);

    fput_hex4(ch->header_chksum, fp);

    fput_comma(fp);

    fput_hex4(ch->data_chksum, fp);

    fput_comma(fp);

    fput_timestamp(ch->creation_time, fp);

    fput_comma(fp);

    fput_objref(&ch->class_definition, fp);
}


void export_object_header(ObjectHeader* oh, FILE* fp)
{
    char* save_filename;
    size_t ref_list_size;
    ReferenceList* ref_list_ptr = NULL;
    int num_objects;
    int i;

    /* none of these should have been used yet */

    assert(is_void_reference(&oh->clone_of));
    assert(is_void_reference(&oh->context));
    assert(is_void_reference(&oh->created_by_user));
    assert(is_void_reference(&oh->created_by_app));
    assert(is_void_reference(&oh->prev_version));
    assert(is_void_reference(&oh->next_version));

    fput_objref(&oh->references, fp);

    if (!is_void_reference(&oh->references))
    {
	save_filename = current_filename;

	current_filename = "reference_list.csv";

	ref_list_size = old_reference_list_size(&oh->references);

	ref_list_ptr = malloc(ref_list_size);

	if (ref_list_ptr == NULL) 
	{
	    perror("reading reference list");
	    exit(1);
	}

	old_read_reference_list_from_disk(&oh->references, ref_list_ptr, ref_list_size);

	num_objects = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	fput_objref(&oh->references, ref_list_fp);

	fput_comma(ref_list_fp);

	export_common_header(&ref_list_ptr->common_header, ref_list_fp);

	/* Because I didn't originally put a reference to the class definition  */
	/* class in it's own reference list, this kludge adds it. */
	if (is_same_object(&oh->references, &old_class_definition_ref_list))
	{
	    fput_comma(ref_list_fp);

	    fput_objref(&old_class_definition_class_ref, ref_list_fp);
	}

	for (i = 0; i < num_objects; i++)
	{
	    fput_comma(ref_list_fp);

	    fput_objref(&ref_list_ptr->references[i], ref_list_fp);
	}

	fput_newline(ref_list_fp);

	free(ref_list_ptr);
	ref_list_ptr = NULL;

	current_filename = save_filename;
    }
}


void export_every_object(ObjRef* ref, EveryObject* eo, FILE* fp)
{
    fput_objref(ref, fp);

    fput_comma(fp);

    export_common_header(&eo->common, fp);

    fput_comma(fp);

    export_object_header(&eo->object, fp);

    fput_comma(fp);
}
    

void export_root(ObjRef* ref, FILE* fp)
{
    C_struct_Root root_obj;

    old_read_object_from_disk(ref, &root_obj, sizeof(root_obj));

    export_every_object(ref, &root_obj.header, fp);

    fput_objref(&root_obj.class_definition_class, fp);

    fput_newline(fp);
}
    

void export_class_definition(ObjRef* ref, FILE* fp)
{
    C_struct_class_definition class_def_obj;

    old_read_object_from_disk(ref, &class_def_obj, sizeof(class_def_obj));

    export_every_object(ref, &class_def_obj.header, fp);

    fput_objref(&class_def_obj.name, fp);

    fput_newline(fp);
}
    

void export_spelling(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*) kludge;
    int i;

    old_read_variable_sized_object_from_disk(ref, spelling_obj_ptr, get_spelling_object_size);

    export_every_object(ref, &spelling_obj_ptr->header, fp);

    fput_char('"', fp);

    for (i = 0; i < spelling_obj_ptr->count; i++)
    {
	fput_char(spelling_obj_ptr->storage[i], fp);
    }

    fput_char('"', fp);

    fput_newline(fp);
}
    

void export_name(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*) kludge;
    int i;

    old_read_variable_sized_object_from_disk(ref, name_obj_ptr, get_name_object_size);

    export_every_object(ref, &name_obj_ptr->header, fp);

    for (i = 0; i < name_obj_ptr->count; i++)
    {
	if (i > 0) fput_comma(fp);

	fput_objref(&name_obj_ptr->spelling[i], fp);
    }

    fput_newline(fp);
}
    

void export_word(ObjRef* ref, FILE* fp)
{
    C_struct_Word word_obj;

    old_read_object_from_disk(ref, &word_obj, sizeof(word_obj));

    export_every_object(ref, &word_obj.header, fp);

    fput_objref(&word_obj.thing, fp);

    fput_comma(fp);

    fput_objref(&word_obj.language, fp);

    fput_comma(fp);

    fput_objref(&word_obj.spelling, fp);

    fput_newline(fp);
}
    

void export_language(ObjRef* ref, FILE* fp)
{
    C_struct_Language language_obj;

    old_read_object_from_disk(ref, &language_obj, sizeof(language_obj));

    export_every_object(ref, &language_obj.header, fp);

    fput_objref(&language_obj.definition, fp);

    fput_newline(fp);
}
    

void export_us_state(ObjRef* ref, FILE* fp)
{
    C_struct_US_State state_obj;

    old_read_object_from_disk(ref, &state_obj, sizeof(state_obj));

    export_every_object(ref, &state_obj.header, fp);

    fput_objref(&state_obj.name, fp);

    fput_comma(fp);

    fput_objref(&state_obj.date, fp);

    fput_comma(fp);

    fput_objref(&state_obj.capital, fp);

    fput_comma(fp);

    fput_objref(&state_obj.postal_code, fp);

    fput_comma(fp);

    fput_uint8(state_obj.number, fp);

    fput_newline(fp);
}
    

void export_us_city(ObjRef* ref, FILE* fp)
{
    C_struct_US_City city_obj;

    old_read_object_from_disk(ref, &city_obj, sizeof(city_obj));

    export_every_object(ref, &city_obj.header, fp);

    fput_objref(&city_obj.name, fp);

    fput_comma(fp);

    fput_objref(&city_obj.state, fp);

    assert(is_void_reference(&city_obj.country));   /* never used this? */

    fput_newline(fp);
}
    

void export_area_code(ObjRef* ref, FILE* fp)
{
    C_struct_Area_Code area_code_obj;
    int i;

    old_read_object_from_disk(ref, &area_code_obj, sizeof(area_code_obj));

    export_every_object(ref, &area_code_obj.header, fp);

    fput_objref(&area_code_obj.state, fp);

    fput_comma(fp);

    assert(is_void_reference(&area_code_obj.area));   /* never used this? */

    fput_char('"', fp);

    for (i = 0; i < sizeof(area_code_obj.storage); i++) fput_char(area_code_obj.storage[i], fp);

    fput_char('"', fp);

    fput_newline(fp);
}

    
void export_phone_number(ObjRef* ref, FILE* fp)
{
    C_struct_Phone_Number phone_number_obj;
    int i;

    old_read_object_from_disk(ref, &phone_number_obj, sizeof(phone_number_obj));

    export_every_object(ref, &phone_number_obj.header, fp);

    assert(is_void_reference(&phone_number_obj.country));   /* never used this? */

    fput_objref(&phone_number_obj.area_code, fp);

    fput_comma(fp);

    fput_char('"', fp);

    for (i = 0; i < sizeof(phone_number_obj.storage); i++) fput_char(phone_number_obj.storage[i], fp);

    fput_char('"', fp);

    fput_newline(fp);
}

    
void export_mobile_phone(ObjRef* ref, FILE* fp)
{
    C_struct_Mobile_Phone mobile_phone_obj;

    old_read_object_from_disk(ref, &mobile_phone_obj, sizeof(mobile_phone_obj));

    export_every_object(ref, &mobile_phone_obj.header, fp);

    fput_objref(&mobile_phone_obj.person, fp);

    fput_comma(fp);

    fput_objref(&mobile_phone_obj.number, fp);

    fput_newline(fp);
}

    
void export_year(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Year* year_obj_ptr = (C_struct_Year*) kludge;
    int i;

    old_read_variable_sized_object_from_disk(ref, year_obj_ptr, get_year_object_size);

    export_every_object(ref, &year_obj_ptr->header, fp);

    fput_char('"', fp);

    for (i = 0; i < year_obj_ptr->count; i++)
    {
	fput_char(year_obj_ptr->storage[i], fp);
    }

    fput_char('"', fp);

    fput_newline(fp);
}
    

void export_month(ObjRef* ref, FILE* fp)
{
    C_struct_Month month_obj;

    old_read_object_from_disk(ref, &month_obj, sizeof(month_obj));

    export_every_object(ref, &month_obj.header, fp);

    assert(is_void_reference(&month_obj.definition));   /* never used this? */
    
    fput_char('"', fp);

    fput_char(month_obj.storage[0], fp);

    fput_char(month_obj.storage[1], fp);

    fput_char('"', fp);

    fput_comma(fp);

    fput_uint8(month_obj.minimum_days, fp);

    fput_comma(fp);

    fput_uint8(month_obj.maximum_days, fp);

    fput_newline(fp);
}
    

void export_date(ObjRef* ref, FILE* fp)
{
    C_struct_Date date_obj;

    old_read_object_from_disk(ref, &date_obj, sizeof(date_obj));

    export_every_object(ref, &date_obj.header, fp);

    fput_objref(&date_obj.year, fp);

    fput_comma(fp);

    fput_objref(&date_obj.month, fp);

    fput_comma(fp);

    fput_uint8(date_obj.day_of_month, fp);

    fput_newline(fp);
}
    

void export_gender(ObjRef* ref, FILE* fp)
{
    C_struct_Gender gender_obj;

    old_read_object_from_disk(ref, &gender_obj, sizeof(gender_obj));

    export_every_object(ref, &gender_obj.header, fp);

    fput_objref(&gender_obj.definition, fp);

    fput_newline(fp);
}
    

void export_person(ObjRef* ref, FILE* fp)
{
    C_struct_Person person_obj;

    old_read_object_from_disk(ref, &person_obj, sizeof(person_obj));

    export_every_object(ref, &person_obj.header, fp);

    fput_objref(&person_obj.gender, fp);

    fput_comma(fp);

    assert(is_void_reference(&person_obj.title));   /* never used this? */
    
    fput_objref(&person_obj.first_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.middle_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.last_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.maiden_name, fp);

    fput_comma(fp);

    assert(is_void_reference(&person_obj.preferred_name));   /* never used this? */
    
    fput_objref(&person_obj.goes_by, fp);

    fput_comma(fp);

    fput_objref(&person_obj.nick_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.birth_date, fp);

    assert(is_void_reference(&person_obj.birth_place));   /* never used this? */

    assert(is_void_reference(&person_obj.death_date));   /* never used this? */

    assert(is_void_reference(&person_obj.death_place));   /* never used this? */
    
    assert(is_void_reference(&person_obj.height));   /* never used this? */

    assert(is_void_reference(&person_obj.mailing_address));   /* never used this? */

    fput_newline(fp);
}
    

void export_class(ObjRef* class_def_ref)
{
    C_struct_class_definition class_def_class_obj;
    size_t ref_list_size;
    ReferenceList* ref_list_ptr = NULL;
    char name[32];
    char filename[36];
    int i;
    int num_objects;
    FILE* fp = NULL;

    old_read_object_from_disk(class_def_ref, &class_def_class_obj, sizeof(class_def_class_obj));

    old_name_to_string(&class_def_class_obj.name, name, sizeof(name));

    printf("class %s: %02x%02x%02x%02x\n",
	   name,
	   class_def_ref->id[0],
	   class_def_ref->id[1],
	   class_def_ref->id[2],
	   class_def_ref->id[3]);

    if (is_void_reference(&class_def_class_obj.header.object.references))
    {
	assert(strcmp(name, "Reference List") == 0);

	num_objects = 0;

	printf("  the reference list class doesn't have a reference list\n");
    }
    else
    {
	ref_list_size = old_reference_list_size(&class_def_class_obj.header.object.references);

	ref_list_ptr = malloc(ref_list_size);

	if (ref_list_ptr == NULL) 
	  {
	    perror("reading reference list");
	    exit(1);
	  }

	old_read_reference_list_from_disk(&class_def_class_obj.header.object.references, ref_list_ptr, ref_list_size);

	num_objects = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	printf("  num_objects: %d\n", num_objects);
    }

    if (num_objects > 0)     /* create the file */
    {
	for (i = 0; name[i] != '\0'; i++)
	{
	    if (name[i] == ' ')
	    {
		filename[i] = '_';
	    }
	    else
	    {
		filename[i] = tolower(name[i]);
	    }
	}

	filename[i++] = '.';
	filename[i++] = 'c';
	filename[i++] = 's';
	filename[i++] = 'v';
	filename[i] = '\0';

	fp = fopen(filename, "w");

	if (fp == NULL)
	{
	    perror(filename);
	    exit(1);
	}

	current_filename = filename;

	if (strcmp(name, "Class Definition") == 0)
	{
	    export_class_definition(&old_class_definition_class_ref, fp);
	    for (i = 0; i < num_objects; i++) export_class_definition(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Root") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_root(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Spelling") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_spelling(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Name") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_name(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Word") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_word(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Language") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_language(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Us State") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_us_state(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Us City") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_us_city(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Area Code") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_area_code(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Phone Number") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_phone_number(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Mobile Phone") == 0)
	{
	    //	for (i = 0; i < num_objects; i++) export_mobile_phone(&ref_list_ptr->references[i], fp);
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		old_get_object_class(&ref_list_ptr->references[i], &class);
		
		if (is_same_object(class_def_ref, &class))
		{
		    export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Home Phone") == 0)
	{
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		old_get_object_class(&ref_list_ptr->references[i], &class);

		if (is_same_object(class_def_ref, &class))
		{
		    //		export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Work Phone") == 0)
	{
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		old_get_object_class(&ref_list_ptr->references[i], &class);

		if (is_same_object(class_def_ref, &class))
		{
		    //		export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Year") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_year(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Month") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_month(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Gender") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_gender(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Date") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_date(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Person") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_person(&ref_list_ptr->references[i], fp);
	}
	else
	{
	    printf("  BUG - unprocessed class: %s\n", name);
	}
    }

    if (fp != NULL)
    {
	if (fclose(fp) != 0)
	{
	    perror(filename);
	    exit(1);
	}

	current_filename = NULL;
    }

    if (ref_list_ptr != NULL)
    {
	free(ref_list_ptr);
	ref_list_ptr = NULL;
    }
}



int main(int argc, char* argv[])
{
    ObjRef old_root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    C_struct_class_definition class_def_class_obj;
    size_t ref_list_size;
    ReferenceList* class_def_ref_list_ptr;
    int num_classes;
    int i;


    printf("\n");

    old_get_key_from_password(big_key, sizeof(big_key));

    printf("\n");

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(old_root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    old_initialize_objectify(bf_key, linear, serial);

    old_set_root_object(&old_root_object_ref);

    ref_list_fp = fopen("reference_list.csv", "w");

    if (ref_list_fp == NULL)
    {
	perror("reference_list.csv");
	exit(1);
    }

    printf("root object ref: %02x%02x%02x%02x\n", 
	   old_root_object_ref.id[0],
	   old_root_object_ref.id[1],
	   old_root_object_ref.id[2],
	   old_root_object_ref.id[3]);

    printf("class_definition_class_ref: %02x%02x%02x%02x\n", 
	   old_class_definition_class_ref.id[0],
	   old_class_definition_class_ref.id[1],
	   old_class_definition_class_ref.id[2],
	   old_class_definition_class_ref.id[3]);

    printf("reference_list_class_ref: %02x%02x%02x%02x\n", 
	   old_reference_list_class_ref.id[0],
	   old_reference_list_class_ref.id[1],
	   old_reference_list_class_ref.id[2],
	   old_reference_list_class_ref.id[3]);

    old_read_object_from_disk(&old_class_definition_class_ref, &class_def_class_obj, sizeof(class_def_class_obj));

    /* kludge because I didn't originally put the class_def_class in it's reference list */
    memcpy(&old_class_definition_ref_list, &class_def_class_obj.header.object.references, sizeof(ObjRef));

    export_class(&old_class_definition_class_ref);

    ref_list_size = old_reference_list_size(&old_class_definition_ref_list);

    num_classes = (ref_list_size - sizeof(ReferenceList)) / sizeof(ObjRef);

    class_def_ref_list_ptr = malloc(ref_list_size);

    old_read_reference_list_from_disk(&class_def_class_obj.header.object.references, class_def_ref_list_ptr, ref_list_size);

    for (i = 0; i < num_classes; i++)
    {
	export_class(&class_def_ref_list_ptr->references[i]);
    }

    if (fclose(ref_list_fp) != 0)
    {
	perror("reference_list.csv");
	exit(1);
    }

    return 0;
}
