/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "crc32.h"
#include "disk_io.h"       /* define IVEC_SIZE */
#include "objectify.h"
#include "security.h"


#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


void print_reference(char* name, ObjRef* ref)
{
  int i;

  printf("%s ", name);

  for (i = 0; i < 4; i++) printf("%02x", ref->id[i]);
}


int main(int argc, char* argv[])
{
    ObjRef ref;
    uint32 hex;
    char* path = DEFAULT_FILE;
    char* endptr;
    uint8 ivec[IVEC_SIZE];
    int seq = 0;
    Extra_Data_Block extra;
    EveryObject* header;
    uint8 chksum[4];
    C_struct_Class_Definition class_def_obj;
    char class_name[64];
    size_t size;
    uint8 *data;
    int i;

    nwos_log_arguments(0, NULL);   /* disable logging */

    printf("\n");

    if (argc < 2 || argc == 3 || argc > 4 || (argc == 2 && *argv[1] == '-') || (argc == 4 && (strcmp(argv[1], "--compressed") != 0 || *argv[2] == '-' || *argv[3] == '-')))
    {
	fprintf(stderr, "usage: %s [--compressed file] object_id_in_hexadecimal\n", argv[0]);
	exit(1);
    }

    hex = strtoul(argv[argc-1], &endptr, 16);

    if (argc == 4)
    {
	path = argv[2];
    }

    if (*endptr != '\0')
    {
	fprintf(stderr, "Invalid reference: %s\n", argv[1]);
	exit(1);
    }

    nwos_word_to_ref(hex, &ref);

    if (nwos_reference_type(&ref) == Public_Reference)
    {
	nwos_initialize_objectify(PUBLIC, NULL);
    }
    else
    {
	nwos_initialize_objectify(READ_ONLY, path);
    }


    //    nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header);

    printf("Id: %08x\n", nwos_ref_to_word(&ref));

    memset(ivec, 0, sizeof(ivec));

    if (!nwos_read_block_from_disk_and_decrypt(&ref, &extra, sizeof(extra), ivec, seq))
    {
	fprintf(stderr, "Error: nwos_read_object_from_disk_and_decrypt failed\n");
	exit(1);
    }

    header = (EveryObject*) &extra;

    nwos_crc32_calculate((uint8*)&header->object, sizeof(header->object), chksum);

    if (memcmp(&header->common.header_chksum, chksum, 4) != 0)
    {
	fprintf(stderr, "Warning: checksum with new decryption algorithm failed, trying old algorithm...\n");

	memset(ivec, 0, sizeof(ivec));

	if (!nwos_read_block_from_disk_and_old_decrypt(&ref, &extra, sizeof(extra), ivec, seq))
	{
	    fprintf(stderr, "Error: nwos_read_object_from_disk_and_old_decrypt failed\n");
	    exit(1);
	}

	nwos_crc32_calculate((uint8*)&header->object, sizeof(header->object), chksum);

	if (memcmp(&header->common.header_chksum, chksum, 4) != 0)
	{
	    fprintf(stderr, "Error: checksum with old decryption algorithm failed as well.\n");
	}
    }

    printf("Creation: %s\n", nwos_time_stamp_to_string(header->common.creation_time));

    printf("Checksums:\n");
    printf("  Header: %02x%02x%02x%02x\n", header->common.header_chksum[0], header->common.header_chksum[1],
	   header->common.header_chksum[2], header->common.header_chksum[3]);
    printf("    Data: %02x%02x%02x%02x\n", header->common.data_chksum[0], header->common.data_chksum[1],
	   header->common.data_chksum[2], header->common.data_chksum[3]);

    nwos_read_class_definition(&header->common.class_definition, &class_def_obj);
    nwos_name_to_string(&class_def_obj.name, class_name, sizeof(class_name));

    printf("Class: %08x - %s\n", nwos_ref_to_word(&header->common.class_definition), class_name);

    printf("Clone Of: %08x\n", nwos_ref_to_word(&header->object.clone_of));
    printf("References: %08x\n", nwos_ref_to_word(&header->object.references));
    printf("Context: %08x\n", nwos_ref_to_word(&header->object.context));
    printf("Creation: %08x\n", nwos_ref_to_word(&header->object.creation));
    printf("Source: %08x\n", nwos_ref_to_word(&header->object.source));
    printf("Prev_version: %08x\n", nwos_ref_to_word(&header->object.prev_version));
    printf("Next_version: %08x\n", nwos_ref_to_word(&header->object.next_version));

    size = nwos_get_object_size(&extra);

    data = malloc(size);

    if (data == NULL)
    {
	perror("allocating space for object");
	exit(1);
    }

    assert(nwos_read_object_from_disk(&ref, data, size));

    printf("\n");

    for (i = sizeof(EveryObject); i < size; i++)
    {
	printf("%02x", data[i]);

	if (i % 16 == 3 || i % 16 == 7 || i % 16 == 11)
	{
	    printf("  ");
	}
	else if (i % 16 == 15)
	{
	    printf("\n");
	}
	else
	{
	    printf(" ");
	}
    }

    printf("\n");

    nwos_terminate_objectify();

    return 0;
}


