/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the old create_person.c file
--   which this file was originally derived from.  I left the old log from the
--   create_person.c file in the computer.c file to make it easier to find.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"


#define MAX_NAME_OR_SPELLING_SIZE (MAX_NAME_OBJ_SIZE > MAX_SPELLING_OBJ_SIZE ? MAX_NAME_OBJ_SIZE : MAX_SPELLING_OBJ_SIZE)


static ObjRef* get_public_name_class_ref()
{
    static ObjRef class_ref;

    if (is_void_reference(&class_ref))
    {
	nwos_find_public_class_definition("NAME", &class_ref);

	assert(!is_void_reference(&class_ref));
    }

    return &class_ref;
}


static ObjRef* get_private_name_class_ref()
{
    static ObjRef class_ref;

    if (is_void_reference(&class_ref))
    {
	if (!nwos_find_private_class_definition("NAME", &class_ref))
	{
	    return NULL;
	}
    }

    return &class_ref;
}


static int split_name(const char* name, char words[21][20])
{
    int result = 0; 
    int i;
    const char* p;

    i = 0;
    for (p = name; *p != '\0'; p++)
    {
	if (*p == ' ')
	{
	    assert(i > 0);
	    words[result][i] = '\0';
	    i = 0;
	    result++;
	}
	else
	{
	    assert(isalnum(*p) || *p == '.' || *p == '\'');   /* allow abbreviation or apostrophe in name */
	    assert(result <= 21);
	    assert(i <= 19);         /* leave room for null */
	    words[result][i] = *p;
	    i++;
	}
    }

    assert(i > 0);
    words[result][i] = '\0';
    result++;

    return result;
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}


bool nwos_find_public_name(const char* name, ObjRef* ref)
{
    uint8 kludge[MAX_NAME_OBJ_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    bool result = true;
    int num_words;
    char words[21][20];
    char temp_name[256];

    num_words = split_name(name, words);

#ifdef DEBUG_NAME
    printf("num_words: %d\n", num_words);
    for (j = 0; j < num_words; j++)
      {
	printf("word[%d]: %s\n", j, words[j]);
      }
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
	if (!nwos_find_public_spelling(words[word], &spelling_ref[word]))
	{
#ifdef DEBUG_NAME
	    printf("did not find word: %s\n", words[word]);
#endif
	    result = false;
	    break;
	}
#ifdef DEBUG_NAME
	else
	{
	    printf("word: %s - %08lx\n", words[word], 
		   nwos_ref_to_word(&spelling_ref[word]));
	}
#endif
    }

    if (result)  /* we found all the words */
    {
#ifdef DEBUG_NAME
	printf("found all words\n");
#endif
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("word: %s  num_spellings: %d\n", words[word], num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	    {
		printf("spelling_ref_list[%d]->references[%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	    }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
		if (is_same_object(&object_class, get_public_name_class_ref()))   /* it is a name object */
		{
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		result = false;
	    }

#ifdef DEBUG_NAME
	    printf("word: %s  num_name_refs: %d\n", words[word], num_spellings[word]);
#endif
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	    printf("current_top: %p in the beginning\n", current_top);
#endif

	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("word[%d]: %s - refs[%d]: %08lx - refs[%d]: %08lx\n", word, words[word],
		   j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]),
		   k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }
#ifdef DEBUG_NAME
	    printf("num_spellings[0]: %d\n", num_spellings[0]);
#endif
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	result = false;   /* assume we won't find it */

	/* Note we won't go through this list at all if all of the references were eliminated in the first word */

	for (j = 0; j < num_spellings[0]; j++)
	{
	    assert(nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size));

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    result = true;
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result)  /* assemble the object to make sure it matches the original string */
    {
	nwos_name_to_string(ref, temp_name, sizeof(temp_name));
	assert(strcasecmp(name, temp_name) == 0);
    }

    return result;
}


bool nwos_find_private_name(const char* name, ObjRef* ref)
{
    uint8 kludge[MAX_NAME_OBJ_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    bool result = true;
    int num_words;
    char words[21][20];
    char temp_name[256];

    if (get_private_name_class_ref() == NULL) return false;

    num_words = split_name(name, words);

#ifdef DEBUG_NAME
    printf("num_words: %d\n", num_words);
    for (j = 0; j < num_words; j++)
      {
	printf("word[%d]: %s\n", j, words[j]);
      }
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
	if (!nwos_find_private_spelling(words[word], &spelling_ref[word]))
	{
#ifdef DEBUG_NAME
	    printf("did not find word: %s\n", words[word]);
#endif
	    result = false;
	    break;
	}
#ifdef DEBUG_NAME
	else
	{
	    printf("word: %s - %08lx\n", words[word], 
		   nwos_ref_to_word(&spelling_ref[word]));
	}
#endif
    }

    if (result)  /* we found all the words */
    {
#ifdef DEBUG_NAME
	printf("found all words\n");
#endif
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("word: %s  num_spellings: %d\n", words[word], num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	    {
		printf("spelling_ref_list[%d]->references[%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	    }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
		if (is_same_object(&object_class, get_private_name_class_ref()))   /* it is a name object */
		{
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		result = false;
	    }

#ifdef DEBUG_NAME
	    printf("word: %s  num_name_refs: %d\n", words[word], num_spellings[word]);
#endif
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; result && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	    printf("current_top: %p in the beginning\n", current_top);
#endif

	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("word[%d]: %s - refs[%d]: %08lx - refs[%d]: %08lx\n", word, words[word],
		   j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]),
		   k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }
#ifdef DEBUG_NAME
	    printf("num_spellings[0]: %d\n", num_spellings[0]);
#endif
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	result = false;   /* assume we won't find it */

	/* Note we won't go through this list at all if all of the references were eliminated in the first word */

	for (j = 0; j < num_spellings[0]; j++)
	{
	    assert(nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size));

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    result = true;
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result)  /* assemble the object to make sure it matches the original string */
    {
	nwos_name_to_string(ref, temp_name, sizeof(temp_name));
	assert(strcasecmp(name, temp_name) == 0);
    }

    return result;
}


/* NOTE: the find part could be extracted from the create name below, but */
/* I'm not sure exactly how to do it right now so I am leaving it as is.  */

ObjCreateResult nwos_create_name(const char* name, ObjRef* ref)
{
    uint8 kludge[MAX_NAME_OBJ_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    EveryObject spelling_header;
    ObjRef* name_class_ref_ptr;
    ObjRef spelling_ref[21];
    ObjRef object_class;
    ReferenceList* spelling_ref_list[21];
    ObjRef* current_top;
    int num_spellings[21];
    int old_num_spellings;
    int word;
    int j;
    int k;
    ObjCreateResult spelling_result = FOUND_EXISTING;
    ObjCreateResult result = CREATED_NEW;
    int num_words;
    char words[21][20];
    char temp_name[256];
#ifdef DEBUG_NAME
    uint8 *p;
#endif
#if 0
    int i;
    char reassemble[256];
    char *p;
#endif

    num_words = split_name(name, words);

#if 0
    printf("words: %d\n", num_words);
    for (i = 0; i < num_words; i++)
      {
	printf("%d: %s\n", i, words[i]);
      }
#endif

#ifdef PUBLIC_MODE
    name_class_ref_ptr = get_public_name_class_ref();
#else
    nwos_find_or_create_private_class_definition("NAME", &object_class);
    name_class_ref_ptr = get_private_name_class_ref();
#endif

    assert(num_words > 0);

    for (word = 0; word < num_words; word++)
    {
#ifdef PUBLIC_MODE
	if (nwos_create_public_spelling(words[word], &spelling_ref[word]) != FOUND_EXISTING)
#else
	if (nwos_create_private_spelling(words[word], &spelling_ref[word]) != FOUND_EXISTING)
#endif
	{
	    spelling_result = CREATED_NEW;
	}
#ifdef DEBUG_NAME
	printf("spelling_ref[%d]: %08lx\n", word, nwos_ref_to_word(&spelling_ref[word]));
#endif
    }

    if (spelling_result == FOUND_EXISTING)     /* all words existed */
    {
	for (word = 0; word < num_words; word++)
	{
	    nwos_read_object_headers_from_disk(&spelling_ref[word], &spelling_header);

	    spelling_ref_list[word] = nwos_malloc_reference_list(&spelling_header.object.references);

	    num_spellings[word] = spelling_ref_list[word]->common_header.num_refs;

#ifdef DEBUG_NAME
	    printf("num_spellings: %d - %d\n", word, num_spellings[word]);
	    for (j = 0; j < num_spellings[word]; j++)
	      {
		p = (uint8*)spelling_ref_list[word];
		for (k = 0; k < ref_list_size; k++) printf("%02x%c", *p++, (k%16 == 15) ? '\n' : ' ');
		printf("spelling_references[%d][%d]: %08lx\n", word, j, nwos_ref_to_word(&spelling_ref_list[word]->references[j]));
	      }
#endif
	}

	/* now go through each reference list and find the name objects and place them at the beginning */

	/* NOTE: We won't have to do this when the reference list is divided up by classes!! */

	for (word = 0; spelling_result == FOUND_EXISTING && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[word]->references[0];   /* create a pointer of where we want to put the good ones */

	    for (j = 0; j < num_spellings[word]; j++)
	    {
		nwos_get_object_class(&spelling_ref_list[word]->references[j], &object_class);   /* find out what kind of object it is */
#ifdef DEBUG_NAME
		printf("object_class[%d][%d]: %08lx\n", word, j, nwos_ref_to_word(&object_class));
#endif
		if (is_same_object(&object_class, name_class_ref_ptr))   /* it is a name object */
		{
#ifdef DEBUG_NAME
		  printf("is a name object\n");
#endif
		    /* found a name, if we have eliminated any, copy this one to the good ones */
		    if (current_top != &spelling_ref_list[word]->references[j])
		    {
#ifdef DEBUG_NAME
		      printf("moving it from %p to %p\n", &spelling_ref_list[word]->references[j], current_top);
#endif
			memcpy(current_top, &spelling_ref_list[word]->references[j], sizeof(ObjRef));
		    }
		    current_top++;   /* another good one */
		}
	    }

	    if (current_top != &spelling_ref_list[word]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[word];   /* save it to make sure the new one is sane */ 
		num_spellings[word] = current_top - &spelling_ref_list[word]->references[0];
		assert(0 <= num_spellings[word] && num_spellings[word] <= old_num_spellings);
#ifdef DEBUG_NAME
		printf("new size of num_spellings[%d]: %d\n", word, num_spellings[word]);
#endif
	    }

	    if (num_spellings[word] == 0)    /* no use continuing if there are no name objects for this word */
	    {
		spelling_result = CREATED_NEW;
	    }
	}

	/* Now go through what we have left and see if there is a name that has all of the words in it.  */
	/* I.E. if there aren't any common names in all of the reference lists, then there isn't a name  */
        /* that conatins all of these words and we can quit looking.  The advantage to this is then we   */
	/* don't have to actually read the objects yet, we can do this with just the reference lists.    */
	/* This can be done two lists at a time, compare two lists and eliminate all of the ones that    */
	/* aren't in both lists.  When it is finished both lists have to be identical and we no longer   */
	/* have to look at the second list and can just compare the 1st list and the 3rd list and so on. */

	for (word = 1; spelling_result == FOUND_EXISTING && word < num_words; word++)
	{
	    current_top = &spelling_ref_list[0]->references[0];   /* create a pointer of where we want to put the good ones */

#ifdef DEBUG_NAME
	  printf("checking word: %d\n", word);
#endif
	    for (j = 0; j < num_spellings[0]; j++)     /* for each reference in the first word's reference list */
	    {
#ifdef DEBUG_NAME
	    printf("compare spelling_ref_list[0]->references[%d]: %08lx\n", j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]));
#endif
		for (k = 0; k < num_spellings[word]; k++)     /* for each reference in the i'th word's reference list */
		{
#ifdef DEBUG_NAME
	    printf("with    spelling_ref_list[%d]->references[%d]: %08lx\n", word, k, nwos_ref_to_word(&spelling_ref_list[word]->references[k]));
#endif
		    if (is_same_object(&spelling_ref_list[0]->references[j], &spelling_ref_list[word]->references[k])) /* same */
		    {
			/* found a name in both lists, if we have eliminated any, copy this one to the good ones */
			if (current_top != &spelling_ref_list[0]->references[j])
			{
			    memcpy(current_top, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
			}
			current_top++;   /* another one in both lists */
		    }
		}
	    }

	    if (current_top != &spelling_ref_list[0]->references[j])   /* recompute how many name objects we have */
	    {
		old_num_spellings = num_spellings[0];   /* save it to make sure the new one is sane */ 
		num_spellings[0] = current_top - &spelling_ref_list[0]->references[0];
		assert(0 <= num_spellings[0] && num_spellings[0] <= old_num_spellings);
	    }

	    if (num_spellings[0] == 0)    /* no use continuing if there are no more name objects for this word */
	    {
		spelling_result = CREATED_NEW;
	    }
	}

	/* We should now have a list in the first words reference list of all the names that contain all of the words. */
	/* We can now open each one of those an see if the words are in the right order. */

	for (j = 0; j < num_spellings[0]; j++)
	{
#ifdef DEBUG_NAME
	    printf("reading spelling_ref_list[0]->references[%d]: %08lx\n", j, nwos_ref_to_word(&spelling_ref_list[0]->references[j]));
#endif
	    assert(nwos_read_variable_sized_object_from_disk(&spelling_ref_list[0]->references[j], kludge, sizeof(kludge), &get_name_object_size));

	    /* remember ptr_name_obj points to the kludge buffer */

	    if (ptr_name_obj->count == num_words)    /* right number of words */
	    {
		for (word = 0; word < num_words; word++)
		{
		    if (!is_same_object(&ptr_name_obj->spelling[word], &spelling_ref[word])) break;  /* it is not the right word */
		}

		if (word == num_words)   /* found it */
		{
		    memcpy(ref, &spelling_ref_list[0]->references[j], sizeof(ObjRef));
		    result = FOUND_EXISTING;
		    break;
		}
	    }
	}

	for (word = 0; word < num_words; word++)
	{
	    nwos_free_reference_list(spelling_ref_list[word]);
	    spelling_ref_list[word] = NULL;
	}
    }

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_name_obj->header.common, ref, name_class_ref_ptr);

	ptr_name_obj->count = num_words;
	for (word = 0; word < num_words; word++)
	{
	    memcpy(&ptr_name_obj->spelling[word], &spelling_ref[word], sizeof(ObjRef));
	    nwos_add_to_references(ref, &spelling_ref[word]);
	}

	nwos_create_reference_list(ref, &ptr_name_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_name_obj->header.object, sizeof(ObjectHeader), ptr_name_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_name_obj->count, sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)) - sizeof(EveryObject), ptr_name_obj->header.common.data_chksum);


	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)));

#if 0
	{
	int i;
	for (i = 0; i < sizeof(C_struct_Name) + (num_words * sizeof(ObjRef)); i++)
	  {
	    printf("%02x%c", kludge[i], (i % 16 == 15) ? '\n' : ' ');
	  }
	}
#endif
        nwos_add_to_references(ref, name_class_ref_ptr);
    }

    /* reassemble the object to make sure it matches the original string */

    nwos_name_to_string(ref, temp_name, sizeof(temp_name));
    assert(strcasecmp(name, temp_name) == 0);
#if 0
    printf("reassemble: %s\n", kludge);
#endif

    return result;
}



/* This is horrible - FIX THIS!! */

static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}


bool nwos_name_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[MAX_NAME_OR_SPELLING_SIZE];
    C_struct_Name* ptr_name_obj;
    C_struct_Spelling* ptr_spelling_obj;
    int word;
    int i;
    int j;
    size_t name_obj_size;

    /* read the name object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_name_object_size));
    ptr_name_obj = (C_struct_Name*)kludge;
    name_obj_size = sizeof(C_struct_Name) + (ptr_name_obj->count * sizeof(ObjRef));
    /* printf("name_obj_size: %d\n", name_obj_size); */
    assert(name_obj_size > sizeof(C_struct_Name));
    ptr_name_obj = malloc(name_obj_size);
    memcpy(ptr_name_obj, kludge, name_obj_size);

    ptr_spelling_obj = (C_struct_Spelling*)kludge;

    i = 0;
    for (word = 0; word < ptr_name_obj->count; word++)
    {
	assert(!is_void_reference(&ptr_name_obj->spelling[word]));
	assert(nwos_read_variable_sized_object_from_disk(&ptr_name_obj->spelling[word], kludge, sizeof(kludge), &get_spelling_object_size));

	string[i++] = toupper(ptr_spelling_obj->storage[0]);    /* first letter is upper case */

	for (j = 1; j < ptr_spelling_obj->count; j++) 
	{
	    assert(i < size);

	    /* if third letter and first two were McX or O'X then make the third letter upper case */
            if (j == 2 &&
		((ptr_spelling_obj->storage[0] == 'm' && ptr_spelling_obj->storage[1] == 'c') ||
		 (ptr_spelling_obj->storage[0] == 'o' && ptr_spelling_obj->storage[1] == '\'')))
	    {
		string[i++] = toupper(ptr_spelling_obj->storage[j]);
	    }
	    else
	    {
		string[i++] = ptr_spelling_obj->storage[j];
	    }
	}

	assert(i < size);

	if (word + 1 == ptr_name_obj->count)   /* it's the last word */
	{
	    string[i++] = '\0';
	}
	else
	{
	    string[i++] = ' ';
	}
    }

    free(ptr_name_obj);

    return true;
}


