/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:34:21 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4265 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "chunk_info.h"
#include "config.h"
#include "header.h"
#include "user_config.h"



static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [--graph] [archive]\n", program);
}


void display_graph(Chunk_Info chunk_info[], uint32 num_chunks)
{
    const int graph_bits = 6;
    const int divisor = 1 << (graph_bits - 4);
    uint32 used_blocks[1 << graph_bits];
    uint32 alloc_blocks[1 << graph_bits];
    uint32 total_blocks[1 << graph_bits];
    uint32 chunk;
    int segment;
    int gigabytes;
    
    int i;


    memset(used_blocks, 0, sizeof(used_blocks));
    memset(alloc_blocks, 0, sizeof(alloc_blocks));
    memset(total_blocks, 0, sizeof(total_blocks));

    for (chunk = MINIMUM_PRIVATE_REFERENCE; chunk < MAXIMUM_PRIVATE_REFERENCE; chunk += USABLE_BLOCKS_PER_CHUNK)
    {
	segment = chunk >> (32 - graph_bits);

	assert((RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)) <= segment && segment < (1 << graph_bits));

	total_blocks[segment] += USABLE_BLOCKS_PER_CHUNK;
    }

    for (i = 0; i < num_chunks; i++)
    {
	segment = (chunk_info[i].ref - BASE_32_BIT_PRIVATE) >> (32 - graph_bits);

	assert((RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)) <= segment && segment < (1 << graph_bits));

	used_blocks[segment] += chunk_info[i].flags_used & CHUNK_INFO_USED_MASK;
	alloc_blocks[segment] += USABLE_BLOCKS_PER_CHUNK;
    }

    for (gigabytes = (1024 / (1 << graph_bits)); gigabytes >= 0; gigabytes--)
    {
	printf(" %2d GB |", gigabytes);

	for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++)
	{
	    if (used_blocks[i] > 0 && (((uint64)used_blocks[i] * (1024 / (1 << graph_bits))) / total_blocks[i]) >= (uint64)gigabytes)
	    {
		putchar('*');
	    }
	    else if (alloc_blocks[i] > 0 && (((uint64)alloc_blocks[i] * (1024 / (1 << graph_bits))) / total_blocks[i]) >= (uint64)gigabytes)
	    {
		putchar('+');
	    }
	    else
	    {
		putchar(' ');
	    }
	}

	putchar('\n');
    }

    printf("       +");
    for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++) putchar('-');
    putchar('\n');

    printf("        ");
    for (i = (RESERVED_PUBLIC_BLOCKS >> (32 - graph_bits)); i < (1 << graph_bits); i++)
    {
	if (i % divisor == 0)
	{
	    if (i / divisor < 10)
	    {
		putchar('0' + i / divisor);
	    }
	    else
	    {
		putchar('A' + i / divisor - 10);
	    }
	}
	else
	{
	    putchar(' ');
	}
    }
    putchar('\n');
    putchar('\n');

    printf("     * = used\n");
    printf("     + = allocated, but unused\n");
}


int main(int argc, char* argv[])
{
    bool graph = false;
    int obj_file_desc;
    struct flock lock;
    const char* obj_file_path;
    char ref_str[128];
    uint64 used_private_blocks;
    uint64 available_private_blocks;
    uint32 chunk_offset;
    uint32 system_blocks;
    uint32 used_blocks;    /* includes blocks used for bit maps */
    uint64 allocated_blocks;
    uint32 unallocated_blocks;
    uint32 total_blocks;
    const uint32 used_system_chunks = 1;
    uint32 total_chunks;
    uint32 total_private_chunks;
    uint32 used_private_chunks;
    uint32 unallocated_chunks;
    uint8 block[FILE_BLOCK_SIZE];
    float divisor;
    char* range;
    Disk_Header disk_header;
    Chunk_Info* chunk_info;
    size_t sizeof_chunk_info;
    int i;


    if (argc == 1)   /* no option or archive specified */
    {
	obj_file_path = nwos_get_private_objects_path();
    }
    else if (argc == 2)
    {
	if (*argv[1] == '-')
	{
	    if (strcmp(argv[1], "--graph") != 0)
	    {
		fprintf(stderr, "\nunknown option: %s\n\n", argv[1]);
		print_usage(argv[0]);
		exit(1);
	    }

	    obj_file_path = nwos_get_private_objects_path();
	    graph = true;
	}
	else
	{
	    obj_file_path = argv[1];
	}
    }
    else if (argc == 3)
    {
	if (((*argv[1] == '-') ^ (*argv[2] == '-')) == false)     /* only one or the other can be an option */
	{
	    print_usage(argv[0]);
	    exit(1);
	}

	if ((*argv[1] == '-' && strcmp(argv[1], "--graph") != 0) || (*argv[2] == '-' && strcmp(argv[2], "--graph") != 0))
	{
	    fprintf(stderr, "\nunknown option: %s\n\n", argv[1]);
	    print_usage(argv[0]);
	    exit(1);
	}

	if (*argv[1] == '-')
	{
	    obj_file_path = argv[2];
	}
	else
	{
	    obj_file_path = argv[1];
	}

	graph = true;
    }
    else
    {
	print_usage(argv[0]);
	exit(1);
    }


    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    if (read(obj_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", obj_file_path);
	perror(ref_str);
	exit(1);
    }

    memcpy(&disk_header, block, sizeof(disk_header));

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", obj_file_path);
	exit(1);
    }

    printf("\n");

    printf("Archive: %s\n", obj_file_path);

    printf("Version: %c%c%c%c\n", 
	   disk_header.version_string[0], disk_header.version_string[1],
	   disk_header.version_string[2], disk_header.version_string[3]);

    printf("Formatted:   %u-%02u-%02u %02u:%02u:%02u\n",
	   nwos_extract_year_from_time_stamp(disk_header.last_prep_disk),
	   nwos_extract_month_from_time_stamp(disk_header.last_prep_disk),
	   nwos_extract_day_of_month_from_time_stamp(disk_header.last_prep_disk),
	   nwos_extract_hour_from_time_stamp(disk_header.last_prep_disk),
	   nwos_extract_minute_from_time_stamp(disk_header.last_prep_disk),
	   nwos_extract_second_from_time_stamp(disk_header.last_prep_disk));

    printf("Last change: %u-%02u-%02u %02u:%02u:%02u\n",
	   nwos_extract_year_from_time_stamp(disk_header.last_change),
	   nwos_extract_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_day_of_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_hour_from_time_stamp(disk_header.last_change),
	   nwos_extract_minute_from_time_stamp(disk_header.last_change),
	   nwos_extract_second_from_time_stamp(disk_header.last_change));

    /* get what the disk thinks */
    nwos_4_uint8_to_uint32(disk_header.total_chunks, &total_chunks);
    nwos_8_uint8_to_uint64(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_offset);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_private_chunks);

    total_private_chunks = total_chunks - used_system_chunks;

    sizeof_chunk_info = used_private_chunks * sizeof(Chunk_Info);
    chunk_info = malloc(sizeof_chunk_info);
    assert(chunk_info != NULL);

    if (read(obj_file_desc, chunk_info, sizeof_chunk_info) != sizeof_chunk_info)
    {
	snprintf(ref_str, sizeof(ref_str), "reading chunk_info from: %s", obj_file_path);
	perror(ref_str);
	exit(1);
    }

    used_blocks = 0;
    for (i = 0; i < (int)used_private_chunks; i++)
    {
#ifndef WORDS_BIGENDIAN
	chunk_info[i].ref = byteswap_uint64(chunk_info[i].ref);
	chunk_info[i].flags_used = byteswap_uint32(chunk_info[i].flags_used);
	chunk_info[i].index = byteswap_uint32(chunk_info[i].index);
#endif
	used_blocks += (chunk_info[i].flags_used & CHUNK_INFO_USED_MASK);
    }

    if (used_blocks != used_private_blocks)
    {
	/* snprintf(log_msg, sizeof(log_msg), */
	fprintf(stderr,
		 "Warning: calculated sum of used blocks (%u) doesn't match stored (%llu)",
		 used_blocks, used_private_blocks);
	/* nwos_log(log_msg); */
	/* fprintf(stderr, "%s\n", log_msg); */
    }

    close(obj_file_desc);


    if (total_chunks < 64)    /* print in megabytes (64 chunks == 1 gigabyte) */
    {
	divisor = 1048576.0f / 256.0f;
	range = "Megabytes";
    }
    else  /* print in gigabytes */
    {
	divisor = 1073741824.0f / 256.0f;
	range = "Gigabytes";
    }

    available_private_blocks = ((total_chunks - used_system_chunks) * USABLE_BLOCKS_PER_CHUNK) - used_private_blocks;

    system_blocks = BLOCKS_IN_CHUNK + (total_private_chunks - used_system_chunks) * BIT_MAP_BLOCKS;

    allocated_blocks = used_private_chunks * USABLE_BLOCKS_PER_CHUNK;

    unallocated_chunks = total_chunks - used_system_chunks - used_private_chunks;
    unallocated_blocks = unallocated_chunks * USABLE_BLOCKS_PER_CHUNK;

    total_blocks = total_chunks * BLOCKS_IN_CHUNK;

    printf("\n");

    printf("              Chunks   Blocks    %s\n", range);

    printf("System:        %5u  %10u    %5.1f  %5.1f%%\n",
	   1,
	   system_blocks,
	   (float)system_blocks / divisor,
	   100.0f * (float)system_blocks / (float)total_blocks);

    printf("Unallocated:   %5u  %10u    %5.1f  %5.1f%%\n",
	   total_private_chunks - used_private_chunks,
	   unallocated_blocks,
	   (float)unallocated_blocks / divisor,
	   100.0f * (float)unallocated_blocks / (float)total_blocks);

    printf("Allocated:     %5u\n", used_private_chunks);

    printf("  Used:               %10llu    %5.1f  %5.1f%%\n",
	   used_private_blocks,
	   (float)used_private_blocks / divisor,
	   100.0f * (float)used_private_blocks / (float)total_blocks);

    printf("  Free:               %10llu    %5.1f  %5.1f%%\n",
	   allocated_blocks - used_private_blocks,
	   (float)(allocated_blocks - used_private_blocks) / divisor,
	   100.0f * (float)(allocated_blocks - used_private_blocks) / (float)total_blocks);

    printf("  Total:              %10llu    %5.1f  %5.1f%%\n",
	   allocated_blocks,
	   (float)allocated_blocks / divisor,
	   100.0f * (float)allocated_blocks / (float)total_blocks);

    printf("\n");


    if (graph)
    {
	display_graph(chunk_info, used_private_chunks);
    }


    printf("\n");

    fflush(stdout);

    return 0;
}

