/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:13:14 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4263 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <errno.h>
#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



bool export_file(char* name)
{
    bool result = false;
    ObjRef path_ref;
    ObjRef assoc_ref;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    int num_files;
    char* p;

    printf("%s: ", name);
    fflush(stdout);

    assert(nwos_find_file_path(name, &path_ref));

    num_files = nwos_number_of_files_for_path(&path_ref);

    if (num_files > 0)
    {
	if (num_files > 1)
	{
	    printf(" WARNING: this version cannot deal with multiple files - skipping\n");
	}
	else
	{
	    assert(nwos_file_path_to_path_and_file_association(&path_ref, 0, &assoc_ref));

	    assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the assoc. object */

	    while (!is_void_reference(&assoc_obj.header.object.next_version))
	    {
		assert(nwos_read_object_from_disk(&assoc_obj.header.object.next_version, &next_assoc_obj, sizeof(next_assoc_obj)));

		if (!is_same_object(&next_assoc_obj.path, &path_ref))   /* it has a different name */
		{
		    assert(is_same_object(&next_assoc_obj.file, &assoc_obj.file));
		    break;
		}

		copy_reference(&assoc_ref, &assoc_obj.header.object.next_version);
		memcpy(&assoc_obj, &next_assoc_obj, sizeof(assoc_obj));
	    }

	    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));  /* read the file object */

	    if (is_void_reference(&file_obj.block_list))
	    {
		printf("not stored in system\n");
	    }
	    else
	      {
		p = strchr(name, '/');
		while (p != NULL)
		{
		    *p = '\0';
		    if (mkdir(name, 0755) != 0 && errno != EEXIST)
		    {
			perror(p);
		    }
		    *p = '/';

		    p = strchr(p + 1, '/');
		}

		result = nwos_restore_file(&assoc_ref, name);

		if (result)
		{
		    printf("OK\n");
		}
		else
		{
		    printf("failed\n");
		}
	    }
	}
    }
    else
    {
	printf("no file associated with that name in system\n");
    }

    return result;
}


int main(int argc, char* argv[])
{
    ObjRef path_class_ref;
    ObjRef object_class;
    ObjRef path_ref;
    ObjRef assoc_ref;
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    int argn;
    int i;
    int j;
    int k;
    const char* path = DEFAULT_FILE;
    char name[256];
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int num_failures = 0;
    bool match;


    if (argc > 1 && *argv[1] == '-')
    {
	if (strcmp(argv[1], "--compressed") != 0 || argc < 3)
	{
	    fprintf(stderr, "usage: %s [--compressed compressed-file] files to extract\n", argv[0]);
	    exit(1);
	}

	path = argv[2];
	argn = 3;
    }
    else
    {
	argn = 1;
    }

    printf("\n");

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_ONLY, path);


    if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref))
    {
	printf("No files are stored in the system!\n");
	exit(1);
    }

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = argn; i < argc; i++)
    {
	match = false;

	for (j = 0; j < num_refs; j++)
	{
	    nwos_get_object_class(&ref_list->references[j], &object_class);

	    if (is_same_object(&object_class, &path_class_ref))
	    {
		assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[j], kludge, sizeof(kludge), &get_path_object_size));

		/* remember ptr_path_obj points to the kludge buffer */

		/* assert(ptr_path_obj->count < sizeof(name)); <<- count is uint8 so always true */

		for (k = 0; k < ptr_path_obj->count; k++) name[k] = ptr_path_obj->storage[k];
		name[k] = '\0';

		if (strcmp(argv[i], name) == 0)   /* found an exact match */
		{
		    if (export_file(name))
		    {
			num_files++;
		    }
		    else
		    {
			num_failures++;
		    }

		    match = true;
		}
		else
		{
		    if (fnmatch(argv[i], name, 0) == 0)   /* found a match */
		    {
			assert(nwos_find_file_path(name, &path_ref));
			assert(nwos_file_path_to_path_and_file_association(&path_ref, 0, &assoc_ref));

			assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the assoc. object */

			while (!is_void_reference(&assoc_obj.header.object.next_version))  /* find latest version */
			{
			    assert(nwos_read_object_from_disk(&assoc_obj.header.object.next_version, &next_assoc_obj, sizeof(next_assoc_obj)));

			    if (!is_same_object(&next_assoc_obj.path, &path_ref))   /* it has a different name */
			    {
				assert(is_same_object(&next_assoc_obj.file, &assoc_obj.file));
				break;    /* don't match old file names to wildcards */
			    }

			    copy_reference(&assoc_ref, &assoc_obj.header.object.next_version);
			    memcpy(&assoc_obj, &next_assoc_obj, sizeof(assoc_obj));
			}

			if (is_void_reference(&assoc_obj.header.object.next_version))
			{
			    if (export_file(name))
			    {
				num_files++;
			    }
			    else
			    {
				num_failures++;
			    }

			    match = true;
			}
		    }
		}
	    }
	}

	if (!match)
	{
	    printf("No file(s) matching '%s' found.\n", argv[i]);

	    num_failures++;
	}
    }

    printf("Number of files exported: %d\n", num_files);
    printf("Number of files failed: %d\n", num_failures);

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return num_failures;    /* return 0 means success */
}


