/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:34:21 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4265 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "crc32.h"

#define MAX_NAME_SIZE 64
#define MAX_DATE_SIZE 16


static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_year_object_size(void* year_obj)
{
    assert(((C_struct_Year*)year_obj)->count > 0);
    return sizeof(C_struct_Year) + ((C_struct_Year*)year_obj)->count;
}


typedef struct {
  char *name;
  size_t size;
  size_t (*func)(void*);
} ClassTbl;

ClassTbl class_tbl[] = {
    { "Spelling",               0,                                       get_spelling_object_size },
    { "Name",                   0,                                       get_name_object_size },
    { "Root",                   0,                                       NULL },
    { "Word",                   sizeof(C_struct_Word),                   NULL },
    { "Language",               sizeof(C_struct_Language),               NULL },
    { "Abbreviation",           sizeof(C_struct_Abbreviation),           NULL },
    { "Year",                   0,                                       get_year_object_size },
    { "Month",                  sizeof(C_struct_Month),                  NULL },
    { "Date",                   sizeof(C_struct_Date),                   NULL },
    { "Us State",               sizeof(C_struct_US_State),               NULL },
    { "Person",                 sizeof(C_struct_Person),                 NULL },
    { "Social Security Number", sizeof(C_struct_Social_Security_Number), NULL },
    { "Gender",                 sizeof(C_struct_Gender),                 NULL },
    { "Area Code",              sizeof(C_struct_Area_Code),              NULL },
    { "Phone Number",           sizeof(C_struct_Phone_Number),           NULL },
    { "Home Phone",             sizeof(C_struct_Home_Phone),             NULL },
    { "Work Phone",             sizeof(C_struct_Work_Phone),             NULL },
    { "Mobile Phone",           sizeof(C_struct_Mobile_Phone),           NULL },
    { "Md5sum",                 sizeof(C_struct_MD5sum),                 NULL },
    { "Us City",                sizeof(C_struct_US_City),                NULL },
    };

#define NUM_CLASSES ((sizeof(class_tbl))/sizeof(ClassTbl))


struct {
  ObjRef ref;
  void *ptr;
  size_t size;
} objs_to_save[2048];

int num_objs_to_save = 0;

ObjRef root_class_ref_list;



void save_objects(ObjRef* ref_list_ref, char* class_name)
{
    ReferenceList* ref_list;
    ObjRef object_class;
    int num_objects;
    int i, j;
    EveryObject header;
    C_struct_Class_Definition class_def_obj;
    char buffer[65];
    uint8 kludge[FILE_BLOCK_SIZE];

    ref_list = nwos_malloc_reference_list(ref_list_ref);

    num_objects = ref_list->common_header.num_refs;

    printf("num_objects: %d\n", num_objects);


    for (i = 0; i < num_objects; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	assert(nwos_read_object_from_disk(&object_class, &class_def_obj, sizeof(class_def_obj)));
	
	nwos_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));   /* get it's name */
    
	printf("reference: %02x%02x%02x%02x   class: %s\n",
	       ref_list->references[i].id[0],
	       ref_list->references[i].id[1],
	       ref_list->references[i].id[2],
	       ref_list->references[i].id[3],
	       buffer);

	if (strcmp(buffer, class_name) == 0)   /* only process if it matches the class */
	{

	    for (j = 0; j < NUM_CLASSES; j++)
	    {
		if (strcmp(buffer, class_tbl[j].name) == 0) break;
	    }

	    if (j < NUM_CLASSES)  /* all is well */
	    {
		if (class_tbl[j].size != 0)
		{
		    memcpy(&objs_to_save[num_objs_to_save].ref, &ref_list->references[i], 4);

		    objs_to_save[num_objs_to_save].ptr = malloc(class_tbl[j].size);
		    assert(objs_to_save[num_objs_to_save].ptr);

		    objs_to_save[num_objs_to_save].size = class_tbl[j].size;

		    assert(nwos_read_object_from_disk(&objs_to_save[num_objs_to_save].ref,
					       objs_to_save[num_objs_to_save].ptr,
					       objs_to_save[num_objs_to_save].size));

		    if (strcmp(buffer, "Word") == 0)
		    {
			C_struct_Word* word_obj = (C_struct_Word*) objs_to_save[num_objs_to_save].ptr;
			printf("word spelling: %02x%02x%02x%02x\n",
			       word_obj->spelling.id[0],
			       word_obj->spelling.id[1],
			       word_obj->spelling.id[2],
			       word_obj->spelling.id[3]);
		    }

		    num_objs_to_save++;
		}
		else if (class_tbl[j].func != NULL)
		{
		    memcpy(&objs_to_save[num_objs_to_save].ref, &ref_list->references[i], 4);

		    assert(nwos_read_variable_sized_object_from_disk(&objs_to_save[num_objs_to_save].ref, kludge, class_tbl[j].func));

		    /* remember ptr_name_obj points to the kludge buffer */

		    objs_to_save[num_objs_to_save].size = class_tbl[j].func(kludge);

		    objs_to_save[num_objs_to_save].ptr = malloc(objs_to_save[num_objs_to_save].size);
		    assert(objs_to_save[num_objs_to_save].ptr);

		    memcpy(objs_to_save[num_objs_to_save].ptr, kludge, objs_to_save[num_objs_to_save].size);

		    if (strcmp(buffer, "Spelling") == 0)
		    {
			int k;
			C_struct_Spelling* spelling_obj = (C_struct_Spelling*) objs_to_save[num_objs_to_save].ptr;
		    
			printf("%d letters: ", spelling_obj->count);
			for (k = 0; k < spelling_obj->count; k++) 
			  if (isprint(spelling_obj->storage[k]))
			    printf("%c", spelling_obj->storage[k]);
			  else
			    printf("(%02x)", spelling_obj->storage[k]);

			printf("\n");
		    }

		    num_objs_to_save++;
		}
		else
		{
		    assert(strcmp(buffer, "Root") == 0);
		    memcpy(&root_class_ref_list, ref_list_ref, sizeof(root_class_ref_list));
		}
	    }
	    else
	    {
		printf("Error: no matching class in class_tbl\n");
		exit(1);
	    }
	}
	else
	{
	    printf("Wrong class, skipped\n");
	}
	

#if 0
	if (is_same_object(&object_class, &class_def_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &class_def_objs[i], sizeof(class_def_objs[0])));

	    strlcpy(buffer, (char*)&header.common.version, 4);

	    buffer[4] = '\0';
	    printf("version: %s", buffer);

	    printf("  name: %s\n", buffer);

	    assert(nwos_read_object_from_disk(&ref_list->references[i], &class_def_objs[i], sizeof(class_def_objs[0])));

	    strlcpy(buffer, (char*)&header.common.version, 4);

	    buffer[4] = '\0';
	    printf("version: %s", buffer);

	    nwos_name_to_string(&class_def_objs[i].name, buffer, sizeof(buffer));   /* get it's name */
	    printf("  name: %s\n", buffer);

	    assert(nwos_read_object_from_disk(&ref_list->references[i], &class_def_objs[i], sizeof(class_def_objs[0])));

	    strlcpy(buffer, (char*)&header.common.version, 4);

	    buffer[4] = '\0';
	    printf("version: %s", buffer);

	    nwos_name_to_string(&class_def_objs[i].name, buffer, sizeof(buffer));   /* get it's name */
	    printf("  name: %s\n", buffer);

	    memcpy(&objs_to_save[num_objs_to_save].ref, &ref_list->references[i], 4);
	    objs_to_save[num_objs_to_save].ptr = &class_def_objs[i];
	    objs_to_save[num_objs_to_save].size = sizeof(class_def_objs[0]);
	    num_objs_to_save++;
	}
#endif
    }

    free(ref_list);
    ref_list = NULL;
}


#define OBJ_PATH_SIZE 128

static void ref_to_path(ObjRef* ref, char path[OBJ_PATH_SIZE])
{
    assert(strlcpy(path, OBJECT_DIRECTORY, OBJECT_PATH_SIZE) < OBJECT_PATH_SIZE);
    nwos_ref_to_name(ref, &path[strlen(path)]);
}

static void write_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    char path[OBJ_PATH_SIZE];
    char msg[OBJ_PATH_SIZE + 16];
    FILE* fp;
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];

    assert(!nwos_read_block(ref, block));

    ref_to_path(ref, path);
#if 0
    sprintf(log_msg, "nwos_write_object_to_disk - %s size: %d", path, size);
    log(log_msg);
#endif
    fp = fopen(path, "w");

    if (fp == NULL) 
    {
	sprintf(msg, "opening %s", path);
	perror(msg);
	exit(1);
    }

    if (fwrite(object, 1, size, fp) != size)
    {
	sprintf(msg, "writing %s", path);
	perror(msg);
	exit(1);
    }

    if (fclose(fp) != 0) 
    {
	sprintf(msg, "closing %s", path);
	perror(msg);
	exit(1);
    }
}


int main(int argc, char* argv[])
{
    ObjRef old_root_object_ref;
    ObjRef new_root_object_ref;
    ObjRef class_def_class_ref;
    ObjRef object_class;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    int i;
    int num_classes;
    C_struct_Root root_obj;
    C_struct_Class_Definition class_def_class_obj;
    C_struct_Class_Definition class_def_objs[24];
    ReferenceList* ref_list;
    char buffer[65];

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(old_root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial);


    nwos_set_root_object(&old_root_object_ref);

    printf("root object ref: %02x%02x%02x%02x\n", 
	   old_root_object_ref.id[0],
	   old_root_object_ref.id[1],
	   old_root_object_ref.id[2],
	   old_root_object_ref.id[3]);

    assert(nwos_read_object_from_disk(&old_root_object_ref, &root_obj, sizeof(root_obj)));

    memcpy(&objs_to_save[num_objs_to_save].ref, &old_root_object_ref, 4);
    objs_to_save[num_objs_to_save].ptr = &root_obj;
    objs_to_save[num_objs_to_save].size = sizeof(root_obj);
    num_objs_to_save++;

    printf("class_definition_class: %02x%02x%02x%02x\n", 
	   root_obj.class_definition_class.id[0],
	   root_obj.class_definition_class.id[1],
	   root_obj.class_definition_class.id[2],
	   root_obj.class_definition_class.id[3]);

    memcpy(&class_def_class_ref, &root_obj.class_definition_class, 4);

    assert(nwos_read_object_from_disk(&class_def_class_ref, &class_def_class_obj, sizeof(class_def_class_obj)));

    printf("class_def_class_references: %02x%02x%02x%02x\n", 
	   class_def_class_obj.header.object.references.id[0],
	   class_def_class_obj.header.object.references.id[1],
	   class_def_class_obj.header.object.references.id[2],
	   class_def_class_obj.header.object.references.id[3]);

    memcpy(&objs_to_save[num_objs_to_save].ref, &class_def_class_ref, 4);
    objs_to_save[num_objs_to_save].ptr = &class_def_class_obj;
    objs_to_save[num_objs_to_save].size = sizeof(class_def_class_obj);
    num_objs_to_save++;

    ref_list = nwos_malloc_reference_list(&class_def_class_obj.header.object.references);

    num_classes = ref_list->common_header.num_refs;

    printf("num_classes: %d\n", num_classes);


    for (i = 0; i < num_classes; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &class_def_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &class_def_objs[i], sizeof(class_def_objs[0])));

	    strlcpy(buffer, (char*)&class_def_objs[i].header.common.version, 4);

	    buffer[4] = '\0';
	    printf("version: %s", buffer);

	    nwos_name_to_string(&class_def_objs[i].name, buffer, sizeof(buffer));   /* get it's name */

	    memcpy(&objs_to_save[num_objs_to_save].ref, &ref_list->references[i], 4);
	    objs_to_save[num_objs_to_save].ptr = &class_def_objs[i];
	    objs_to_save[num_objs_to_save].size = sizeof(class_def_objs[0]);
	    num_objs_to_save++;

	    printf("  name: %s   reference list: %02x%02x%02x%02x\n", 
		   buffer,
		   class_def_objs[i].header.object.references.id[0],
		   class_def_objs[i].header.object.references.id[1],
		   class_def_objs[i].header.object.references.id[2],
		   class_def_objs[i].header.object.references.id[3]);

	    if (is_void_reference(&class_def_objs[i].header.object.references))
	    {
		printf("references is void\n");
	    }
	    else
	    {
		save_objects(&class_def_objs[i].header.object.references, buffer);
	    }
	}
	else
	{
	    printf("Error: not a class definition object!\n");
	}
    }

    free(ref_list);
    ref_list = NULL;
#if 0
	assert(nwos_read_object_from_disk(&ref_list->references[i], &old_month_obj, sizeof(old_month_obj)));
#endif

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(new_root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial);

    ref_list = nwos_malloc_reference_list(&root_class_ref_list);

    /* change the root object in the root class reference list */

    assert(ref_list->common_header.num_refs == 1);

    memcpy(&ref_list->references[0].id, &new_root_object_ref, sizeof(ObjRef));

    nwos_remove_object(&root_class_ref_list);

    /* I seriously doubt this works anymore, reference lists are encrypted now! */
    write_object_to_disk(&root_class_ref_list, ref_list, sizeof(ReferenceList) + sizeof(ObjRef));

    /* delete the old root object */

    printf("delete old root object: %02x%02x%02x%02x\n",
	   objs_to_save[i].ref.id[0],
	   objs_to_save[i].ref.id[1],
	   objs_to_save[i].ref.id[2],
	   objs_to_save[i].ref.id[3]);

    nwos_remove_object(&old_root_object_ref);

    assert(memcmp(&objs_to_save[0].ref, &old_root_object_ref, 4) == 0);
    
    printf("new root object: %02x%02x%02x%02x  ptr: %p  size: %d\n",
	   new_root_object_ref.id[0],
	   new_root_object_ref.id[1],
	   new_root_object_ref.id[2],
	   new_root_object_ref.id[3],
	   objs_to_save[i].ptr,
	   objs_to_save[i].size);

    nwos_write_object_to_disk(&new_root_object_ref, objs_to_save[0].ptr, objs_to_save[0].size);

    for (i=1; i < num_objs_to_save; i++)
    {
	printf("reference: %02x%02x%02x%02x  ptr: %p  size: %d\n",
	       objs_to_save[i].ref.id[0],
	       objs_to_save[i].ref.id[1],
	       objs_to_save[i].ref.id[2],
	       objs_to_save[i].ref.id[3],
	       objs_to_save[i].ptr,
	       objs_to_save[i].size);

	nwos_remove_object(&objs_to_save[i].ref);

	nwos_write_object_to_disk(&objs_to_save[i].ref, objs_to_save[i].ptr, objs_to_save[i].size);
    }

    return 0;
}

