/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-13 20:06:13 -0600 (Thu, 13 Aug 2009) $
--   $Revision: 4329 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revisions of this file and the objectify.c file which this file
--   was taken from.  (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>      /* define memset */
#include <time.h>

#include "backup.h"
#include "bit_map.h"
#include "chunk_info.h"
#include "disk_io.h"
#include "gen_id.h"
#include "log.h"


/* some blocks don't come out even when dividing usable chunks per block, so don't allocate them */

#define RESERVED_PRIVATE_CHUNKS (RESERVED_PRIVATE_BLOCKS / USABLE_BLOCKS_PER_CHUNK)
/*                              (0xeff00000              / 65504)    = 61454 */

ObjRef nwos_next_public_ref;

#ifndef PUBLIC_MODE

#ifdef CHANGE_SECURITY_TO_DENSITY
static Security_Level security_level = Security_None;
#endif
static uint32 block_estimate;
static uint32 last_used_blocks;

static ObjRef last_ref;

#ifdef CHANGE_SECURITY_TO_DENSITY
static uint32 chunk_density_target;
static uint32 chunk_skip_forward;
#endif

#endif


#define SECURITY_VERY_LOW_N  7     /* sort of arbitrary, but it's always fun to use prime numbers */
#define SECURITY_LOW_N      43     /* in this case each digit is represented */
#define SECURITY_MEDIUM_N  509
#define SECURITY_HIGH_N   2861


/********************************************/
/* Function to initialize the gen_id module */
/********************************************/

#ifdef DISABLE_SECURITY_FEATURES
static uint32 randomxxx;
#endif

void nwos_initialize_gen_id(void)
{
#ifdef HAVE_CLOCK_GETTIME
    struct timespec ts;

    clock_gettime(CLOCK_REALTIME, &ts);
    srandom((uint32)ts.tv_sec ^ (uint32)ts.tv_nsec);
#else
    struct timeval tv;

    gettimeofday(&tv, NULL);
    srandom((uint32)tv.tv_sec ^ (uint32)tv.tv_usec);
#endif

#ifdef DISABLE_SECURITY_FEATURES
    randomxxx = 0x00000100;     /* start objects here */
#endif
}


/*****************************************************/
/* Function to log the results of the block estimate */
/*****************************************************/

static void log_estimate_results()
{
#ifndef PUBLIC_MODE
    char log_msg[128];

    uint32 actual = nwos_used_private_blocks - last_used_blocks;

    snprintf(log_msg, sizeof(log_msg), "  block estimate: %u  actual: %u", block_estimate, actual);

    nwos_log(log_msg);
#endif
}

/********************************************/
/* Function to terminate the gen_id module */
/********************************************/

void nwos_terminate_gen_id(void)
{
    log_estimate_results();
}



#ifndef PUBLIC_MODE

#ifdef CHANGE_SECURITY_TO_DENSITY
static void nwos_set_allocation_parameters(uint32 density, uint32 skip_forward)
{
    assert(1 <= density && density <= USABLE_BLOCKS_PER_CHUNK);
    assert(1 <= skip_forward && skip_forward <= RESERVED_PRIVATE_CHUNKS);

    chunk_density_target = density;
    chunk_skip_forward = skip_forward;
}


void nwos_restart_id_generation()
{
    void_reference(&last_ref);
}


void nwos_set_security_level(Security_Level new_level)
{
    bool Security_Not_Set = false;

    assert(Security_None < new_level && new_level <= Security_Extreme);

    security_level = new_level;

    switch (new_level)
    {
      case Security_Minimal:
	nwos_set_allocation_parameters( ((USABLE_BLOCKS_PER_CHUNK * 7) / 8), 1);
	break;

      case Security_Very_Low:
	nwos_set_allocation_parameters( (USABLE_BLOCKS_PER_CHUNK / SECURITY_VERY_LOW_N), 4);
	break;

      case Security_Low:
	nwos_set_allocation_parameters( (USABLE_BLOCKS_PER_CHUNK / SECURITY_LOW_N), 16);
	break;

      case Security_Medium:
	nwos_set_allocation_parameters( (USABLE_BLOCKS_PER_CHUNK / SECURITY_MEDIUM_N), 32);
	break;

      case Security_High:
	nwos_set_allocation_parameters( (USABLE_BLOCKS_PER_CHUNK / SECURITY_HIGH_N), 128);
	break;

      case Security_Extreme:
	nwos_set_allocation_parameters(1, RESERVED_PRIVATE_CHUNKS);
	break;

      default:
	assert(Security_Not_Set);
	break;
    }

    nwos_restart_id_generation();   /* clear last_ref so this security level starts fresh */
}
#endif  /* CHANGE_SECURITY_TO_DENSITY */


/*************************/
/* Block estimate stuff. */
/*************************/

static uint32 blocks_in_contiguous_chunks(int info_index, uint32 maximum_chunks)
{
    int max_index;
    int i;
    uint32 result;

    max_index = info_index + maximum_chunks;

    if (max_index > nwos_used_private_chunks) max_index = nwos_used_private_chunks;

    result = USABLE_BLOCKS_PER_CHUNK - nwos_blocks_used_in_chunk(info_index);

    for (i = info_index + 1; i < max_index; i++)
    {
	if (nwos_chunk_info[i-1].ref + USABLE_BLOCKS_PER_CHUNK != nwos_chunk_info[i].ref) break;  /* not contiguous */

	result += USABLE_BLOCKS_PER_CHUNK - nwos_blocks_used_in_chunk(i);
    }

    return result;
}


static int find_starting_chunk_for_blocks(uint32 estimated_blocks)
{
    uint32 maximum_chunks;
    uint32 available_blocks;
    int info_index;

    maximum_chunks = estimated_blocks / USABLE_BLOCKS_PER_CHUNK + 2;     /* always allow at least 2 chunks so that we use space at the end of the chunk */

    for (info_index = 0; info_index < nwos_used_private_chunks; info_index++)
    {
	available_blocks = blocks_in_contiguous_chunks(info_index, maximum_chunks);

	if (available_blocks > estimated_blocks) break;
    }

    assert(info_index <= nwos_used_private_chunks);

    if (info_index == nwos_used_private_chunks)    /* didn't find a spot */
    {
	info_index = -1;
    }

    return info_index;
}


static int number_of_chunk_islands(void)
{
    int i;
    int result = 1;

    for (i = 1; i < nwos_used_private_chunks; i++)
    {
	if (nwos_chunk_info[i-1].ref + USABLE_BLOCKS_PER_CHUNK != nwos_chunk_info[i].ref)  /* not contiguous */
	{
	    result++;
	}
    }

    return result;
}


/* returns the info index of the last chunk in the chunk island with the fewest chunks */
static int find_smallest_chunk_island(uint32 needed_chunks)
{
    uint32 chunks_in_island;
    uint64 chunks_after_island;
    uint64 next_ref;
    uint32 min_chunks = nwos_used_private_chunks;
    int i;
    int info_index = nwos_used_private_chunks - 1;

    i = 1;

    while (i < nwos_used_private_chunks)
    {
	chunks_in_island = 1;

	while (i < nwos_used_private_chunks && nwos_chunk_info[i-1].ref + USABLE_BLOCKS_PER_CHUNK == nwos_chunk_info[i].ref)    /* count contiguous chunks */
	{
	    chunks_in_island++;
	    i++;
	}

	if (i == nwos_used_private_chunks)    /* this is the last chunk */
	{
	    next_ref = MAXIMUM_32_BIT_PRIVATE_REFERENCE + 1;
	}
	else
	{
	    next_ref = nwos_chunk_info[i].ref;
	}

	chunks_after_island = (next_ref - nwos_chunk_info[i-1].ref) / USABLE_BLOCKS_PER_CHUNK - 1;

	assert((chunks_after_island + 1) * USABLE_BLOCKS_PER_CHUNK == next_ref - nwos_chunk_info[i-1].ref);

	if (chunks_in_island < min_chunks && needed_chunks <= chunks_after_island)
	{
	    info_index = i - 1;
	    min_chunks = chunks_in_island;
	}

	i++;
    }

    return info_index;
}


static uint64 find_new_random_chunk_location(uint32 needed_chunks)
{
    uint64 lower_ref;
    uint64 upper_ref;
    uint32 num_chunks;
    uint64 ref = 0;
    int i;

    lower_ref = MINIMUM_32_BIT_PRIVATE_REFERENCE;
    upper_ref = nwos_chunk_info[0].ref;

    for (i = 1; i < nwos_used_private_chunks; i++)
    {
	if (nwos_chunk_info[i-1].ref + USABLE_BLOCKS_PER_CHUNK != nwos_chunk_info[i].ref)  /* not contiguous */
	{
	    if (nwos_chunk_info[i].ref - nwos_chunk_info[i-1].ref > upper_ref - lower_ref)
	    {
		lower_ref = nwos_chunk_info[i-1].ref;
		upper_ref = nwos_chunk_info[i].ref;
	    }
	}
    }

    if (MAXIMUM_32_BIT_PRIVATE_REFERENCE + 1 - nwos_chunk_info[i-1].ref > upper_ref - lower_ref)
    {
	lower_ref = nwos_chunk_info[i-1].ref;
	upper_ref = MAXIMUM_32_BIT_PRIVATE_REFERENCE + 1;
    }

    assert((lower_ref - MINIMUM_32_BIT_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK == 0);
    assert((upper_ref - MINIMUM_32_BIT_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK == 0);

    num_chunks = (upper_ref - lower_ref) / USABLE_BLOCKS_PER_CHUNK;

    if (needed_chunks <= num_chunks)
    {
	ref = lower_ref + (random() % (num_chunks - needed_chunks)) * USABLE_BLOCKS_PER_CHUNK;

	assert((ref - MINIMUM_32_BIT_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK == 0);
    }

    return ref;
}


bool nwos_check_blocks_available(uint32 estimated_blocks)
{
    uint64 total_blocks;
    uint32 needed_blocks;
    uint32 needed_chunks;
    uint32 allocated_blocks;
#ifdef CHANGE_SECURITY_TO_DENSITY
    uint32 skip;
    uint32 max;
#endif
    uint64 ref;
    uint32 offset;
    char msg[256];
    int i;
    int num_islands;
    int info_index = -1;


    assert(estimated_blocks > 0);

#ifdef CHANGE_SECURITY_TO_DENSITY
    assert(1 <= chunk_density_target && chunk_density_target <= USABLE_BLOCKS_PER_CHUNK);
    assert(1 <= chunk_skip_forward && chunk_skip_forward <= RESERVED_PRIVATE_CHUNKS);
#endif

    if (block_estimate > 0)   /* there was a previous estimate */
    {
	log_estimate_results();
    }

    block_estimate = 0;

    last_used_blocks = nwos_used_private_blocks;

    /* make sure we have enough space */

    total_blocks = nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK;
    needed_blocks = nwos_used_private_blocks + estimated_blocks + (USABLE_BLOCKS_PER_CHUNK / 2);

    /* first see if there may be a spot for it already */

    if (total_blocks < needed_blocks)     /* not enough room */
    {
	needed_chunks = (needed_blocks + USABLE_BLOCKS_PER_CHUNK - 1) / USABLE_BLOCKS_PER_CHUNK;
    }
    else                                  /* there might be room */
    {
	needed_chunks = 0;   /* assume we will find a place */

	info_index = find_starting_chunk_for_blocks(estimated_blocks);

	if (info_index < 0)
	{
	    needed_chunks = (needed_blocks + USABLE_BLOCKS_PER_CHUNK - 1) / USABLE_BLOCKS_PER_CHUNK;
	}
    }

#if 0
    printf("Total chunks: %u  blocks: %llu\n", nwos_total_private_chunks, total_blocks);
    printf("Used chunks: %u  blocks: %llu\n", nwos_used_private_chunks, nwos_used_private_blocks);
#endif

    if (needed_chunks > nwos_total_private_chunks)
    {
	if (nwos_archive_is_block_device())
	{
	    fprintf(stderr, "\n");
	    fprintf(stderr, "Archive is too full:\n");
	    fprintf(stderr, "      Total blocks: %llu\n", total_blocks);
	    fprintf(stderr, "       Used blocks: %llu\n", nwos_used_private_blocks);
	    fprintf(stderr, "  Available blocks: %llu\n", total_blocks - nwos_used_private_blocks);
	    fprintf(stderr, "  Estimated blocks: %u\n", estimated_blocks);
	    fprintf(stderr, "\n");

	    return false;
	}
	else   /* archive is a file, expand it */
	{
	    while (needed_chunks > nwos_total_private_chunks)
	    {
		nwos_write_empty_chunk(nwos_total_private_chunks);
	    }
	}
    }

    assert(nwos_used_private_blocks + estimated_blocks < nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK);

    allocated_blocks = nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK;

    while (allocated_blocks < needed_blocks)
    {
	assert(nwos_used_private_chunks < nwos_total_private_chunks);

	/* temporary hack to handle the situation where we have reached the end of the private chunks */
	if (nwos_chunk_info[nwos_used_private_chunks-1].ref + USABLE_BLOCKS_PER_CHUNK > MAXIMUM_32_BIT_PRIVATE_REFERENCE)
	{
	    snprintf(msg, sizeof(msg), "WARNING: reached the end of private space, working backwards.");
	    nwos_log(msg);
	    fprintf(stderr, "%s\n", msg);

	    for (i = nwos_used_private_chunks - 1; i > 0; i--)
	    {
		if (nwos_chunk_info[i-1].ref < nwos_chunk_info[i].ref - USABLE_BLOCKS_PER_CHUNK) break;
	    }

	    ref = nwos_chunk_info[i].ref - USABLE_BLOCKS_PER_CHUNK;
	}
#ifdef CHANGE_SECURITY_TO_DENSITY
	else if (chunk_skip_forward == 1)
	{
	    ref = nwos_chunk_info[nwos_used_private_chunks-1].ref + USABLE_BLOCKS_PER_CHUNK;
	}
	else if (chunk_skip_forward == RESERVED_PRIVATE_CHUNKS)
	{
	    max = MAXIMUM_32_BIT_PRIVATE_REFERENCE - nwos_chunk_info[nwos_used_private_chunks-1].ref;
	    skip = 1 + random() % (max / USABLE_BLOCKS_PER_CHUNK);
	    ref = nwos_chunk_info[nwos_used_private_chunks-1].ref + (USABLE_BLOCKS_PER_CHUNK * skip);
	}
	else
	{
	    max = (MAXIMUM_32_BIT_PRIVATE_REFERENCE - nwos_chunk_info[nwos_used_private_chunks-1].ref) / USABLE_BLOCKS_PER_CHUNK;
	    if (max > chunk_skip_forward) max = chunk_skip_forward;
	    skip = 1 + random() % max;
	    ref = nwos_chunk_info[nwos_used_private_chunks-1].ref + (USABLE_BLOCKS_PER_CHUNK * skip);
	}
#else
	else
	{
	    ref = nwos_chunk_info[nwos_used_private_chunks-1].ref + USABLE_BLOCKS_PER_CHUNK;
	}
#endif
	snprintf(msg, sizeof(msg), "Allocating chunk: %llx", ref);
	nwos_log(msg);

	nwos_allocate_new_chunk((uint32)(ref - BASE_32_BIT_PRIVATE));

	/* make sure it increased */
	assert(nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK > allocated_blocks);

	allocated_blocks = nwos_used_private_chunks * USABLE_BLOCKS_PER_CHUNK;

	/* if we now theoretically have enough blocks, see if there is a spot that will work */
	if (needed_blocks <= allocated_blocks && info_index < 0 && nwos_used_private_chunks < nwos_total_private_chunks)
	{
	    info_index = find_starting_chunk_for_blocks(estimated_blocks);

	    if (info_index < 0)    /* didn't find a good spot */
	    {
		needed_blocks += USABLE_BLOCKS_PER_CHUNK;    /* allocate another chunk and try again */
	    }
	}
    }

    block_estimate = estimated_blocks;

    /* if we didn't already find a spot, find a place to start it */

    if (info_index < 0)
    {
	info_index = find_starting_chunk_for_blocks(estimated_blocks);

	if (info_index < 0)
	{
	    if (nwos_archive_is_block_device() && nwos_used_private_chunks == nwos_total_private_chunks)
	    {
		fprintf(stderr, "WARNING: could not find contiguous chunks for storage and cannot allocate more chunks because archive is full\n");

		for (info_index = 0; info_index < nwos_used_private_chunks; info_index++)    /* let the pieces fall where they may */
		{
		    if (!nwos_chunk_is_full(info_index)) break;
		}
	    }
	    else   /* make more room */
	    {
		needed_chunks = (estimated_blocks + USABLE_BLOCKS_PER_CHUNK - 1) / USABLE_BLOCKS_PER_CHUNK;    /* new chunks needed */

		if (nwos_archive_is_file())
		{
		    for (i = 0; i < needed_chunks; i++)
		    {
			nwos_write_empty_chunk(nwos_total_private_chunks);
		    }
		}

		num_islands = number_of_chunk_islands();

		if (random() % (num_islands + 1) == 0)   /* if one island there is a 50% chance of creating new, if two a 33% chance and so on */
		{
		    ref = find_new_random_chunk_location(needed_chunks);
		    info_index = -1;
		}
		else
		{
		    info_index = find_smallest_chunk_island(needed_chunks);
		    ref = nwos_chunk_info[info_index].ref + USABLE_BLOCKS_PER_CHUNK;
		}

		for (i = 0; i < needed_chunks; i++)
		{
		    if (nwos_hash_uint32_ref((uint32)(ref - BASE_32_BIT_PRIVATE)) != 0)     /* only allocate if this chunk isn't already allocated */
		    {
			snprintf(msg, sizeof(msg), "WARNING: tried to allocate chunk that was already allocated - i: %d  info_index: %d  ref: %llx", i, info_index, ref);
			nwos_log(msg);
			fprintf(stderr, "%s\n", msg);

			while (ref < MAXIMUM_32_BIT_PRIVATE_REFERENCE && nwos_hash_uint32_ref((uint32)(ref - BASE_32_BIT_PRIVATE)) != 0)    /* attempt to find the next available spot */
			{
			    ref += USABLE_BLOCKS_PER_CHUNK;
			}

			if (ref == MAXIMUM_32_BIT_PRIVATE_REFERENCE)    /* unsucessful */
			{
			    snprintf(msg, sizeof(msg), "BUG: could not find a chunk to allocate - i: %d  info_index: %d  ref: %llx", i, info_index, ref);
			    nwos_log(msg);
			    fprintf(stderr, "%s\n", msg);

			    return false;      /* say we couldn't find enough space */
			}
		    }

		    nwos_allocate_new_chunk((uint32)(ref - BASE_32_BIT_PRIVATE));

		    if (info_index < 0)    /* this is the first time through, find the info_index for the chunk we just created */
		    {
			for (i = 0; i < nwos_used_private_chunks; i++)
			{
			    if (nwos_chunk_info[i].ref == ref)
			    {
				info_index = i;
				break;
			    }
			}

			assert(0 <= info_index && info_index < nwos_used_private_chunks);
		    }

		    ref += USABLE_BLOCKS_PER_CHUNK;
		}
	    }
	}
    }

    if (block_estimate < 5 && nwos_blocks_used_in_chunk(info_index) < 5)       /* allow random location for root objects */
    {
	offset = random() % (USABLE_BLOCKS_PER_CHUNK - 5);
	nwos_word_to_ref((uint32)(nwos_chunk_info[info_index].ref - BASE_32_BIT_PRIVATE) + offset, &last_ref);
    }
    else
    {
	nwos_word_to_ref((uint32)(nwos_chunk_info[info_index].ref - BASE_32_BIT_PRIVATE), &last_ref);
    }

    return true;
}

#endif


/**********************************/
/* Functions to generate new id's */
/**********************************/

#ifdef PUBLIC_MODE
void nwos_generate_new_public_id(ObjRef* ref)
{
    int i;
    int acc;
    uint8 diff[4];

    copy_reference(ref, &nwos_next_public_ref);

    for (i = sizeof(nwos_next_public_ref)-1; i > -1; i--)
    {
	nwos_next_public_ref.id[i]++;
	if (nwos_next_public_ref.id[i] != 0) break;
    }

    acc = (int)nwos_next_public_ref.id[3] - (int)ref->id[3];
    diff[3] = acc;
    acc = (int)nwos_next_public_ref.id[2] - (int)ref->id[2] + (acc >> 8);
    diff[2] = acc;
    acc = (int)nwos_next_public_ref.id[1] - (int)ref->id[1] + (acc >> 8);
    diff[1] = acc;
    acc = (int)nwos_next_public_ref.id[0] - (int)ref->id[0] + (acc >> 8);
    diff[0] = acc;

    assert(diff[0] == 0 && diff[1] == 0 && diff[2] == 0 && diff[3] == 1);

    nwos_used_public_blocks++;
}
#else


#ifdef CHANGE_SECURITY_TO_DENSITY

static void generate_new_completely_random_id(ObjRef* ref)
{
    int i;
    int info_index = 0;
    uint32 word;
#ifdef LOG_ID_GENERATION
    char msg[128];
#endif

    assert(nwos_used_private_chunks > 0);

    if (nwos_used_private_chunks > 1)
    {
	/* first try to find one with some empties at random */
	for (i = 0; i < nwos_used_private_chunks; i++)
	{
	    info_index = random() % nwos_used_private_chunks;

	    if (!nwos_chunk_is_full(chunk)) break;
	}

	if (i == nwos_used_private_chunks)   /* didn't find one, just do a search */
	{
	    for (info_index = 0; info_index < nwos_used_private_chunks; info_index++)
	    {
		if (!nwos_chunk_is_full(chunk)) break;
	    }
	}
    }

    assert(!nwos_chunk_is_full(chunk));

    word = nwos_find_random_block_in_chunk(info_index);

    nwos_word_to_ref(word, ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "generate_new_completely_random_id(%08x)", word);
    nwos_log(msg);
#endif

    assert(!nwos_block_used(ref));
}


static void nwos_generate_new_completely_random_id(ObjRef* ref)
{
    generate_new_completely_random_id(ref);
    nwos_set_bit_in_map(nwos_hash_ref(ref));
}



/* density is how many blocks to use 1-7 out of 8 blocks. */

static void nwos_generate_new_closely_spaced_id(int density, ObjRef* new_ref)
{
    uint32 ref;
    uint32 block;
    int    byte_num = 0;
    int    max_used;
    uint16 chunk_index;
    static int info_index = 0;
#ifdef LOG_ID_GENERATION
    char   msg[128];
#endif

    assert(1 <= density && density <= 7);
    assert(block_estimate > 0);

    max_used = (USABLE_BLOCKS_PER_CHUNK * (8 - density)) / 8;

    /* if the estimated blocks will fit in one chunk, reduce it to what we need */
    if (block_estimate <= USABLE_BLOCKS_PER_CHUNK - max_used)
    {
	max_used = USABLE_BLOCKS_PER_CHUNK - block_estimate;
    }

    if (is_void_reference(&last_ref))
    {
	block = nwos_block_offset_to_chunks + nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK;
    }
    else
    {
	block = nwos_hash_ref(&last_ref) + 1;
    }


    /* at this point 'block' should have the block number we are starting the search at */

    while (true)
    {
	/* first see if block has gone past the end of this chunk */

	if ((block - nwos_block_offset_to_chunks) % BLOCKS_IN_CHUNK < BIT_MAP_BLOCKS)
	{
	    chunk_index = (block - nwos_block_offset_to_chunks) / BLOCKS_IN_CHUNK;

	    info_index = nwos_chunk_index_to_info_index(chunk_index);

	    while (nwos_blocks_used_in_chunk(info_index) > max_used)
	    {
		/* see if the chunk after this one is usable, and if so back into this chunk */
		if (info_index + 1 == nwos_total_private_chunks)
		{
		    /* need to allocate a new chunk */
		    bool need_to_fix_allocation_code = false;
		    assert(need_to_fix_allocation_code);
		}
		else if (nwos_blocks_used_in_chunk(info_index + 1) <= max_used)
		{
		    byte_num = nwos_find_low_density_in_chunk(info_index, density);

		    if (BIT_MAP_BYTES - byte_num > 1) break;

		    byte_num = 0;    /* reset this to what it was */
		}

		info_index++;
	    }

	    block = nwos_block_offset_to_chunks + nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK + BIT_MAP_BLOCKS + (byte_num * 8);
	}

	/* at this point we have the block to try */

	block = nwos_find_next_empty_block(block, density);

	if ((block - nwos_block_offset_to_chunks) % BLOCKS_IN_CHUNK >= BIT_MAP_BLOCKS)
	{
	    break;  /* found an empty in that chunk */
	}
    }

    assert(nwos_test_bit_in_map(block) == false);

    nwos_set_bit_in_map(block);

    ref = nwos_chunk_info[info_index].ref + (block - nwos_block_offset_to_chunks - nwos_chunk_info[info_index].index * BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS);

    nwos_word_to_ref(ref, new_ref);

    copy_reference(&last_ref, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_closely_spaced_id(%08x)", ref);
    nwos_log(msg);
#endif
}


/* generate approximately 1 block out of N */

static void nwos_generate_new_1_in_N_id(uint32 n, ObjRef* new_ref)
{
    uint32 ref;
    uint32 hash;
#ifdef LOG_ID_GENERATION
    char   msg[128];
#endif


    if (is_void_reference(&last_ref))
    {
	generate_new_completely_random_id(new_ref);

	ref = nwos_ref_to_word(new_ref);
    }
    else
    {
	ref = nwos_ref_to_word(&last_ref) + (random() % n) + 1;
    }

    while (true)
    {
	hash = nwos_hash_uint32_ref(ref);

	if (hash == 0)  /* continue this somewhere else */
	{
	    generate_new_completely_random_id(new_ref);

	    ref = nwos_ref_to_word(new_ref);

	    hash = nwos_hash_uint32_ref(ref);
	}	    
	else if (!nwos_test_bit_in_map(hash))   /* available */
	{
	    break;
	}

	ref++;    /* try the next one */
    }

    assert(hash == nwos_hash_uint32_ref(ref));
    assert(!nwos_test_bit_in_map(hash));

    nwos_set_bit_in_map(hash);

    nwos_word_to_ref(ref, new_ref);

    copy_reference(&last_ref, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_1_in_N_id(%08x)", ref);
    nwos_log(msg);
#endif
}

#endif  /* CHANGE_SECURITY_TO_DENSITY */


void nwos_generate_new_private_id(ObjRef* new_ref)
{
    uint32 ref;
    uint32 hash;
    int info_index;
    char   msg[128];

    assert(!is_void_reference(&last_ref));

    ref = nwos_ref_to_word(&last_ref);

    while (true)
    {
	hash = nwos_hash_uint32_ref(ref);

	if (hash == 0)  /* continue this somewhere else */
	{
	    snprintf(msg, sizeof(msg), "BUG: nwos_generate_new_private_id could not allocate all IDs had to restart");
	    nwos_log(msg);
	    fprintf(stderr, "%s\n", msg);

	    snprintf(msg, sizeof(msg), "   last_ref: %08x", nwos_ref_to_word(&last_ref));
	    nwos_log(msg);
	    fprintf(stderr, "%s\n", msg);

	    for (info_index = 0; info_index < nwos_used_private_chunks; info_index++)
	    {
		if (!nwos_chunk_is_full(info_index)) break;
	    }

	    ref = nwos_chunk_info[info_index].ref;
	}
	else if (nwos_test_bit_in_map(hash))   /* not available */
	{
	    ref++;    /* try the next one */
	}
	else   /* available */
	{
	    break;
	}
    }

    assert(hash == nwos_hash_uint32_ref(ref));
    assert(!nwos_test_bit_in_map(hash));

    nwos_set_bit_in_map(hash);

    nwos_word_to_ref(ref, new_ref);

    copy_reference(&last_ref, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_1_in_N_id(%08x)", ref);
    nwos_log(msg);
#endif
}
#endif  /* not PUBLIC_MODE */


void nwos_generate_new_id(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    nwos_generate_new_public_id(ref);

    assert(nwos_reference_type(ref) == Public_Reference);
#else
    nwos_generate_new_private_id(ref);

    assert(nwos_reference_type(ref) == Private_Reference);
#endif
}


/* This is the old code from nwos_generate_new_id that needs to be changed to control density not security */
#ifdef CHANGE_SECURITY_TO_DENSITY
#if 0
    bool Cant_Use_Security_None = false;
#endif
    bool Security_Not_Set = false;

    switch (security_level)
    {
#if 0
      case Security_None:
	assert(Cant_Use_Security_None);
	break;
#endif

      case Security_Minimal:
	nwos_generate_new_closely_spaced_id(7, ref);
	break;

      case Security_Very_Low:
	nwos_generate_new_1_in_N_id(SECURITY_VERY_LOW_N, ref);
	break;

      case Security_Low:
	nwos_generate_new_1_in_N_id(SECURITY_LOW_N, ref);
	break;

      case Security_Medium:
	nwos_generate_new_1_in_N_id(SECURITY_MEDIUM_N, ref);
	break;

      case Security_High:
	nwos_generate_new_1_in_N_id(SECURITY_HIGH_N, ref);
	break;

      default:
	assert(Security_Not_Set);
	break;

      case Security_Extreme:
	nwos_generate_new_completely_random_id(ref);
	break;
    }

    assert(nwos_reference_type(ref) == Private_Reference);
#endif


