/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-21 05:54:56 -0700 (Thu, 21 Jan 2010) $
--   $Revision: 4470 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


int main(int argc, char* argv[])
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_MD5sum md5_obj;
    C_struct_SHA1sum sha1_obj;
    C_struct_SHA256sum sha256_obj;
    ObjRef path_class_ref;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    ObjRef assoc_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    char name[256];
    char answer[8];


    if (argc != 3 || strcmp(argv[1], "--all-traces") != 0 || *argv[2] == '-')
    {
	fprintf(stderr, "usage: %s --all-traces file\n", argv[0]);
	exit(1);
    }

    printf("\n");

    if (getenv(TEST_ENVIRONMENT_VARIABLE) == NULL)
    {
	printf("WARNING: while this program has been fairly thoroughly tested, it is possible\n");
	printf("that it could damage your archive in some way.  It is also incompatible with\n");
	printf("backup (.dif) files.  You must make a backup of your archive before using this\n");
	printf("program and immediately after you use it!\n\n");

	if (nwos_backup_is_enabled())
	{
	    printf("Since you have backups (.dif files) enabled you will need to disable them\n");
	    printf("before this program can be used.  One way of doing this is to set the\n");
	    printf("OBJECTIFY_BACKUP_PATH environment variable to NULL, for example:\n\n");
	    printf(" $ export OBJECTIFY_BACKUP_PATH=\n\n");
	    printf("I repeat that, at this time, you should avoid using this program!\n\n");

	    exit(1);
	}
	else
	{
	    printf("Have you made a backup of your archive? (enter `yes' to remove %s) ", argv[2]);

	    fflush(stdout);

	    if (fgets(answer, sizeof(answer), stdin) == NULL)
	    {
		fputc('\n', stderr);

		if (feof(stdin))
		{
		    fprintf(stderr, "ERROR: received end of input from standard input when reading response\n");
		    exit(1);
		}
		else
		{
		    perror("reading response from standard input");
		    exit(1);
		}
	    }

	    printf("\n");

	    if (strcmp(answer, "yes\n") != 0)
	    {
		printf("No changes made!\n\n");
		exit(1);
	    }
	}
    }

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);


    assert(nwos_find_private_class_definition("FILE PATH", &path_class_ref));

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));


    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size));

	    /* remember ptr_path_obj points to the kludge buffer */

	    for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

	    name[j] = '\0';

	    if (strcmp(argv[2], name) == 0)
	    {
		copy_reference(&assoc_ref, &ref_list->references[i]);
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    if (i < num_refs)   /* found the file */
    {
	printf("path: %08x\n", nwos_ref_to_word(&assoc_obj.path));
	printf("association: %08x\n", nwos_ref_to_word(&assoc_ref));
	printf("file: %08x\n", nwos_ref_to_word(&assoc_obj.file));
	assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));
	printf("md5: %08x\n",  nwos_ref_to_word(&file_obj.md5sum));
	printf("sha1: %08x\n",  nwos_ref_to_word(&file_obj.sha1sum));
	printf("list: %08x\n", nwos_ref_to_word(&file_obj.block_list));


	/* check path object */

	ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);
	num_refs = ref_list->common_header.num_refs;

	if (num_refs == 1 && is_same_object(&ref_list->references[0], &assoc_ref))
	{
	    printf("Path reference OK\n");
	    nwos_remove_object(&ptr_path_obj->header.object.references);
	    nwos_remove_object(&assoc_obj.path);
	    nwos_remove_from_references(&assoc_obj.path, &ptr_path_obj->header.common.class_definition);
	}
	else
	{
	    printf("Path reference count: %d\n", num_refs);
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;


	/* check file object */

	ref_list = nwos_malloc_reference_list(&file_obj.header.object.references);
	num_refs = ref_list->common_header.num_refs;

	if (num_refs == 1 && is_same_object(&ref_list->references[0], &assoc_ref))
	{
	    printf("File reference OK\n");
	    nwos_remove_object(&file_obj.header.object.references);
	    nwos_remove_object(&assoc_obj.file);
	    nwos_remove_from_references(&assoc_obj.file, &file_obj.header.common.class_definition);
	}
	else
	{
	    printf("File reference count: %d\n", num_refs);
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	/* check MD5 sum object */

	assert(nwos_read_object_from_disk(&file_obj.md5sum, &md5_obj, sizeof(md5_obj)));

	ref_list = nwos_malloc_reference_list(&md5_obj.header.object.references);
	num_refs = ref_list->common_header.num_refs;

	if (num_refs == 1 && is_same_object(&ref_list->references[0], &assoc_obj.file))
	{
	    printf("MD5 reference OK\n");
	    nwos_remove_object(&md5_obj.header.object.references);
	    nwos_remove_object(&file_obj.md5sum);
	    nwos_remove_from_references(&file_obj.md5sum, &md5_obj.header.common.class_definition);
	}
	else
	{
	    printf("MD5 reference count: %d\n", num_refs);
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;


	/* check SHA1 sum object */

	assert(nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_obj, sizeof(sha1_obj)));

	ref_list = nwos_malloc_reference_list(&sha1_obj.header.object.references);
	num_refs = ref_list->common_header.num_refs;

	if (num_refs == 1 && is_same_object(&ref_list->references[0], &assoc_obj.file))
	{
	    printf("SHA1 reference OK\n");
	    nwos_remove_object(&sha1_obj.header.object.references);
	    nwos_remove_object(&file_obj.sha1sum);
	    nwos_remove_from_references(&file_obj.sha1sum, &sha1_obj.header.common.class_definition);
	}
	else
	{
	    printf("SHA1 reference count: %d\n", num_refs);
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;


	/* check SHA256 sum object */

	if (!is_void_reference(&file_obj.sha256sum))
	{
	    assert(nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_obj, sizeof(sha256_obj)));

	    ref_list = nwos_malloc_reference_list(&sha256_obj.header.object.references);
	    num_refs = ref_list->common_header.num_refs;

	    if (num_refs == 1 && is_same_object(&ref_list->references[0], &assoc_obj.file))
	    {
		printf("SHA256 reference OK\n");
		nwos_remove_object(&sha256_obj.header.object.references);
		nwos_remove_object(&file_obj.sha256sum);
		nwos_remove_from_references(&file_obj.sha256sum, &sha256_obj.header.common.class_definition);
	    }
	    else
	    {
		printf("SHA256 reference count: %d\n", num_refs);
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;
	}

	/* do the block list */

	ref_list = nwos_malloc_reference_list(&file_obj.block_list);
	num_refs = ref_list->common_header.num_refs;

	printf("number of blocks: %d\n", num_refs);

	for (i = 0; i < num_refs; i++)
	{
	    nwos_remove_object(&ref_list->references[i]);
	}

	nwos_remove_object(&file_obj.block_list);

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	nwos_remove_object(&assoc_obj.header.object.references);
	nwos_remove_object(&assoc_ref);
	nwos_remove_from_references(&assoc_ref, &assoc_obj.header.common.class_definition);
    }
    else
    {
	printf("No file with the name: '%s' found in system\n", argv[2]);
    }

    nwos_terminate_objectify();

    return 0;
}


