/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-20 08:10:57 -0700 (Wed, 20 Jan 2010) $
--   $Revision: 4465 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "backup.h"
#include "bit_map.h"
#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "gen_id.h"
#include "log.h"
#include "progress_bar.h"
#include "storage.h"
#include "user_config.h"


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s compressed-file\n", program);
}


int main(int argc, char* argv[])
{
    const char* obj_file_path;
    uint8 block[FILE_BLOCK_SIZE];
    size_t bytes_read;
    FILE* ifp;
    uint32 ref;
    uint32 hash;
    uint64 num_blocks;
    uint64 used_private_blocks;
    uint32 used_private_blocks_32_bits;
    uint32 used_private_chunks;
    uint32 chunk_count;
    Disk_Header compressed_header;
    Disk_Header_0022_to_0029 old_header;
#if 0
    Disk_Header_0021 old_header;
#endif
    char answer[8];
    char msg[128];
    bool pre_0028 = false;


    if (argc != 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    obj_file_path = nwos_get_private_objects_path();

    nwos_initialize_storage(READ_WRITE, DEFAULT_FILE);

    nwos_terminate_backup(false, NULL);   /* don't create a backup file */

    /* Verify that is what s/he really wants to do */

    fprintf(stderr, "\n");
    fprintf(stderr, "WARNING: the contents of %s will be OVERWRITTEN!!!\n", obj_file_path);
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you want to continue? (enter `yes' to write %s) ", obj_file_path);

    fflush(stderr);

    if (fgets(answer, sizeof(answer), stdin) == NULL)
    {
	fputc('\n', stderr);

	if (feof(stdin))
	{
	    fprintf(stderr, "ERROR: received end of input from standard input\n");
	    exit(1);
	}
	else
	{
	    perror("reading answer to question");
	    exit(1);
	}
    }

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }


    /* open the compressed archive */

    ifp = fopen(argv[1], "r");

    if (ifp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	printf("Compressed file missing magic number: %02x%02x%02x%02x\n",
	       block[0],
	       block[1],
	       block[2],
	       block[3]);
	fclose(ifp);
	exit(1);
    }


    /* verify that the number of used blocks is consistent with empty disk */

    if (nwos_used_private_blocks != 0 || nwos_used_private_chunks != 0)
    {
	fprintf(stderr, "ERROR: disk not empty, chunks used: %u  blocks used: %llu!.\n", 
		nwos_used_private_chunks, nwos_used_private_blocks);
	exit(1);
    }


    /* Version 0023, 0024 and 0025 archives are identical so allow expanding 0023 or 0024 */

    if (memcmp(VERSION_STRING, &block[4], 4) != 0)
    {
	if (memcmp(VERSION_STRING, &block[4], 4) < 0)
	{
	    printf("ERROR: compressed file is version: %c%c%c%c, which newer than this version of software.\n",
		   block[4], block[5], block[6], block[7]);
	    printf("       Please upgrade your software to the latest version\n");
	    fclose(ifp);
	    exit(1);
	}
	else if (memcmp(OLDEST_COMPATIBLE_COMPRESSED_VERSION, &block[4], 4) <= 0)
	{
	    fprintf(stderr, "WARNING: compressed file is version 00%c%c, is compatible.\n", block[6], block[7]);
	    printf("WARNING: compressed file is version 00%c%c, is compatible.\n", block[6], block[7]);

	    sleep(10);
	}
	else
	{
	    printf("ERROR: compressed file is version: %c%c%c%c, which is incompatible.\n",
		   block[4], block[5], block[6], block[7]);
	    fclose(ifp);
	    exit(1);
	}

	/* if old versions need special handling put it here and remove braces below */ 

	if (memcmp(&block[4], "0028", 4) < 0) pre_0028 = true;   /* need to warn at the end */

	memcpy(&old_header, block, sizeof(old_header));

	nwos_4_uint8_to_uint32(old_header.used_blocks, &used_private_blocks_32_bits);
	nwos_4_uint8_to_uint32(old_header.used_chunks, &used_private_chunks);

	used_private_blocks = used_private_blocks_32_bits;

	if (memcmp(&block[4], "0030", 4) < 0)    /* if 0029 version or less the chunk count may be differen */
	{
	    fprintf(stderr, "WARNING: number of chunks may change between pre-0030 versions and post-0029\n");
	    fprintf(stderr, "         versions.  Scanning compressed file to determine number of chunks\n");
	    fprintf(stderr, "         needed to expand this compressed file.\n");

	    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

	    nwos_start_progress_bar();

	    num_blocks = 0;
	    hash = 0;
	    chunk_count = 0;

	    while (bytes_read == FILE_BLOCK_SIZE)
	    {
		nwos_update_progress_bar((float)num_blocks / (float)used_private_blocks);

		nwos_4_uint8_to_uint32(&block[4], &ref);

		/* Yes this is a hack!  The new chunks are exactly 65536 blocks so we just have to compare */
		/* the upper 16 bits of the reference to deterimine if the blocks are in different chunks. */
		if ((hash & 0xffff0000) != (ref & 0xffff0000))
		{
		    chunk_count++;
		    hash = ref;
		}

		num_blocks++;

		bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);
	    }

	    nwos_finish_progress_bar();

	    if (ferror(ifp))
	    {
		perror(argv[1]);
		exit(1);
	    }

	    if (num_blocks != used_private_blocks)
	    {
		printf("ERROR: blocks in file (%llu) is does not match number of used private blocks in header (%llu)\n", num_blocks, used_private_blocks);
		exit(1);
	    }

	    fprintf(stderr, "chunks in old header: %u  chunks required now: %u\n", used_private_chunks, chunk_count);

	    if (used_private_chunks < chunk_count)
	    {
		used_private_chunks = chunk_count;
	    }

	    if (fseek(ifp, FILE_BLOCK_SIZE, SEEK_SET) != 0)
	    {
		perror(argv[1]);
		exit(1);
	    }
	}
    }
    else
    {
	printf("compressed file version: %c%c%c%c\n", block[4], block[5], block[6], block[7]);
	sleep(5);
#if 0
    }  /* remove these two braces if the the blocks and chunks */
    {  /* need to be read differently for the old file format */ 
#endif
	memcpy(&compressed_header, block, sizeof(compressed_header));

	nwos_8_uint8_to_uint64(compressed_header.used_blocks, &used_private_blocks);
	nwos_4_uint8_to_uint32(compressed_header.used_chunks, &used_private_chunks);

    }

    if (nwos_archive_is_block_device())
    {
	printf("   total_private_blocks: %u  used_private_blocks: %llu\n",
	       nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK, used_private_blocks);

	if (used_private_chunks > nwos_total_private_chunks)
	{
	    fprintf(stderr, "used chunks: %u greater than disk space: %u\n",
		    used_private_chunks, nwos_total_private_chunks);
	    exit(1);
	}
    }
    else if (nwos_total_private_chunks < used_private_chunks)
    {
	printf("Expanding file...\n");

	nwos_start_progress_bar();

	while (nwos_total_private_chunks < used_private_chunks)
	{
	    nwos_update_progress_bar((float)nwos_total_private_chunks / (float)used_private_chunks);

	    nwos_write_empty_chunk(nwos_total_private_chunks);
	} 

	nwos_finish_progress_bar();
    }


    /* finally copy the objects */

    bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);

    printf("Writing blocks...\n");

    nwos_start_progress_bar();

    num_blocks = 0;

    while (bytes_read == FILE_BLOCK_SIZE)
    {
	nwos_update_progress_bar((float)num_blocks / (float)used_private_blocks);

	nwos_4_uint8_to_uint32(&block[4], &ref);

	hash = nwos_hash_uint32_ref(ref);

	if (hash == 0)
	{
	    nwos_allocate_new_chunk(ref);

	    hash = nwos_hash_uint32_ref(ref);

	    assert(hash != 0);
	}

#if 0  
	/*************** save this for a verbose mode? *************/
	printf("id: %08x  block: %08x\n", ref, hash);
	fflush(stdout);
#endif

	if (nwos_test_bit_in_map(hash))
	{
	    fprintf(stderr, "ERROR: block is already used!\n");
	    exit(1);
	}

	nwos_set_bit_in_map(hash);

	nwos_write_block((ObjRef*)&block[4], block);

	num_blocks++;

	assert(num_blocks == nwos_used_private_blocks);

	bytes_read = fread(block, 1, FILE_BLOCK_SIZE, ifp);
    }

    nwos_finish_progress_bar();

    if (ferror(ifp))
    {
	perror(argv[1]);
	exit(1);
    }

    if (fclose(ifp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    nwos_terminate_storage();

    snprintf(msg, sizeof(msg), "Number of blocks written: %llu  chunks_used: %u", num_blocks, nwos_used_private_chunks);
    nwos_log(msg);
    printf("%s\n", msg);

    if (used_private_blocks != nwos_used_private_blocks)
    {
	snprintf(msg, sizeof(msg), "WARNING: used_private_blocks mismatch, disk: %llu  compressed %llu\n",
		nwos_used_private_blocks, used_private_blocks);
	fprintf(stderr, "%s\n", msg);
    }

    if (pre_0028)
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "******************************************************************************\n");
	fprintf(stderr, " WARNING: compressed file is older than version Alpha_28, you need to run the\n");
	fprintf(stderr, " 'update_files_0028' program to move the revision links to the correct place.\n");
	fprintf(stderr, "******************************************************************************\n");
	fprintf(stderr, "\n");
    }

    return 0;
}

