/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-20 08:10:57 -0700 (Wed, 20 Jan 2010) $
--   $Revision: 4465 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file, the compress_sparse.c file and
--   the expand_sparse.c file which it was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "backup.h"
#include "chunk_info.h"
#include "bit_map.h"
#include "config.h"
#include "disk_io.h"
#include "gen_id.h"
#include "header.h"
#include "log.h"
#include "mem_alloc.h"
#include "progress_bar.h"
#include "storage.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "user_config.h"


extern void nwos_set_private_objects_path(const char* path);


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [source-archive] destination-archive\n", program);
}


static bool control_c_received = false;

static void control_c_handler(int signal_num)
{
    signal(SIGINT, SIG_DFL);    /* restore to default control-C handler */

    fputs("^C", stdout);     /* indicate that control-C was received */
    fflush(stdout);

    control_c_received = true;
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    off_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block[FILE_BLOCK_SIZE];
    uint8 old_block[FILE_BLOCK_SIZE];
    int i;
    int j;
    size_t bytes_read;
    int num_blocks;
    uint64 used_private_blocks;
    uint32 chunk_block_offset;
    uint32 blocks_on_disk;
    char answer[8];
    char msg[128];
    Disk_Header disk_header;
    uint32 ref;
    uint32 hash;
    int chunk_num;
    uint32 used_chunks;
    uint32 total_chunks;
    uint32 empty_chunks;
    Chunk_Info* chunk_info;
    int blocks_with_bad_ids = 0;
    char* source;
    char* destination;
    size_t source_len = 0;
    bool full_copy = false;
    uint32 num_chunks_block_size_mismatch;
    uint32 num_chunks_md5_mismatch;


    if (argc != 2 && argc != 3)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-' || (argc == 3 && *argv[2] == '-'))
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }

    if (argc == 2)
      {
	source_len = strlen(nwos_get_private_objects_path()) + 1;

	source = nwos_malloc(source_len);

	strlcpy(source, nwos_get_private_objects_path(), source_len);

	destination = argv[1];
    }
    else
    {
	source = argv[1];
	destination = argv[2];
    }

    if (strcmp(source, destination) == 0)
    {
	fprintf(stderr, "Error: the source and destination cannot be the same!\n");
	print_usage(argv[0]);
	exit(1);
    }


    nwos_log_arguments(argc, argv);

    nwos_set_private_objects_path(destination);

    assert(strcmp(nwos_get_private_objects_path(), destination) == 0);

    nwos_initialize_storage(READ_WRITE, DEFAULT_FILE);

    nwos_terminate_backup(false, NULL);   /* don't create a backup file */

    /* See if the number of used blocks is consistent with empty disk */
    if (nwos_used_private_blocks == 0 && nwos_used_private_chunks == 0)
    {
	full_copy = true;
    }
    else if (nwos_used_private_blocks != 0  && nwos_used_private_chunks != 0)
    {
	full_copy = false;
    }
    else
    {
	fprintf(stderr, "ERROR: unusual condition in archive: %s, chunks used: %u  blocks used: %llu!.\n", 
		destination, nwos_used_private_chunks, nwos_used_private_blocks);
	fprintf(stderr, "Will not continue\n");
	nwos_terminate_storage();
	exit(1);
    }


    /* Open the storage drive and verify the header info */

    obj_file_desc = open(source, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(source);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block, sizeof(block));

    if (bytes_read != sizeof(block))
    {
	perror("reading first block");
	exit(1);
    }

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", source);
	exit(1);
    }

    if (memcmp(&block[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", source);
	exit(1);
    }

    memcpy(&disk_header, block, sizeof(disk_header));

    nwos_4_uint8_to_uint32(disk_header.total_chunks, &total_chunks);
    nwos_8_uint8_to_uint64(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_block_offset);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

    assert(used_chunks > 0);

    chunk_info = malloc(used_chunks * sizeof(Chunk_Info));
    assert(chunk_info != NULL);

    bytes_read = read(obj_file_desc, chunk_info, used_chunks * sizeof(Chunk_Info));

    if (bytes_read != used_chunks * sizeof(Chunk_Info))
    {
	perror("reading chunk info");
	exit(1);
    }

    /* fix the byte order on little endian machines */
#ifndef WORDS_BIGENDIAN
	{
	  int i;
	  for (i = 0; i < used_chunks; i++)
	  {
	      chunk_info[i].ref = byteswap_uint64(chunk_info[i].ref);
	      chunk_info[i].flags_used = byteswap_uint32(chunk_info[i].flags_used);
	      chunk_info[i].index = byteswap_uint32(chunk_info[i].index);
	  }
	}
#endif

    blocks_on_disk = total_chunks * BLOCKS_IN_CHUNK; 


    printf("header: %c%c%c%c %c%c%c%c\n",
	   disk_header.magic_number[0], disk_header.magic_number[1], disk_header.magic_number[2], disk_header.magic_number[3], 
	   disk_header.version_string[0], disk_header.version_string[1], disk_header.version_string[2], disk_header.version_string[3]);

    printf("Source archive (%s)        last modified: %s, used blocks: %llu\n", source, nwos_time_stamp_to_string(disk_header.last_change), used_private_blocks);
    printf("Destination archive (%s) - last modified: %s, used blocks: %llu\n", destination, nwos_time_stamp_to_string(nwos_last_change), nwos_used_private_blocks);


    if (used_private_blocks < nwos_used_private_blocks)
    {
	fprintf(stderr, "ERROR: the source archive (%s) has fewer blocks (%llu) than the destination archive (%s) archive (%llu).\n",
		source, used_private_blocks, destination, nwos_used_private_blocks);
	fprintf(stderr, "This version of software cannot remove blocks so something is wrong.\n");
	nwos_terminate_storage();
	close(obj_file_desc);
	exit(1);
    }


    /* first make sure everything is kosher */

    printf("Checking chunk_info tables...\n");

    j = 0;
    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	while (j < used_chunks && chunk_info[j].ref != nwos_chunk_info[i].ref) j++;
	if (j == used_chunks) break;
	j++;
    }

    if (i < nwos_used_private_chunks)   /* refs didn't all match */
    {
	fprintf(stderr, "ERROR: the destination archive (%s) has chunks that don't exist in the source archive (%s) archive.\n", destination, source);
	fprintf(stderr, "This means either the archives are different or chunks have been added to the destination since it was sync'd.\n");
	nwos_terminate_storage();
	close(obj_file_desc);
	exit(1);
    }

    num_chunks_block_size_mismatch = 0;
    num_chunks_md5_mismatch = 0;

    j = 0;
    empty_chunks = 0;
    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	while (j < used_chunks && chunk_info[j].ref != nwos_chunk_info[i].ref)
	{
	    if ((chunk_info[j].flags_used & CHUNK_INFO_USED_MASK) == 0) empty_chunks++;
	    j++;
	}

	if (j == used_chunks) break;

	if ((chunk_info[j].flags_used & CHUNK_INFO_USED_MASK) != (nwos_chunk_info[i].flags_used & CHUNK_INFO_USED_MASK)) num_chunks_block_size_mismatch++;
	if (memcmp(chunk_info[j].md5_digest, nwos_chunk_info[i].md5_digest, MD5_DIGEST_SIZE) != 0) num_chunks_md5_mismatch++;

	j++;
    }

    if (j < used_chunks && (chunk_info[j].flags_used & CHUNK_INFO_USED_MASK) == 0) empty_chunks++;

    printf("Number of new chunks: %u\n", used_chunks - nwos_used_private_chunks);
    printf("Number of empty chunks: %u\n", empty_chunks);
    printf("Number of chunks that changed: %u\n", num_chunks_md5_mismatch);

    assert(num_chunks_md5_mismatch >= num_chunks_block_size_mismatch);

    if (nwos_used_private_chunks == used_chunks - empty_chunks && num_chunks_md5_mismatch == 0)
    {
	printf("\nArchives are the same, sync not needed!\n\n");
	nwos_terminate_storage();
	close(obj_file_desc);
	exit(0);
    }


    /* Verify that is what s/he really wants to do */

    fprintf(stderr, "\n");
    fprintf(stderr, "WARNING: the contents of %s will be %s!!!\n", nwos_private_path, full_copy ? "OVERWRITTEN" : "UPDATED");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you want to continue? (enter `yes' to write %s) ", nwos_private_path);

    fflush(stderr);

    if (fgets(answer, sizeof(answer), stdin) == NULL)
    {
	fputc('\n', stderr);

	if (feof(stdin))
	{
	    fprintf(stderr, "ERROR: received end of input from standard input\n");
	    exit(1);
	}
	else
	{
	    perror("reading answer to question");
	    exit(1);
	}
    }

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }


    if (nwos_archive_is_block_device())
    {
	printf("   total blocks on disk: %u  used blocks: %llu\n",
	       nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK, used_private_blocks);

	printf("   total chunks on disk: %u  used chunks: %u\n",
	       nwos_total_private_chunks, used_chunks);

	if (used_chunks > nwos_total_private_chunks)
	{
	    fprintf(stderr, "ERROR: used chunks: %u greater than disk space: %u\n",
		    used_chunks, nwos_total_private_chunks);
	    exit(1);
	}
    }
    else if (nwos_total_private_chunks < used_chunks)
    {
	printf("Expanding file...\n");

	nwos_start_progress_bar();

	while (nwos_total_private_chunks < used_chunks)
	{
	    nwos_update_progress_bar((float)nwos_total_private_chunks / (float)used_chunks);

	    nwos_write_empty_chunk(nwos_total_private_chunks);
	} 

	nwos_finish_progress_bar();
    }

    if (full_copy)
    {
	printf("Copying archive...\n");
    }
    else
    {
	printf("Syncing archive...\n");
    }


    num_blocks = 0;

    if (signal(SIGINT, control_c_handler) == SIG_ERR)
    {
	perror("setting SIGINT signal to catch control-C");
    }

    nwos_start_progress_bar();

    // for now we can skip over public blocks because they should always stay the same
    for (chunk_num = 0; chunk_num < used_chunks && !control_c_received; chunk_num++)
    {
	nwos_update_progress_bar((float)chunk_num / (float)used_chunks);

	chunk = chunk_block_offset + chunk_info[chunk_num].index * BLOCKS_IN_CHUNK;

	if (lseek(obj_file_desc, chunk << 8, SEEK_SET) < 0)
	{
	    snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	if (bytes_read != sizeof(block_map))
	{
	    snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	if (full_copy || chunk_num >= nwos_used_private_chunks || chunk_info[chunk_num].ref != nwos_chunk_info[chunk_num].ref || memcmp(chunk_info[chunk_num].md5_digest, nwos_chunk_info[chunk_num].md5_digest, MD5_DIGEST_SIZE) != 0)
	{
	    for (i = 0; i < USABLE_BLOCKS_PER_CHUNK; i++)
	    {
		if ((block_map[i/8] & (0x80 >> (i%8))) != 0)
		{
		    if (lseek(obj_file_desc, (chunk + BIT_MAP_BLOCKS + i) << 8, SEEK_SET) < 0)
		    {
			snprintf(msg, sizeof(msg), "lseek block:%08x", (uint32)(chunk + i));
			perror(msg);
			exit(1);
		    }

		    bytes_read = read(obj_file_desc, block, sizeof(block));

		    if (bytes_read != sizeof(block))
		    {
			snprintf(msg, sizeof(msg), "reading block: %u", (uint32)(chunk + i));
			perror(msg);
			exit(1);
		    }
		
		    ref = (uint32)block[4] << 24 | (uint32)block[5] << 16 | (uint32)block[6] << 8 | (uint32)block[7];

		    if (ref == 0)
		    {
			nwos_erase_progress_bar();

			blocks_with_bad_ids++;

			printf("WARNING: reference in block is zero, not written:\n");

			for (j = 0; j < FILE_BLOCK_SIZE; j += 16)
			{
			    printf("%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x\n",
				   block[j+0],  block[j+1],  block[j+2],  block[j+3],
				   block[j+4],  block[j+5],  block[j+6],  block[j+7],
				   block[j+8],  block[j+9],  block[j+10], block[j+11],
				   block[j+12], block[j+13], block[j+14], block[j+15]);
			}

			nwos_start_progress_bar();
		    }
		    else
		    {
#if 0
			/* save this for verbose mode? */
			printf("id: %08x  block: %08x\n", ref, (uint32)chunk+i);
			/* printf("id: %08x\n", ref); */
			fflush(stdout);
#endif

			nwos_4_uint8_to_uint32(&block[4], &ref);

			hash = nwos_hash_uint32_ref(ref);

			if (hash == 0)
			{
			    nwos_allocate_new_chunk(ref);

			    hash = nwos_hash_uint32_ref(ref);

			    assert(hash != 0);
			}

			if (nwos_test_bit_in_map(hash))
			{
			    if (full_copy)
			    {
				fprintf(stderr, "ERROR: block is already used!\n");
				exit(1);
			    }

			    nwos_read_block((ObjRef*)&block[4], old_block);

			    assert(memcmp(&block[4], &old_block[4], sizeof(ObjRef)) == 0);   /* make sure reference matches */

			    if (memcmp(old_block, block, FILE_BLOCK_SIZE) != 0)
			    {
				nwos_write_block((ObjRef*)&block[4], block);
			    }
			}
			else
			{
			    nwos_set_bit_in_map(hash);

			    nwos_write_block((ObjRef*)&block[4], block);

			    num_blocks++;
			}

			if (full_copy)
			{
			    assert(num_blocks == nwos_used_private_blocks);
			}
		    }
		}
	    }
	}
    }

    if (signal(SIGINT, SIG_DFL) == SIG_ERR)
    {
	perror("setting SIGINT signal to default");
    }

    nwos_finish_progress_bar();

    printf("Number of blocks: %d\n", num_blocks);

    if (blocks_with_bad_ids > 0)
    {
	printf("WARNING: %d blocks had a reference ID of zero and weren't written!\n", blocks_with_bad_ids);
    }

    close(obj_file_desc);

    nwos_terminate_storage();

    snprintf(msg, sizeof(msg), "Number of blocks written: %d  chunks_used: %u",
	     num_blocks, nwos_used_private_chunks);
    nwos_log(msg);
    printf("%s\n", msg);

    if (used_private_blocks != nwos_used_private_blocks)
    {
	snprintf(msg, sizeof(msg), "WARNING: used_private_blocks mismatch, %s: %llu  %s: %llu\n",
		 source, used_private_blocks, destination, nwos_used_private_blocks);
	nwos_log(msg);
	fprintf(stderr, "%s\n", msg);
    }

    if (source_len > 0)
    {
	nwos_free(source);
    }

    /* check to see if the chunk_info tables matched */

    nwos_initialize_storage(READ_ONLY, DEFAULT_FILE);

    num_chunks_block_size_mismatch = 0;
    num_chunks_md5_mismatch = 0;

    if (!control_c_received && nwos_used_private_chunks == used_chunks - empty_chunks)
    {
	printf("Checking chunk_info tables...\n");

	/* first make sure the references all match */

	for (i = 0; i < nwos_used_private_chunks; i++)
	{
	    if (chunk_info[i].ref != nwos_chunk_info[i].ref) break;
	}

	if (i < nwos_used_private_chunks)   /* refs didn't all match */
	{
	    printf("Reference mismatch - index: %d  %s: %llx  %s: %llx\n", i, source, chunk_info[i].ref, destination, nwos_chunk_info[i].ref);
	}
	else
	{
	    for (i = 0; i < nwos_used_private_chunks; i++)
	    {
		if ((chunk_info[i].flags_used & CHUNK_INFO_USED_MASK) != (nwos_chunk_info[i].flags_used & CHUNK_INFO_USED_MASK)) num_chunks_block_size_mismatch++;
		if (memcmp(chunk_info[i].md5_digest, nwos_chunk_info[i].md5_digest, MD5_DIGEST_SIZE) != 0) num_chunks_md5_mismatch++;
	    }

	    printf("Number of chunks with block counts that mismatch:  %u\n", num_chunks_block_size_mismatch);
	    printf("Number of chunks with MD5 checksums that mismatch: %u\n", num_chunks_md5_mismatch);
	}
   }

    nwos_terminate_storage();

    return control_c_received;    /* exit with 0 if normal completion, exit with 1 if control-C was hit */
}



