/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-20 08:10:57 -0700 (Wed, 20 Jan 2010) $
--   $Revision: 4465 $
--
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"    /* define nwos_change_total_chunks() */
#include "header.h"
#include "log.h"
#include "storage.h"
#include "user_config.h"


extern void nwos_set_private_objects_path(const char* path);


/**********************************************************/
/* Determine if path is a regular file or a block device. */
/**********************************************************/

static bool is_device(const char* path)
{
    assert(path[0] == '/');   /* must be a full path specified */

    return path[1] == 'd' && path[2] == 'e' && path[3] == 'v' && path[4] == '/';
}


/***************************************************************/
/* figure out how many blocks are available on a block device. */
/***************************************************************/

static unsigned max_blocks_on_device(const char* path)
{
    int fd;
    off_t lower = 0LL;
    off_t upper = 1099511627776LL;
    off_t middle;
    off_t result;
    unsigned num_sectors;


    fd = open(path, O_RDONLY);

    if (fd < 0)
    {
	perror(path);
	exit(1);
    }

    while ((upper - lower) > 512)
    {
	middle = lower + (upper - lower) / 2;

	result = lseek(fd, middle, SEEK_SET);

	/* printf("%zd: %zd\n", middle, result); */

	if (result < 0)
	{
	    upper = middle;
	}
	else
	{
	    lower = middle;
	}
    }

    close(fd);

    num_sectors = (unsigned) (lower / 512);

    if (((off_t)num_sectors * 512) != lower)
    {
	printf("something is wrong, not a multiple of 512 bytes: %lld\n", (int64)lower);
    }

    return num_sectors * 2;
}


/****************************************************/
/* Print the usage information, is can be different */
/* depending upon if writing to a file or device.   */
/****************************************************/

static void print_usage(char* name)
{
    fprintf(stderr, "usage: %s [--no-private] [archive]\n\n", name);

    fprintf(stderr, "  if no archive is specified it uses the default archive.\n");
    fprintf(stderr, "  --no-private: don't create the initial private blocks (if importing old data)\n\n");
}



/****************************************************************************/
/* Program to resize a disk partition after the partition has been changed. */
/* Now there is a bit map of used blocks as the first block in every chunk  */
/* 8192 * 8 = 65536 blocks * 256 bytes per block = 16777216 bytes in chunk  */
/****************************************************************************/

int main(int argc, char* argv[])
{
    const char* private_path;
    uint32 total_blocks;
    uint32 total_chunks;
    char msg[128];
    bool failed = false;


    printf("\n");

    nwos_log_arguments(argc, argv);


    /**************************/
    /* sort out the arguments */
    /**************************/


    if (argc == 1)   /* no option or archive specified */
    {
    }
    else if (argc == 2)
    {
	if (*argv[1] == '-')
	{
	    fprintf(stderr, "\nThis program has no options!\n\n");
	    print_usage(argv[0]);
	    exit(1);
	}
	else
	{
	    nwos_set_private_objects_path(argv[1]);
	}
    }
    else
    {
	print_usage(argv[0]);
	exit(1);
    }

    private_path = nwos_get_private_objects_path();
       
    if (!is_device(private_path))
    {
	fprintf(stderr, "\nThis program only functions on disk partitions, %s doesn't appear to be a disk partition\n", private_path);
	exit(1);
    }


    /* convert arguments */

    total_blocks = max_blocks_on_device(private_path);   /* zero means unlimited */

    total_chunks = total_blocks / BLOCKS_IN_CHUNK - 1;   /* one chunk is for the system (header and chunk info table) */


    snprintf(msg, sizeof(msg), "new total chunks: %u  private chunks: %u", total_chunks + 1, total_chunks);
    printf("\n%s\n", msg);
    nwos_log(msg);

    nwos_initialize_storage(READ_WRITE, DEFAULT_FILE);

    snprintf(msg, sizeof(msg), "old total chunks: %u  private chunks: %u  used: %u", nwos_total_private_chunks + 1, nwos_total_private_chunks, nwos_used_private_chunks);
    printf("\n%s\n", msg);

    if (total_chunks < nwos_used_private_chunks)
    {
	fprintf(stderr, "ERROR: new partition size is smaller than used chunks, cannot resize\n");
	failed = true;
    }
    else
    {
	nwos_change_total_chunks(total_chunks);

	snprintf(msg, sizeof(msg), "   private blocks: %9u  used: %10llu", total_chunks * USABLE_BLOCKS_PER_CHUNK, nwos_used_private_blocks);
	nwos_log(msg);
	printf("\n%s\n", msg);
    }

    nwos_terminate_storage();

    return failed;
}

