/*
--         This file is part of the New World OS and Objectify projects
--                    Copyright (C) 2004-2009  QRW Software
--              J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                         http://www.qrwsoftware.com
--                         http://nwos.sourceforge.com
--                      http://objectify.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: header.c,v $
-- Revision 1.7  2009/07/13 04:27:14  jsedwards
-- Changed to not test for the file type_code if for versions prior to 0030.
--
-- Revision 1.6  2009/07/11 22:05:35  jsedwards
-- Deleted nwos_total_private_blocks because it was redundant and changed used
-- public and private blocks to 64 bits instead of 32 bits.
--
-- Revision 1.5  2009/07/11 13:29:07  jsedwards
-- Renamed nwos_check_disk_header function to check_disk_header, added three
-- type_code paramters, made it static, and changed it to make sure the
-- type_code matches one of the type codes passed in.
--
-- Revision 1.4  2009/07/02 13:40:58  jsedwards
-- Removed includes of gen_id.h and reference.h because they weren't needed.
-- Moved all code related to logging back to the disk_io.c file and removed
-- the include of the log.h file.
--
-- Revision 1.3  2009/07/02 13:19:28  jsedwards
-- Added new nwos_check_disk_header function to test the magic number and
-- version in the header and changed to call it in the load functions instead
-- of doing the tests inline.
--
-- Revision 1.2  2009/07/02 12:57:04  jsedwards
-- Changed to pass in a buffer instead of a header and copy the header to and
-- from the buffer.
--
-- Revision 1.1  2009/06/30 11:11:47  jsedwards
-- Created with variables and code taken from the disk_io.c file.
--
*/

#include <stdio.h>
#include <string.h>

#include "chunk_info.h"     /* define BLOCKS_IN_CHUNK */
#include "config.h"
#include "header.h"


Disk_Header nwos_public_disk_header;    /* these should only be accessed by storage.c to pass to backup */
Disk_Header nwos_private_disk_header;

uint64 nwos_used_public_blocks;
uint64 nwos_used_private_blocks;

uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_chunks = 0;
uint32 nwos_used_private_chunks;

char   nwos_version_string[5];   /* this is the version of the archive being read, can be older if it is a compressed file. */


static const char* check_disk_header(Disk_Header* disk_header, bool compressed_file, const char* type_code_1, const char* type_code_2, const char* type_code_3)
{
    assert(type_code_1 != NULL);

    if (memcmp(disk_header->magic_number, MAGIC_NUMBER, 4) != 0)
    {
	return "Missing magic number in disk header";
    }

    if (memcmp(disk_header->version_string, VERSION_STRING, 4) != 0)
    {
	// allow more compressed files to be other compatible versions

	if ((compressed_file && memcmp(disk_header->version_string, OLDEST_COMPATIBLE_COMPRESSED_VERSION, 4 < 0)) ||
	    (!compressed_file && memcmp(disk_header->version_string, VERSION_STRING, 4) != 0))
	{
	    return "Incorrect version string in disk header";
	}

	if (compressed_file && memcmp(disk_header->version_string, "0030", 4) < 0)
	{
	    return NULL;
	}
    }

    if (memcmp(disk_header->type_code, type_code_1, 4) != 0)
    {
	if (type_code_2 == NULL || memcmp(disk_header->type_code, type_code_2, 4) != 0)
	{
	    if (type_code_3 == NULL || memcmp(disk_header->type_code, type_code_3, 4) != 0)
            {
		return "Incorrect file type";
	    }
	}
    }

    return NULL;   /* no errors */
}


const char* nwos_check_disk_header(Disk_Header* disk_header, bool compressed_file, const char* type_code)
{
    return check_disk_header(disk_header, compressed_file, type_code, NULL, NULL);
}


const char* nwos_load_public_data(void* buffer, size_t buffer_size)
{
    const char* error_msg;

    assert(buffer_size >= sizeof(nwos_public_disk_header));

    memcpy(&nwos_public_disk_header, buffer, sizeof(nwos_public_disk_header));

    error_msg = nwos_check_disk_header(&nwos_public_disk_header, false, TYPE_CODE_PUBLIC);   /* don't allow older versions */

    if (error_msg == NULL)
    {
	nwos_8_uint8_to_uint64(nwos_public_disk_header.used_blocks, &nwos_used_public_blocks);
    }

    return error_msg;
}


#ifdef PUBLIC_MODE
void nwos_store_public_data(void* buffer, size_t buffer_size)
{
    nwos_used_public_blocks = nwos_next_public_ref;

    nwos_uint64_to_8_uint8(&nwos_used_public_blocks, &nwos_public_disk_header.used_blocks);

    nwos_get_time_stamp(nwos_public_disk_header.last_change);

    assert(buffer_size >= sizeof(nwos_public_disk_header));

    memcpy(buffer, &nwos_public_disk_header, sizeof(nwos_public_disk_header));
}
#endif


const char* nwos_load_private_data(void* buffer, size_t buffer_size, bool compressed_file)
{
    const char* error_msg;

    assert(buffer_size >= sizeof(nwos_private_disk_header));

    memcpy(&nwos_private_disk_header, buffer, sizeof(nwos_private_disk_header));

    error_msg = check_disk_header(&nwos_private_disk_header, compressed_file, TYPE_CODE_DISK, TYPE_CODE_FILE, TYPE_CODE_COMPRESSED);

    if (error_msg == NULL)
    {
	assert(sizeof(nwos_version_string) == 5);
	memcpy(nwos_version_string, nwos_private_disk_header.version_string, 4);
	nwos_version_string[4] = '\0';

	nwos_4_uint8_to_uint32(nwos_private_disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(nwos_private_disk_header.total_chunks, &nwos_total_private_chunks);
	nwos_4_uint8_to_uint32(nwos_private_disk_header.used_chunks, &nwos_used_private_chunks);
	nwos_8_uint8_to_uint64(nwos_private_disk_header.used_blocks, &nwos_used_private_blocks);

	assert(nwos_total_private_chunks > 0);
    }

    return error_msg;
}


void nwos_store_private_data(void* buffer, size_t buffer_size, bool archive_is_file)
{
    uint32 total_private_chunks;

    nwos_uint64_to_8_uint8(&nwos_used_private_blocks, nwos_private_disk_header.used_blocks);
    nwos_uint32_to_4_uint8(&nwos_used_private_chunks, nwos_private_disk_header.used_chunks);

    if (archive_is_file)   /* totals could have changed */
    {
	nwos_uint32_to_4_uint8(&nwos_total_private_chunks, nwos_private_disk_header.total_chunks);
    }
    else                   /* it's a disk, make sure it didn't */
    {
	nwos_4_uint8_to_uint32(nwos_private_disk_header.total_chunks, &total_private_chunks);
	assert(nwos_total_private_chunks == total_private_chunks);
    }

    nwos_get_time_stamp(nwos_private_disk_header.last_change);

    assert(buffer_size >= sizeof(nwos_private_disk_header));

    memcpy(buffer, &nwos_private_disk_header, sizeof(nwos_private_disk_header));
}


