/*
--             This file is part of the New World OS project
--                 Copyright (C) 2008-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: progress_bar.c,v $
-- Revision 1.7  2009/03/13 12:54:27  jsedwards
-- Added include of progress_bar.h file.
--
-- Revision 1.6  2009/01/22 11:36:14  jsedwards
-- Fix year in copyright.  NO code changes.
--
-- Revision 1.5  2009/01/19 14:08:04  jsedwards
-- Changed so update can be called with 100% (1.0f) more than one time and
-- verify it is never greater than 1.0f.  (Changes made on 2009-01-15)
--
-- Revision 1.4  2008/09/16 13:07:39  jsedwards
-- Added new nwos_erase_progress_bar function.
--
-- Revision 1.3  2008/08/06 03:23:47  jsedwards
-- Fix Bug #2038862 - change to not assert if it is at 102 when finish is
-- called.
--
-- Revision 1.2  2008/07/19 11:51:12  jsedwards
-- Fixed asserts so that calling with progress bar at 100 is okay.
--
-- Revision 1.1  2008/07/19 11:24:27  jsedwards
-- Moved progress bar functions from prep_disk.c and renamed with nwos_ prefix.
--
-- Revision 1.50  2008/01/17 15:07:22  jsedwards
-- Moved progress bar code into prep_disk.c from disk_io.c and made into
-- functions.  Added a function pointer parameter to the allocate_all_hack
-- function which is called with progress.
*/

#include <assert.h>
#include <stdio.h>

#include "progress_bar.h"
#include "types.h"            /* define uint */


/***************************************/
/*  Functions to print a progress bar  */
/***************************************/

static uint progress_bar_percent;
static int num_chars;

static void put_char(char c)
{
    fputc(c, stdout);
    fflush(stdout);
    num_chars++;
}


static void put_string(char* s)
{
    while (*s != '\0')
    {
	put_char(*s);
	s++;
    }
}


static void put_decade(uint d)
{
    if (d == 0)
    {
	put_string("0");
    }
    else if (d > 99)
    {
	put_string("100");
    }
    else
    {
	put_char('0' + d / 10);
	put_char('0');
    }
}


void nwos_start_progress_bar()
{
    progress_bar_percent = 0;

    put_string("Percent complete: ");
}


void nwos_update_progress_bar(float percent)  /* 0.0 <= percent < 1.0 */
{
    assert(percent <= 1.0f);
    assert(progress_bar_percent <= 102);

    while (percent >= (float)progress_bar_percent / 100)
    {
	if (progress_bar_percent % 10 == 0)
	{
	    put_decade(progress_bar_percent);
	}
	else
	{
	    put_char('.');
	}
	fflush(stdout);

	progress_bar_percent += 2;
    }
}


void nwos_erase_progress_bar()
{
    int i;

    for (i = 0; i < num_chars; i++) fputc('\b', stdout);
    fflush(stdout);

    num_chars = 0;
    progress_bar_percent = 0;
}


void nwos_finish_progress_bar()
{
    assert(progress_bar_percent <= 102);

    while (progress_bar_percent <= 100)
    {
	if (progress_bar_percent % 10 == 0)
	{
	    put_decade(progress_bar_percent);
	}
	else
	{
	    put_char('.');
	}

	progress_bar_percent += 2;
    }

    fputc('\n', stdout);

    num_chars = 0;
}


