/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2004, 2005, 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-10-06 08:06:23 -0600 (Tue, 06 Oct 2009) $
--   $Revision: 4384 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify_private.h"


char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);

    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_year_object_size(void* year_obj)
{
    assert(((C_struct_Year*)year_obj)->count > 0);
    return sizeof(C_struct_Year) + ((C_struct_Year*)year_obj)->count;
}

static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

static size_t get_disc_list_object_size(void* disc_list_obj)
{
    uint32 count = nwos_decode_variable_sized_count(((C_struct_Disc_List*)disc_list_obj)->count);

    assert(0 < count && count <= MAX_FILES_PER_DISC_LIST);

    return sizeof(C_struct_Disc_List) + count * sizeof(ObjRef);
}

bool nwos_name_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* ptr_name_obj;
    C_struct_Spelling* ptr_spelling_obj;
    int word;
    int i;
    int j;
    size_t name_obj_size;

    /* read the name object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_name_object_size);
    ptr_name_obj = (C_struct_Name*)kludge;
    name_obj_size = sizeof(C_struct_Name) + (ptr_name_obj->count * sizeof(ObjRef));
    /* printf("name_obj_size: %d\n", name_obj_size); */
    assert(name_obj_size > sizeof(C_struct_Name));
    ptr_name_obj = malloc(name_obj_size);
    memcpy(ptr_name_obj, kludge, name_obj_size);

    ptr_spelling_obj = (C_struct_Spelling*)kludge;

    i = 0;
    for (word = 0; word < ptr_name_obj->count; word++)
    {
	assert(!is_void_reference(&ptr_name_obj->spelling[word]));
	nwos_read_variable_sized_object_from_disk(&ptr_name_obj->spelling[word], kludge, sizeof(kludge), &get_spelling_object_size);

	string[i++] = toupper(ptr_spelling_obj->storage[0]);    /* first letter is upper case */

	for (j = 1; j < ptr_spelling_obj->count; j++) 
	{
	    assert(i < size);
	    string[i++] = ptr_spelling_obj->storage[j];
	}

	assert(i < size);

	if (word + 1 == ptr_name_obj->count)   /* it's the last word */
	{
	    string[i++] = '\0';
	}
	else
	{
	    string[i++] = ' ';
	}
    }

    free(ptr_name_obj);

    return true;
}



void fput_char(char c, FILE* fp)
{
    if (fputc(c, fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void fput_comma(FILE* fp)
{
    fput_char(',', fp);
}


void fput_uint8(uint8 n, FILE* fp)
{
    uint8 m;

    if (n > 99) fput_char('0' + (n / 100), fp);

    m = n % 100;

    if (m > 9) fput_char('0' + (m / 10), fp);

    fput_char('0' + (m % 10), fp);
}


void fput_uint32(uint32 n, FILE* fp)
{
    int i;
    uint32 m;
    uint32 decades[9] = { 1000000000, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10 };

    m = n;

    for (i = 0; i < 9; i++) if (n >= decades[i]) break;

    while (i < 9)
    {
	fput_char('0' + (m / decades[i]), fp);
	m = m % decades[i];
	i++;
    }

    fput_char('0' + m, fp);
}


void fput_newline(FILE* fp)
{
    fput_char('\n', fp);
}


void fput_hex4(uint8 hex[4], FILE* fp)
{
    int i;
    char hex_digit[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    if (fputs("0x", fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }

    for (i = 0; i < 4; i++)
    {
	if (fputc(hex_digit[hex[i] >> 4], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}

	if (fputc(hex_digit[hex[i] & 0xF], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}
    }
}


void fput_hex16(uint8 hex[16], FILE* fp)
{
    int i;
    char hex_digit[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    if (fputs("0x", fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }

    for (i = 0; i < 16; i++)
    {
	if (fputc(hex_digit[hex[i] >> 4], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}

	if (fputc(hex_digit[hex[i] & 0xF], fp) == EOF)
	{
	    perror(current_filename);
	    exit(1);
	}
    }
}


void fput_objref(ObjRef* ref, FILE* fp)
{
    fput_hex4(ref->id, fp);
}


void fput_timestamp(TimeStamp ts, FILE* fp)
{
    if (fprintf(fp, "\"%04u-%02u-%02u (%u) %02u:%02u:%02u.%06u\"",
		nwos_extract_year_from_time_stamp(ts),
		nwos_extract_month_from_time_stamp(ts),
		nwos_extract_day_of_month_from_time_stamp(ts),

		nwos_extract_day_of_week_from_time_stamp(ts),

		nwos_extract_hour_from_time_stamp(ts),
		nwos_extract_minute_from_time_stamp(ts),
		nwos_extract_second_from_time_stamp(ts),
		nwos_extract_microsecond_from_time_stamp(ts)) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void export_common_header(CommonHeader* ch, FILE* fp)
{
    assert(ch->flags == 0);

    fput_hex4(ch->header_chksum, fp);

    fput_comma(fp);

    fput_hex4(ch->data_chksum, fp);

    fput_comma(fp);

    fput_timestamp(ch->creation_time, fp);

    fput_comma(fp);

    fput_objref(&ch->class_definition, fp);
}


void export_object_header(ObjectHeader* oh, FILE* fp)
{
    char* save_filename;
    ReferenceList* ref_list_ptr = NULL;
    int num_objects;
    int i;

    /* none of these should have been used yet */

    assert(is_void_reference(&oh->clone_of));
    assert(is_void_reference(&oh->context));
    assert(is_void_reference(&oh->created_by_user));
    assert(is_void_reference(&oh->created_by_app));
    assert(is_void_reference(&oh->prev_version));
    assert(is_void_reference(&oh->next_version));

    fput_objref(&oh->references, fp);

    if (!is_void_reference(&oh->references))
    {
	save_filename = current_filename;

	current_filename = "reference_list.csv";

	ref_list_ptr = nwos_malloc_reference_list(&oh->references);

	num_objects = ref_list_ptr->common_header.num_refs;

	fput_objref(&oh->references, ref_list_fp);

	fput_comma(ref_list_fp);

	export_common_header(&ref_list_ptr->common_header, ref_list_fp);
#if 0
	/* Because I didn't originally put a reference to the class definition  */
	/* class in it's own reference list, this kludge adds it. */
	if (is_same_object(&oh->references, &old_class_definition_ref_list))
	{
	    fput_comma(ref_list_fp);

	    fput_objref(&old_class_definition_class_ref, ref_list_fp);
	}
#endif
	for (i = 0; i < num_objects; i++)
	{
	    fput_comma(ref_list_fp);

	    fput_objref(&ref_list_ptr->references[i], ref_list_fp);
	}

	fput_newline(ref_list_fp);

	nwos_free_reference_list(ref_list_ptr);
	ref_list_ptr = NULL;

	current_filename = save_filename;
    }
}


void export_every_object(ObjRef* ref, EveryObject* eo, FILE* fp)
{
    fput_objref(ref, fp);

    fput_comma(fp);

    export_common_header(&eo->common, fp);

    fput_comma(fp);

    export_object_header(&eo->object, fp);

    fput_comma(fp);
}
    

void export_root(ObjRef* ref, FILE* fp)
{
    C_struct_Root root_obj;

    nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj));

    export_every_object(ref, &root_obj.header, fp);

    fput_objref(&root_obj.class_definition_class, fp);

    fput_newline(fp);
}
    

void export_class_definition(ObjRef* ref, FILE* fp)
{
    C_struct_Class_Definition class_def_obj;

    nwos_read_object_from_disk(ref, &class_def_obj, sizeof(class_def_obj));

    export_every_object(ref, &class_def_obj.header, fp);

    fput_objref(&class_def_obj.name, fp);

    fput_newline(fp);
}
    

void export_spelling(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*) kludge;
    int i;

    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), get_spelling_object_size);

    export_every_object(ref, &spelling_obj_ptr->header, fp);

    fput_char('"', fp);

    for (i = 0; i < spelling_obj_ptr->count; i++)
    {
	fput_char(spelling_obj_ptr->storage[i], fp);
    }

    fput_char('"', fp);

    fput_newline(fp);
}
    

void export_name(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*) kludge;
    int i;

    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), get_name_object_size);

    export_every_object(ref, &name_obj_ptr->header, fp);

    for (i = 0; i < name_obj_ptr->count; i++)
    {
	if (i > 0) fput_comma(fp);

	fput_objref(&name_obj_ptr->spelling[i], fp);
    }

    fput_newline(fp);
}
    

void export_word(ObjRef* ref, FILE* fp)
{
    C_struct_Word word_obj;

    nwos_read_object_from_disk(ref, &word_obj, sizeof(word_obj));

    export_every_object(ref, &word_obj.header, fp);

    fput_objref(&word_obj.thing, fp);

    fput_comma(fp);

    fput_objref(&word_obj.language, fp);

    fput_comma(fp);

    fput_objref(&word_obj.spelling, fp);

    fput_newline(fp);
}
    

void export_language(ObjRef* ref, FILE* fp)
{
    C_struct_Language language_obj;

    nwos_read_object_from_disk(ref, &language_obj, sizeof(language_obj));

    export_every_object(ref, &language_obj.header, fp);

    fput_objref(&language_obj.definition, fp);

    fput_newline(fp);
}
    

void export_us_state(ObjRef* ref, FILE* fp)
{
    C_struct_US_State state_obj;

    nwos_read_object_from_disk(ref, &state_obj, sizeof(state_obj));

    export_every_object(ref, &state_obj.header, fp);

    fput_objref(&state_obj.name, fp);

    fput_comma(fp);

    fput_objref(&state_obj.date, fp);

    fput_comma(fp);

    fput_objref(&state_obj.capital, fp);

    fput_comma(fp);

    fput_objref(&state_obj.postal_code, fp);

    fput_comma(fp);

    fput_uint8(state_obj.number, fp);

    fput_newline(fp);
}
    

void export_us_city(ObjRef* ref, FILE* fp)
{
    C_struct_US_City city_obj;

    nwos_read_object_from_disk(ref, &city_obj, sizeof(city_obj));

    export_every_object(ref, &city_obj.header, fp);

    fput_objref(&city_obj.name, fp);

    fput_comma(fp);

    fput_objref(&city_obj.state, fp);

    assert(is_void_reference(&city_obj.country));   /* never used this? */

    fput_newline(fp);
}
    

void export_area_code(ObjRef* ref, FILE* fp)
{
    C_struct_Area_Code area_code_obj;
    int i;

    nwos_read_object_from_disk(ref, &area_code_obj, sizeof(area_code_obj));

    export_every_object(ref, &area_code_obj.header, fp);

    fput_objref(&area_code_obj.state, fp);

    fput_comma(fp);

    assert(is_void_reference(&area_code_obj.area));   /* never used this? */

    fput_char('"', fp);

    for (i = 0; i < sizeof(area_code_obj.storage); i++) fput_char(area_code_obj.storage[i], fp);

    fput_char('"', fp);

    fput_newline(fp);
}

    
void export_phone_number(ObjRef* ref, FILE* fp)
{
    C_struct_Phone_Number phone_number_obj;
    int i;

    nwos_read_object_from_disk(ref, &phone_number_obj, sizeof(phone_number_obj));

    export_every_object(ref, &phone_number_obj.header, fp);

    assert(is_void_reference(&phone_number_obj.country));   /* never used this? */

    fput_objref(&phone_number_obj.area_code, fp);

    fput_comma(fp);

    fput_char('"', fp);

    for (i = 0; i < sizeof(phone_number_obj.storage); i++) fput_char(phone_number_obj.storage[i], fp);

    fput_char('"', fp);

    fput_newline(fp);
}

    
void export_mobile_phone(ObjRef* ref, FILE* fp)
{
    C_struct_Mobile_Phone mobile_phone_obj;

    nwos_read_object_from_disk(ref, &mobile_phone_obj, sizeof(mobile_phone_obj));

    export_every_object(ref, &mobile_phone_obj.header, fp);

    fput_objref(&mobile_phone_obj.person, fp);

    fput_comma(fp);

    fput_objref(&mobile_phone_obj.number, fp);

    fput_newline(fp);
}

    
void export_year(ObjRef* ref, FILE* fp)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Year* year_obj_ptr = (C_struct_Year*) kludge;
    int i;

    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), get_year_object_size);

    export_every_object(ref, &year_obj_ptr->header, fp);

    fput_char('"', fp);

    for (i = 0; i < year_obj_ptr->count; i++)
    {
	fput_char(year_obj_ptr->storage[i], fp);
    }

    fput_char('"', fp);

    fput_newline(fp);
}
    

void export_month(ObjRef* ref, FILE* fp)
{
    C_struct_Month month_obj;

    nwos_read_object_from_disk(ref, &month_obj, sizeof(month_obj));

    export_every_object(ref, &month_obj.header, fp);

    assert(is_void_reference(&month_obj.definition));   /* never used this? */
    
    fput_char('"', fp);

    fput_char(month_obj.storage[0], fp);

    fput_char(month_obj.storage[1], fp);

    fput_char('"', fp);

    fput_comma(fp);

    fput_uint8(month_obj.minimum_days, fp);

    fput_comma(fp);

    fput_uint8(month_obj.maximum_days, fp);

    fput_newline(fp);
}
    

void export_date(ObjRef* ref, FILE* fp)
{
    C_struct_Date date_obj;

    nwos_read_object_from_disk(ref, &date_obj, sizeof(date_obj));

    export_every_object(ref, &date_obj.header, fp);

    fput_objref(&date_obj.year, fp);

    fput_comma(fp);

    fput_objref(&date_obj.month, fp);

    fput_comma(fp);

    fput_uint8(date_obj.day_of_month, fp);

    fput_newline(fp);
}
    

void export_gender(ObjRef* ref, FILE* fp)
{
    C_struct_Gender gender_obj;

    nwos_read_object_from_disk(ref, &gender_obj, sizeof(gender_obj));

    export_every_object(ref, &gender_obj.header, fp);

    fput_objref(&gender_obj.definition, fp);

    fput_newline(fp);
}
    

void export_person(ObjRef* ref, FILE* fp)
{
    C_struct_Person person_obj;

    nwos_read_object_from_disk(ref, &person_obj, sizeof(person_obj));

    export_every_object(ref, &person_obj.header, fp);

    fput_objref(&person_obj.gender, fp);

    fput_comma(fp);

    assert(is_void_reference(&person_obj.title));   /* never used this? */
    
    fput_objref(&person_obj.first_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.middle_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.last_name, fp);

    fput_comma(fp);

    fput_objref(&person_obj.maiden_name, fp);

    fput_comma(fp);

    assert(is_void_reference(&person_obj.preferred_name));   /* never used this? */
    
    fput_objref(&person_obj.goes_by, fp);

    fput_comma(fp);

    fput_objref(&person_obj.nickname, fp);

    fput_comma(fp);

    fput_objref(&person_obj.birth_date, fp);

    assert(is_void_reference(&person_obj.birth_place));   /* never used this? */

    assert(is_void_reference(&person_obj.death_date));   /* never used this? */

    assert(is_void_reference(&person_obj.death_place));   /* never used this? */
    
    assert(is_void_reference(&person_obj.height));   /* never used this? */

    assert(is_void_reference(&person_obj.mailing_address));   /* never used this? */

    fput_newline(fp);
}


void export_md5sum(ObjRef* ref, FILE* fp)
{
    C_struct_MD5sum md5sum_obj;

    nwos_read_object_from_disk(ref, &md5sum_obj, sizeof(md5sum_obj));

    /* bad thing happened here, md5sum objects were not cleared to zero so we will have to clear them now */

    md5sum_obj.header.object.clone_of.word = 0;
    md5sum_obj.header.object.context.word = 0;
    md5sum_obj.header.object.created_by_user.word = 0;
    md5sum_obj.header.object.created_by_app.word = 0;
    md5sum_obj.header.object.prev_version.word = 0;
    md5sum_obj.header.object.next_version.word = 0;

    export_every_object(ref, &md5sum_obj.header, fp);

    fput_hex16(md5sum_obj.md5sum, fp);

    fput_newline(fp);
}
    

void export_file(ObjRef* ref, FILE* fp)
{
    C_struct_File file_obj;
    uint32 size;

    nwos_read_object_from_disk(ref, &file_obj, sizeof(file_obj));

    export_every_object(ref, &file_obj.header, fp);

    size = (file_obj.size[0] << 24) | (file_obj.size[1] << 16) | (file_obj.size[2] << 8) | (file_obj.size[3]);

    fput_uint32(size, fp);

    fput_comma(fp);

    fput_timestamp(file_obj.modification_time, fp);

    fput_comma(fp);

    fput_objref(&file_obj.md5sum, fp);

    fput_comma(fp);

    assert(is_void_reference(&file_obj.sha1sum));
    assert(is_void_reference(&file_obj.media));

    fput_objref(&file_obj.block_list, fp);

    fput_newline(fp);
}
    

void export_file_path(ObjRef* ref, FILE* fp)
{
    int i;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* file_path_obj_ptr = (C_struct_File_Path*)kludge;

    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);

    export_every_object(ref, &file_path_obj_ptr->header, fp);

    fput_objref(&file_path_obj_ptr->file, fp);

    fput_comma(fp);

    fput_char('"', fp);

    for (i = 0; i < file_path_obj_ptr->count; i++)
    {
	fput_char(file_path_obj_ptr->storage[i], fp);
    }

    fput_char('"', fp);

    fput_newline(fp);
}


void export_disc_list(ObjRef* ref, FILE* fp)
{
    int i;
    int count;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;

    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_disc_list_object_size);

    export_every_object(ref, &ptr_disc_obj->header, fp);

    fput_char('"', fp);

    for (i = 0; i < sizeof(ptr_disc_obj->id); i++)
    {
	fput_char(ptr_disc_obj->id[i], fp);
    }

    fput_char('"', fp);

    assert(is_void_reference(&ptr_disc_obj->name));
    assert(is_void_reference(&ptr_disc_obj->note));

    count = nwos_decode_variable_sized_count(ptr_disc_obj->count);

    for (i = 0; i < count; i++)
    {
	fput_comma(fp);

	fput_objref(&ptr_disc_obj->files[i], fp);
    }

    fput_newline(fp);
}


void export_disc_copy(ObjRef* ref, FILE* fp)
{
    C_struct_Disc_Copy disc_copy_obj;

    nwos_read_object_from_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

    /* made the same mistake with disc copies as with md5sum objects, */
    /* objects were not cleared to zero so we will have to clear them now */

    disc_copy_obj.header.object.clone_of.word = 0;
    disc_copy_obj.header.object.context.word = 0;
    disc_copy_obj.header.object.created_by_user.word = 0;
    disc_copy_obj.header.object.created_by_app.word = 0;
    disc_copy_obj.header.object.prev_version.word = 0;
    disc_copy_obj.header.object.next_version.word = 0;

    export_every_object(ref, &disc_copy_obj.header, fp);

    fput_objref(&disc_copy_obj.disc_list, fp);

    fput_comma(fp);

    /* nasty! because the object wasn't cleared we can't depend upon these being null either */
    /*assert(is_void_reference(&disc_copy_obj.configuration)); */
    /*assert(is_void_reference(&disc_copy_obj.date)); */
    /*assert(is_void_reference(&disc_copy_obj.media)); */

    fput_objref(&disc_copy_obj.location, fp);

    fput_comma(fp);

    fput_uint8(disc_copy_obj.copy_number, fp);

    fput_newline(fp);
}


void export_storage_location(ObjRef* ref, FILE* fp)
{
    C_struct_Storage_Location loc_obj;

    nwos_read_object_from_disk(ref, &loc_obj, sizeof(loc_obj));

    /* made the same mistake with disc copies as with md5sum objects, */
    /* objects were not cleared to zero so we will have to clear them now */

    loc_obj.header.object.clone_of.word = 0;
    loc_obj.header.object.context.word = 0;
    loc_obj.header.object.created_by_user.word = 0;
    loc_obj.header.object.created_by_app.word = 0;
    loc_obj.header.object.prev_version.word = 0;
    loc_obj.header.object.next_version.word = 0;

    export_every_object(ref, &loc_obj.header, fp);

    fput_objref(&loc_obj.name, fp);

    fput_newline(fp);
}
    

void export_class(ObjRef* class_def_ref)
{
    C_struct_Class_Definition class_def_class_obj;
    size_t ref_list_size;
    ReferenceList* ref_list_ptr = NULL;
    char name[32];
    char filename[36];
    int i;
    int num_objects;
    FILE* fp = NULL;

    nwos_read_object_from_disk(class_def_ref, &class_def_class_obj, sizeof(class_def_class_obj));

    nwos_name_to_string(&class_def_class_obj.name, name, sizeof(name));

    printf("class %s: %02x%02x%02x%02x\n",
	   name,
	   class_def_ref->id[0],
	   class_def_ref->id[1],
	   class_def_ref->id[2],
	   class_def_ref->id[3]);

    if (is_void_reference(&class_def_class_obj.header.object.references))
    {
	assert(strcmp(name, "Reference List") == 0);

	num_objects = 0;

	printf("  the reference list class doesn't have a reference list\n");
    }
    else
    {
	ref_list_size = nwos_reference_list_size(&class_def_class_obj.header.object.references);

	ref_list_ptr = malloc(ref_list_size);

	if (ref_list_ptr == NULL) 
	  {
	    perror("reading reference list");
	    exit(1);
	  }

	nwos_read_reference_list_from_disk(&class_def_class_obj.header.object.references, ref_list_ptr, ref_list_size);

	num_objects = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	printf("  num_objects: %d\n", num_objects);
    }

    if (num_objects > 0)     /* create the file */
    {
	for (i = 0; name[i] != '\0'; i++)
	{
	    if (name[i] == ' ')
	    {
		filename[i] = '_';
	    }
	    else
	    {
		filename[i] = tolower(name[i]);
	    }
	}

	filename[i++] = '.';
	filename[i++] = 'c';
	filename[i++] = 's';
	filename[i++] = 'v';
	filename[i] = '\0';

	fp = fopen(filename, "w");

	if (fp == NULL)
	{
	    perror(filename);
	    exit(1);
	}

	current_filename = filename;

	if (strcmp(name, "Class Definition") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_class_definition(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Root") == 0)
	{
	  /* for (i = 0; i < num_objects; i++) export_root(&ref_list_ptr->references[i], fp); */
	}
	else if (strcmp(name, "Spelling") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_spelling(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Name") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_name(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Word") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_word(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Language") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_language(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Us State") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_us_state(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Us City") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_us_city(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Area Code") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_area_code(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Phone Number") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_phone_number(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Mobile Phone") == 0)
	{
	    //	for (i = 0; i < num_objects; i++) export_mobile_phone(&ref_list_ptr->references[i], fp);
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		nwos_get_object_class(&ref_list_ptr->references[i], &class);
		
		if (is_same_object(class_def_ref, &class))
		{
		    export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Home Phone") == 0)
	{
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		nwos_get_object_class(&ref_list_ptr->references[i], &class);

		if (is_same_object(class_def_ref, &class))
		{
		    //		export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Work Phone") == 0)
	{
	    for (i = 0; i < num_objects; i++)
	    {
		ObjRef class;

		nwos_get_object_class(&ref_list_ptr->references[i], &class);

		if (is_same_object(class_def_ref, &class))
		{
		    //		export_mobile_phone(&ref_list_ptr->references[i], fp);
		}
		else
		{
		    printf("  WARNING non-class object reference to class definition: ");
		    printf("%02x%02x%02x%02x: %02x%02x%02x%02x\n",
			   ref_list_ptr->references[i].id[0],
			   ref_list_ptr->references[i].id[1],
			   ref_list_ptr->references[i].id[2],
			   ref_list_ptr->references[i].id[3],
			   class.id[0],
			   class.id[1],
			   class.id[2],
			   class.id[3]);
		}
	    }
	}
	else if (strcmp(name, "Year") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_year(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Month") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_month(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Gender") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_gender(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Date") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_date(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Person") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_person(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Md5sum") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_md5sum(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "File") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_file(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "File Path") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_file_path(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Disc List") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_disc_list(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Disc Copy") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_disc_copy(&ref_list_ptr->references[i], fp);
	}
	else if (strcmp(name, "Storage Location") == 0)
	{
	    for (i = 0; i < num_objects; i++) export_storage_location(&ref_list_ptr->references[i], fp);
	}
	else
	{
	    printf("  BUG - unprocessed class: %s\n", name);
	}
    }

    if (fp != NULL)
    {
	if (fclose(fp) != 0)
	{
	    perror(filename);
	    exit(1);
	}

	current_filename = NULL;
    }

    if (ref_list_ptr != NULL)
    {
	free(ref_list_ptr);
	ref_list_ptr = NULL;
    }
}


#if 0
void export_class(ObjRef* class_ref)
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    C_struct_Class_Definition class_def_obj;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef public_class_ref;

    printf("Reading class definition: %08x\n", nwos_ref_to_word(class_ref));

    nwos_read_object_from_disk(class_ref, &class_def_obj, sizeof(class_def_obj));

    nwos_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    if (!is_same_object(&class_def_obj.header.common.class_definition, &nwos_class_definition_class_ref))
    {
	printf("Class is wrong: %08x  should be: %08x\n",
	       nwos_ref_to_word(&class_def_obj.header.common.class_definition),
	       nwos_ref_to_word(&nwos_class_definition_class_ref));
    }
    else if (strcasecmp(buffer, "REFERENCE LIST") != 0)    /* reference lists don't have reference lists */
    {
	printf("Reading reference list: %08x\n", nwos_ref_to_word(&class_def_obj.header.object.references));
	fflush(stdout);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	printf("Number of refrences: %d\n", num_refs);

	if (nwos_find_class_definition(buffer, &public_class_ref))
	{
	    printf("Found public class\n");
	}
	else
	{
	    printf("Did NOT find public class\n");
	}

#if 0
	if (strcasecmp(buffer, "SPELLING") == 0)
	{
		check_spelling(&ref_list->references[i]);
	}
	else if (strcasecmp(buffer, "NAME") == 0)
	{
		check_name(&ref_list->references[i]);
	}
	else if (strcasecmp(buffer, "MONTH") == 0)
	{
		check_month(&ref_list->references[i]);
	}
#endif
	/* else */ if (strcasecmp(buffer, "FILE PATH") == 0)
	{
	    for (i = 0; i < num_refs; i++)
	    {
		export_file_path(&ref_list->references[i]);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
}
#endif


int main(int argc, char* argv[])
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    C_struct_Class_Definition class_def_obj;
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ReferenceList* ref_list;
    int num_refs;

    if (argc != 2)
    {
	fprintf(stderr, "usage: %s compressed-file\n", argv[0]);
	exit(1);
    }

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];


    nwos_initialize_objectify(bf_key, linear, serial, Compressed_File_RO, argv[1]);

    printf("Setting root object: %08x\n", nwos_ref_to_word(&root_object_ref));

    nwos_set_root_object(&root_object_ref);


    ref_list_fp = fopen("reference_list.csv", "w");

    if (ref_list_fp == NULL)
    {
	perror("reference_list.csv");
	exit(1);
    }


    printf("Reading class definition class: %08x\n", nwos_ref_to_word(&nwos_class_definition_class_ref));

    nwos_read_object_from_disk(&nwos_class_definition_class_ref, &class_def_obj, sizeof(class_def_obj));

    nwos_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    printf("Reading class definition class reference list: %08x\n", nwos_ref_to_word(&class_def_obj.header.object.references));

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("Number of refrences: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	export_class(&ref_list->references[i]);
    }

    if (fclose(ref_list_fp) != 0)
    {
	perror("reference_list.csv");
	exit(1);
    }

    nwos_free_reference_list(ref_list);

    nwos_terminate_objectify();

    return 0;
}

