/*
--          This file is part of the New World OS and Objectify projects
--                     Copyright (C) 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-10-06 08:06:23 -0600 (Tue, 06 Oct 2009) $
--   $Revision: 4384 $
--
--
--  Program to fix problem with two files created when file names were NOT
--  case senstive.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "crc32.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

static bool checkit(char* file_name)
{
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_ref;
    ObjRef assoc_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    printf("Checking name: %s\n", file_name);

    if (nwos_find_file_path(file_name, &path_ref))
    {
	printf("Found: %s  ref: %08x\n", file_name, nwos_ref_to_word(&path_ref));

	if (!nwos_read_variable_sized_object_from_disk(&path_ref, kludge, sizeof(kludge), &get_path_object_size))
	{
	    fprintf(stderr, "Reading path object failed\n");
	    return false;
	}

	ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);
	num_refs = ref_list->common_header.num_refs;
	printf("num_refs: %d\n", num_refs);
	for (i = 0; i < num_refs; i++) printf("  %d: %08x\n", i, nwos_ref_to_word(&ref_list->references[i]));

	if (num_refs != 1)
	{
	    fprintf(stderr, "Error: should only be one ref in list!\n");
	    return false;
	}

	copy_reference(&assoc_ref, &ref_list->references[0]);
	nwos_free_reference_list(ref_list);

	nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj));

	printf("Association object: %08x  path: %08x  file: %08x\n", nwos_ref_to_word(&assoc_obj.header.common.id), nwos_ref_to_word(&assoc_obj.path), nwos_ref_to_word(&assoc_obj.file));
	if (!is_same_object(&assoc_obj.path, &path_ref))
	{
	    fprintf(stderr, "Association object path: %08x doesn't match path: %08x\n", nwos_ref_to_word(&assoc_obj.path), nwos_ref_to_word(&path_ref));
	    return false;
	}
    }
    else    /* didn't find the name */
    {
	fprintf(stderr, "ERROR: The name %s is not in the system\n", file_name);
	return false;
    }

    return true;
}


static bool doit(char* upper_case_name)
{
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge_upper[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_upper_obj = (C_struct_File_Path*)kludge_upper;
    ObjRef upper_case_ref;
    ObjRef lower_case_ref;
    ObjRef assoc_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char lower_case_name[64];


    if (nwos_find_file_path(upper_case_name, &upper_case_ref))
    {
	printf("Found: %s  ref: %08x\n", upper_case_name, nwos_ref_to_word(&upper_case_ref));

	if (!nwos_read_variable_sized_object_from_disk(&upper_case_ref, kludge_upper, sizeof(kludge_upper), &get_path_object_size))
	{
	    fprintf(stderr, "Reading upper case path object failed\n");
	    return false;
	}

	ref_list = nwos_malloc_reference_list(&ptr_upper_obj->header.object.references);
	num_refs = ref_list->common_header.num_refs;
	printf("num_refs: %d\n", num_refs);
	for (i = 0; i < num_refs; i++) printf("  %d: %08x\n", i, nwos_ref_to_word(&ref_list->references[i]));
	copy_reference(&assoc_ref, &ref_list->references[1]);
	nwos_free_reference_list(ref_list);

	nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj));

	printf("Association object: %08x  path: %08x  file: %08x\n", nwos_ref_to_word(&assoc_obj.header.common.id), nwos_ref_to_word(&assoc_obj.path), nwos_ref_to_word(&assoc_obj.file));

	if (!is_same_object(&assoc_obj.path, &upper_case_ref))
	{
	    fprintf(stderr, "Association object path: %08x doesn't match upper case path: %08x\n", nwos_ref_to_word(&assoc_obj.path), nwos_ref_to_word(&upper_case_ref));
	    return false;
	}

	for (i = 0; upper_case_name[i] != '\0'; i++) lower_case_name[i] = tolower(upper_case_name[i]);
	lower_case_name[i] = '\0';

	if (nwos_create_file_path(lower_case_name, &lower_case_ref) != CREATED_NEW)
	{
	    fprintf(stderr, "Lower case name already existed: %08x\n", nwos_ref_to_word(&lower_case_ref));
	    return false;
	}

	printf("New lower case ref: %08x\n", nwos_ref_to_word(&lower_case_ref));

	nwos_add_to_references(&assoc_ref, &lower_case_ref);
	nwos_remove_from_references(&assoc_ref, &upper_case_ref);

	copy_reference(&assoc_obj.path, &lower_case_ref);
	nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);
	nwos_overwrite_object_to_disk(&assoc_obj.header.common.id, &assoc_obj, sizeof(assoc_obj));
	printf("Association object: %08x  path: %08x  file: %08x\n", nwos_ref_to_word(&assoc_obj.header.common.id), nwos_ref_to_word(&assoc_obj.path), nwos_ref_to_word(&assoc_obj.file));

	checkit(upper_case_name);
	checkit(lower_case_name);
    }
    else    /* didn't find the name */
    {
	fprintf(stderr, "ERROR: The name %s is not in the system\n", upper_case_name);
	return false;
    }

    return true;
}


int main(int argc, char* argv[])
{
    char* path = DEFAULT_FILE;

    if (argc != 1)
    {
	fprintf(stderr, "usage: %s\n", argv[0]);
	fprintf(stderr, "  This repairs the two img files that ended up with upper case names.\n");
	exit(1);
    }

    //    nwos_log_arguments(argc, argv);
    nwos_log_arguments(0, NULL);

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, path);
    //nwos_initialize_objectify(READ_ONLY, path);

    nwos_check_blocks_available(4);

    doit("IMG_3213.jpg");
    doit("IMG_3247.jpg");

    nwos_terminate_objectify();

    return 0;
}


