/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-11-06 18:56:08 -0700 (Sun, 06 Nov 2011) $
--   $Revision: 4934 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the list_files.c file which it
--   was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <fnmatch.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "strlcxx.h"        /* in case strlcpy and strlcat are not provided by the system */
#include "time_stamp.h"

static struct section {
  char* anchor;
  char* section;
  char* prefix;
  char* homepage;
} sections[] = 
  {
    { "blag",         "Blag",              "BLAG",         "http://www.blagblagblag.org/" },
    { "centos",       "CentOS",            "CentOS",       "http://www.centos.org/" },
    { "dragora",      "Dragora",           "dragora",      "http://dragora.usla.org.ar/wiki/:en:wiki" },
    { "dynebolic",    "Dyne:bolic",        "dynebolic",    "http://dynebolic.org/" },
    { "edubuntu",     "Edubuntu",          "edubuntu",     "http://edubuntu.org/" },
    { "fedora",       "Fedora",            "yarrow",       "http://fedoraproject.org/" },
    { "gnewsense",    "gNewSense",         "gnewsense",    "http://www.gnewsense.org/Main/HomePage" },
    { "gobuntu",      "Gobuntu",           "gobuntu",      "https://wiki.ubuntu.com/Gobuntu" },
    { "jeos",         "Jeos (Ubuntu)",     "jeos",         "https://help.ubuntu.com/community/JeOS" },
    { "knoppix",      "Knoppix",           "KNOPPIX",      "http://www.knoppix.net/" },
    { "kubuntu",      "Kubuntu",           "kubuntu",      "http://www.kubuntu.org/" },
    { "mythbuntu",    "Mythbuntu",         "mythbuntu",    "http://www.mythbuntu.org/" },
    { "objectify",    "Objectify",         "objectify",    "https://savannah.nongnu.org/projects/objectify" },
    { "trisquel",     "Trisquel",          "trisquel",     "http://trisquel.info/en" },
    { "ubuntu",       "Ubuntu",            "warty",        "http://www.ubuntu.com/" },
    { "ubuntustudio", "UbuntuStudio",      "ubuntustudio", "http://ubuntustudio.org/" },
    { "venenux",      "Venenux",           "venenux",      "http://venenux.org/" },
    { "xubuntu",      "Xubuntu",           "xubuntu",      "http://xubuntu.org/" },
    { "yellowdog",    "YellowDog",         "yellowdog",    "http://www.yellowdoglinux.com/" },
  };

static const int num_sections = sizeof(sections) / sizeof(struct section);

static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


static bool has_prefix(char* string, char* prefix)
{
    while (*prefix != '\0')
    {
	if (*string == '\0' || *string != *prefix) return false;
	string++;
	prefix++;
    }

    return true;
}


void print_unknown(int field_width)
{
    int i;
    int spaces_before;
    int spaces_after;

    spaces_before = (field_width - 9) / 2;
    spaces_after = field_width - 9 - spaces_before;

    for (i = 0; i < spaces_before; i++) printf("&nbsp;");

    printf("&lang;Unknown&rang;");

    for (i = 0; i < spaces_after; i++) printf("&nbsp;");
}


void print_md5(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_MD5sum md5_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */
    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
}


void print_sha1(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA1sum sha1_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha1sum))
    {
             /*          1         2         3         4 */
             /* 1234567890123456789012345678901234567890 */
	print_unknown(40);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object));

	for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
    }
}


void print_sha256(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA256sum sha256_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha256sum))
    {
             /*          1         2         3         4         5         6     */
             /* 1234567890123456789012345678901234567890123456789012345678901234 */
	print_unknown(64);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_object, sizeof(sha256_object));

	for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_object.sha256sum[j]);
    }
}


void print_sha512(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA512sum sha512_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha512sum))
    {
	print_unknown(128);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha512sum, &sha512_object, sizeof(sha512_object));

	for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_object.sha512sum[j]);
    }
}


void print_time(TimeStamp ts)
{
    if (nwos_time_stamp_is_zero(ts))
    {
	print_unknown(19);
    }
    else
    {
	printf("%u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));
    }
}


typedef enum { Option_Error = -1, No_Option, Option_Time, Option_MD5, Option_SHA1, Option_SHA256, Option_SHA512 } Option;


void get_path_name(ObjRef* path_ref, char* name, size_t name_size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    int j;

    nwos_read_variable_sized_object_from_disk(path_ref, kludge, sizeof(kludge), &get_path_object_size);

    /* remember ptr_path_obj points to the kludge buffer */

    assert(ptr_path_obj->count < name_size);

    for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

    name[j] = '\0';
}


void get_assoc_path(ObjRef* assoc_ref, char* name, size_t name_size)
{
    C_struct_Path_And_File_Association assoc_obj;

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));

    get_path_name(&assoc_obj.path, name, name_size);
}


void insert_character(char* s, char c, int index, size_t size)
{
    int length;
    int i;

    length = strlen(s);

    assert(length + 2 <= size);
    assert(index < length);

    for (i = length; i >= index; i--) s[i+1] = s[i];

    s[index] = c;
}


void reverse_characters(char* string, int first, int pivot, int last)
{
    char save[64];
    int i;
    int j;

    assert(pivot - first + 1 < sizeof(save));

    for (i = first; i < pivot; i++)
    {
	save[i-first] = string[i];
    }

    j = first;

    while (i <= last)
    {
	string[j] = string[i];
	i++;
	j++;
    }

    i = 0;
    while (j <= last)
    {
	string[j] = save[i];
	i++;
	j++;
    }
}


/* This function gets the file name, but morphs a bit so the sort order is better */
/* For example fixes it so that ubuntu-10.04 follows ubuntu-9.10                  */

void get_assoc_sort_path(ObjRef* assoc_ref, char* name, size_t name_size)
{
    int i;

    get_assoc_path(assoc_ref, name, name_size);

    /* take care of some special cases */

    if (strcmp(name, "kubuntu-kde4.0-i386.iso") == 0)
    {
	strlcpy(name, "kubuntu-kde4-7.10-i386.iso", name_size);
    }

    if (strcmp(name, "ubuntu-moblin-remix-9.10-moblin-remix-i386.iso") == 0)
    {
	strlcpy(name, "ubuntu-9.10-moblin-remix-i386.iso", name_size);
    }

    if (strcmp(name, "liveBLAG-70000.iso") == 0)
    {
	strlcpy(name, "BLAG-70000-live.iso", name_size);
    }

    if (has_prefix(name, "warty-release"))
    {
	name[0] = 'u';
	name[1] = 'b';
	name[2] = 'u';
	name[3] = 'n';
	name[4] = 't';
	name[5] = 'u';
	name[6] = '-';
	name[7] = '4';
	name[8] = '.';
	name[9] = '1';
	name[10] = '0';
	name[11] = '-';
	name[12] = 'w';
    }

    if (has_prefix(name, "ubuntu-server-5.10"))
    {
	reverse_characters(name, 7, 14, 18);
    }

    /* fix to ignore the case of CentOS */

    if (has_prefix(name, "CentOS"))
    {
	name[0] = 'c';
	name[4] = 'o';
	name[5] = 's';
    }

    /* Fix up Dyne:bolic releases */

    if (has_prefix(name, "dyne-"))
    {
	insert_character(name, 'b', 4, name_size);
	insert_character(name, 'o', 5, name_size);
	insert_character(name, 'l', 6, name_size);
	insert_character(name, 'i', 7, name_size);
	insert_character(name, 'c', 8, name_size);
    }

    if (has_prefix(name, "dynebolic") && name[14] == 'i')
    {
	insert_character(name, '0', 14, name_size);
	insert_character(name, '.', 15, name_size);
    }


    /* Fix up early Fedora releases */

    if (has_prefix(name, "yarrow"))
    {
	name[0] = 'f';
	name[1] = 'e';
	name[2] = 'd';
	name[3] = 'o';
	name[4] = 'r';
	name[5] = 'a';

	insert_character(name, '-', 6, name_size);
	insert_character(name, '0', 7, name_size);
	insert_character(name, '1', 8, name_size);
    }

    if (has_prefix(name, "Fedora-"))
    {
	name[0] = 'f';
	insert_character(name, '0', 7, name_size);
    }

    if (has_prefix(name, "FC-"))
    {
	name[0] = 'f';
	name[1] = 'e';

	insert_character(name, 'd', 2, name_size);
	insert_character(name, 'o', 3, name_size);
	insert_character(name, 'r', 4, name_size);
	insert_character(name, 'a', 5, name_size);
	insert_character(name, '0', 7, name_size);
    }

    if (has_prefix(name, "FC"))
    {
	name[0] = 'f';
	name[1] = 'e';

	insert_character(name, 'd', 2, name_size);
	insert_character(name, 'o', 3, name_size);
	insert_character(name, 'r', 4, name_size);
	insert_character(name, 'a', 5, name_size);
	insert_character(name, '-', 6, name_size);
	insert_character(name, '0', 7, name_size);
    }

    if (has_prefix(name, "F-"))
    {
	name[0] = 'f';

	insert_character(name, 'e', 1, name_size);
	insert_character(name, 'd', 2, name_size);
	insert_character(name, 'o', 3, name_size);
	insert_character(name, 'r', 4, name_size);
	insert_character(name, 'a', 5, name_size);
	insert_character(name, '0', 7, name_size);
    }

    /* Change KNOPPIX to knoppix */

    if (has_prefix(name, "KNOPPIX"))
    {
	name[0] = 'k';
	name[1] = 'n';
	name[2] = 'o';
	name[3] = 'p';
	name[4] = 'p';
	name[5] = 'i';
	name[6] = 'x';
    }

    /* fix to sort special trisquel versions with the regular versions */

    if (has_prefix(name, "trisquel-edu_2.2.2_") || has_prefix(name, "trisquel-pro_2.2.2_"))
    {
	reverse_characters(name, 8, 12, 17);
    }

    if (has_prefix(name, "trisquel-mini_4.0_"))
    {
	reverse_characters(name, 8, 13, 16);
    }

    if (has_prefix(name, "trisquel-netinst_4.0_"))
    {
	reverse_characters(name, 8, 16, 19);
    }

    if (has_prefix(name, "trisquel"))
    {
	if (strstr(name, "2.2.2") != NULL || strstr(name, "3.0.1") != NULL)
	{
	    name[14] ^= ('_' ^ '-');     /* flip the separator so the sort is nicer */
	}
	else if (strstr(name, "3.5") != NULL || strstr(name, "4.0") != NULL)
	{
	    name[12] ^= ('_' ^ '-');     /* flip the separator so the sort is nicer */
	}
    }


    /* insert a zero in front of all single digit numbers */

    if (strlen(name) > 2)       /* the code below assumes the string is at least 3 characters long */
    {
	for (i = 1; i < name_size; i++)      /* start at one because there aren't any distros that have a digit in the first character */
	{
	    if (name[i+1] == '\0') break;

	    if (ispunct(name[i-1]) && isdigit(name[i]) && ispunct(name[i+1]))
	    {
		insert_character(name, '0', i, name_size);
	    }
	}
    }
}


void list_files(Option option)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    ObjRef prev_ref;
    ObjRef next_ref;
    ObjRef save;
    ObjRef* sorted_assoc_refs;
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int i;
    int j;
    int k;
    int section;
    int revision;
    char left_name[PATH_MAX];
    char right_name[PATH_MAX];
    bool do_revision = false;    /* currently there are no revisions in the public files */


    nwos_log_arguments(0, NULL);   /* disable logging */

    nwos_initialize_objectify(PUBLIC, NULL);

    if (!nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	printf("ERROR: could not find public 'PATH AND FILE ASSOCIATION CLASS'.");
	nwos_terminate_objectify();
	return;
    }

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    sorted_assoc_refs = nwos_malloc(sizeof(ObjRef) * num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    get_assoc_sort_path(&ref_list->references[i], left_name, sizeof(left_name));

	    j = i + 1;

	    while (j / 2 > 0)
	    {
		get_assoc_sort_path(&sorted_assoc_refs[(j / 2) - 1], right_name, sizeof(right_name));

		if (strcmp(left_name, right_name) < 0) break;

		/* parent is smaller, move it lower in the tree */
		copy_reference(&sorted_assoc_refs[j - 1], &sorted_assoc_refs[(j / 2) - 1]);
		j = j / 2;
	    }

	    copy_reference(&sorted_assoc_refs[j - 1], &ref_list->references[i]);

	    num_files++;
	}
    }

    for (i = num_files; i > 1; i--)
    {
	/* save the index at the top of the tree (smallest) */
	copy_reference(&save, &sorted_assoc_refs[i - 1]);

	/* move the root of the the (largest) to the top */
	copy_reference(&sorted_assoc_refs[i - 1], &sorted_assoc_refs[0]);

	j = 1;
	while (j * 2 < i)
	{
	    k = j * 2;

	    /* if there is a right child */
	    if (k < i - 1)
	    {
		get_assoc_sort_path(&sorted_assoc_refs[k - 1], left_name, sizeof(left_name));
		get_assoc_sort_path(&sorted_assoc_refs[k + 1 - 1], right_name, sizeof(right_name));

		if (strcmp(left_name, right_name) < 0)
		{
		    k++;
		}
	    }

	    get_assoc_sort_path(&save, left_name, sizeof(left_name));
	    get_assoc_sort_path(&sorted_assoc_refs[k-1], right_name, sizeof(right_name));

	    if (strcmp(left_name, right_name) >= 0) break;

	    copy_reference(&sorted_assoc_refs[j - 1], &sorted_assoc_refs[k - 1]);

	    j = k;
	}

	copy_reference(&sorted_assoc_refs[j - 1], &save);
    }


    //    printf("Number of files: %d\n", num_files);

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    section = 0;

    for (i = 0; i < num_files; i++)
    {
	get_assoc_path(&sorted_assoc_refs[i], left_name, sizeof(left_name));

	if (section < num_sections && has_prefix(left_name, sections[section].prefix))
	{
	    printf("  <H2><A NAME=\"%s\">%s</A></H2>\n", sections[section].anchor, sections[section].section);

	    printf("    Home Page: <A HREF=\"%s\">%s</A></BR></BR>\n", sections[section].homepage, sections[section].homepage);

	    section++;
	}

	nwos_read_object_from_disk(&sorted_assoc_refs[i], &assoc_obj, sizeof(assoc_obj));

	printf("    <CODE>");

	switch (option)
	{
	  case Option_MD5:
	    print_md5(&assoc_obj.file);
	    printf(" &nbsp;");
	    break;

	  case Option_SHA1:
	    print_sha1(&assoc_obj.file);
	    printf(" &nbsp;");
	    break;

	  case Option_SHA256:
	    print_sha256(&assoc_obj.file);
	    printf(" &nbsp;");
	    break;

	  case Option_SHA512:
	    print_sha512(&assoc_obj.file);
	    printf(" &nbsp;");
	    break;

	  case Option_Time:
	    print_time(assoc_obj.modification_time);
	    printf(" &nbsp;");
	    break;

	  default:
	    break;
	}

	get_path_name(&assoc_obj.path, left_name, sizeof(left_name));

	printf("%s", left_name);

	if (do_revision)
	{
	    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));
	    revision = 1;
	    while (!is_void_reference(&file_obj.header.object.prev_version))
	    {
		copy_reference(&next_ref, &file_obj.header.common.id);
		copy_reference(&prev_ref, &file_obj.header.object.prev_version);
		nwos_read_object_from_disk(&prev_ref, &file_obj, sizeof(file_obj));
		assert(is_same_object(&file_obj.header.object.next_version, &next_ref));
		revision++;
	    }
	    printf(";%d", revision);
	}

	printf("</CODE><BR>\n");
    }

    nwos_terminate_objectify();
}


int main(int argc, char* argv[])
{
    char* query_string;
    char* title = "ERROR: invalid query";
    Option option = Option_Error;
    int i;

    printf("Content-type: text/html\n\n");

    printf("<HTML>\n");

    query_string = getenv("QUERY_STRING");

    if (query_string == NULL || query_string[0] == '\0')
    {
	option = No_Option;
	title = "";
    }
    else if (strcmp(query_string, "time") == 0)
    {
	option = Option_Time;
	title = "with Creation Time";
    }
    else if (strcmp(query_string, "md5") == 0)
    {
	option = Option_MD5;
	title = "with MD5 Checksum";
    }
    else if (strcmp(query_string, "sha1") == 0)
    {
	option = Option_SHA1;
	title = "with SHA1 Checksum";
    }
    else if (strcmp(query_string, "sha256") == 0)
    {
	option = Option_SHA256;
	title = "with SHA256 Checksum";
    }
    else if (strcmp(query_string, "sha512") == 0)
    {
	option = Option_SHA512;
	title = "with SHA512 Checksum";
    }

    printf("<HEAD>\n");
    printf("  <TITLE>www.worldwide-database.org: List of Files %s</TITLE>\n", title);
    printf("</HEAD>\n");

    /* printf("<!--#include virtual=\"header.html\"-->\n"); */    /* server side include */

    printf("<BODY BGCOLOR=\"#ffffff\">\n");

    printf("  <CENTER><H1>List of Files %s</H1></CENTER>\n", title);

    printf("<P>\n");

    printf("<UL>\n");
    for (i = 0; i < num_sections; i++)
    {
	printf("<LI><A HREF=\"#%s\">%s</A></LI>\n", sections[i].anchor, sections[i].section);
    }
    printf("</UL>\n");

    if (option == Option_Error)
    {
	printf("ERROR: invalid QUERY_STRING");
    }
    else
    {
	list_files(option);
    }

    printf("</P>\n");

    printf("</BODY>\n");
    printf("</HTML>\n");

    return 0;
}


