/*
--          This file is part of the New World OS and Objectify projects
--      Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-12-03 11:36:21 -0700 (Sat, 03 Dec 2011) $
--   $Revision: 4979 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revisions of this file and the objectify.c file, since this file
--   was created from code taken from it.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "chunk_info.h"        /* define BLOCKS_IN_CHUNK - THIS needs to be fixed!! */
#include "config.h"
#include "disk_io.h"
#include "error.h"
#include "header.h"
#include "log.h"
#include "mem_alloc.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "user_config.h"


int nwos_verbosity_level = 0;


const char*  nwos_public_path;
const char*  nwos_private_path;

static int    public_file_desc;
static int    private_file_desc;
static bool    read_only;
static bool    modified;
static bool    total_chunks_modified = false;

ObjRef* nwos_storage_index;
int     nwos_storage_count;



/*****************/
/* Archive type. */
/*****************/

bool nwos_archive_is_block_device(void)
{
    assert(nwos_private_path[0] == '/');   /* must be a full path specified */

    return nwos_private_path[1] == 'd' && nwos_private_path[2] == 'e' && nwos_private_path[3] == 'v' && nwos_private_path[4] == '/';
}

bool nwos_archive_is_file(void)
{
    return !nwos_archive_is_block_device();
}


/*****************************************/
/* Function that indexes compressed file */
/*****************************************/

#define INDEX_ALLOC_SIZE 262144 

void index_compressed_file()
{
    int i;
    int size_of_index = INDEX_ALLOC_SIZE;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    uint8 header[FILE_BLOCK_SIZE];
    char* index_path = NULL;
    FILE* index_fp;

#ifdef NO_INDEX_FILE
    index_fp = NULL;
#else
    assert(private_file_desc > 0);

    /* first see if there is an existing index file */

    i = strlen(nwos_private_path);
    index_path = nwos_malloc(i+5);  /* 4 for .ndx and 1 for null */
    strlcpy(index_path, nwos_private_path, i+5);

    if (index_path[i-4] == '.' && index_path[i-3] == 'o' && index_path[i-2] == 'b' && index_path[i-1] == 'j')
    {
	i = i - 4;   /* write over the old .obj */
    }

    index_path[i++] = '.';
    index_path[i++] = 'n';
    index_path[i++] = 'd';
    index_path[i++] = 'x';
    index_path[i++] = '\0';

    fprintf(stderr, "index_path: %s\n", index_path);

    index_fp = fopen(index_path, "r");
#endif

    nwos_storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(nwos_storage_index != NULL);

    /* read the first block, it contains the header */
    bytes_read = read(private_file_desc, header, sizeof(header));

    assert(bytes_read == sizeof(header));

    /* if there is an index file see if the header matches */
    if (index_fp != NULL)
    {
	bytes_read = fread(buffer, 1, FILE_BLOCK_SIZE, index_fp);
	assert(bytes_read == sizeof(buffer));

	fprintf(stderr, "Index file found: ");
	fflush(stderr);

	if (memcmp(header, buffer, FILE_BLOCK_SIZE) == 0)   /* we have a winner */
	{
	    fprintf(stderr, "match\n");
	}
	else
	{
	    fprintf(stderr, "no match\n");
	    fclose(index_fp);
	    index_fp = NULL;
	}
    }
    else /* no index file */
    {
	errno = 0;   /* clear errno for later */
    }

    fprintf(stderr, "Indexing");
    fflush(stderr);

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += INDEX_ALLOC_SIZE;
	    nwos_storage_index = realloc(nwos_storage_index, size_of_index * sizeof(ObjRef));
	    assert(nwos_storage_index != NULL);
	}

	assert(i < size_of_index);

	if (index_fp != NULL)   /* read index from file */
	{
	    bytes_read = fread(&nwos_storage_index[i], sizeof(ObjRef), 1, index_fp);

	    if (bytes_read != 1) break;
	}
	else
	{
	    bytes_read = read(private_file_desc, buffer, sizeof(buffer));

	    if (bytes_read != sizeof(buffer)) break;

	    memcpy(&nwos_storage_index[i], &buffer[4], sizeof(ObjRef));
	}

	i++;
    }

    nwos_storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);

    if (index_fp != NULL)
    {
	if (ferror(index_fp))
	{
	    nwos_perror(index_path);
	    exit(1);
	}
	fclose(index_fp);
    }
    else
    {
	if (errno != 0)
	{
	    nwos_perror(nwos_private_path);
	    exit(1);
	}

#ifndef NO_INDEX_FILE
	index_fp = fopen(index_path, "w");

	if (index_fp == NULL)
	{
	    fprintf(stderr, "Could not create index file: ");
	    nwos_perror(index_path);
	    fprintf(stderr, "Index will not be saved!\n");
	}
	else
	{
	    fprintf(stderr, "Writing index file: %s\n", index_path);

	    if (fwrite(header, 1, sizeof(header), index_fp) != sizeof(header))
	    {
		nwos_perror(index_path);
		exit(1);
	    }

	    if (fwrite(&nwos_storage_index[1], sizeof(ObjRef), i, index_fp) != i)
	    {
		nwos_perror(index_path);
		exit(1);
	    }

	    if (fclose(index_fp))
	    {
		nwos_perror(index_path);
		exit(1);
	    }
	}
#endif
    }

    assert(bytes_read == 0);

    if (index_path != NULL)
    {
	nwos_free(index_path);
    }
}



/**********************************************/
/* Function to initialize the disk I/O module */
/**********************************************/

void nwos_initialize_disk_io(AccessType type, const char* path)
{
    char log_msg[256];
    bool compressed_file = false;
    struct flock lock;
    const char* error_msg;
    uint8 buffer[FILE_BLOCK_SIZE];

    /* make sure the storage is something we can deal with */
#ifdef PUBLIC_MODE
    assert(type == PUBLIC);
#else
    assert(type == PUBLIC || type == READ_ONLY || type == READ_WRITE);
#endif

    assert(nwos_private_path == NULL);   /* make sure this is only called once */
    assert(private_file_desc == 0);
    assert(public_file_desc == 0);
    assert(modified == false);

    nwos_public_path = nwos_get_public_objects_path();

    if (type == PUBLIC)
    {
	assert(path == NULL);
    }
    else
    {
	if (path == NULL)    /* no compressed file was passed in */
	{
	    nwos_private_path = nwos_get_private_objects_path();
//	    compressed_file = (strstr(nwos_private_path, "compressed") != NULL);   -- for now assume that compressed file can't be set via config
	}
	else
	{
	    assert(type == READ_ONLY);
	    nwos_private_path = path;
	    compressed_file = true;
	}
    }

    /********************************/
    /* open the public objects file */
    /********************************/

#ifdef PUBLIC_MODE
    public_file_desc = open(nwos_public_path, O_RDWR);
#else
    public_file_desc = open(nwos_public_path, O_RDONLY);
#endif

    if (public_file_desc < 0)
    {
	nwos_perror(nwos_public_path);
	exit(1);
    }

#ifdef PUBLIC_MODE
    lock.l_type = F_WRLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(public_file_desc, F_SETLK, &lock) != 0)
    {
	nwos_perror(nwos_public_path);
	exit(1);
    }
#endif

    snprintf(log_msg, sizeof(log_msg), " public_path: %s", nwos_public_path);
    nwos_log(log_msg);

    if (read(public_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
    {
	snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_public_path);
	nwos_perror(log_msg);
	exit(1);
    }

    error_msg = nwos_load_public_data(buffer, sizeof(buffer));

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, nwos_public_path);
	exit(1);
    }

    snprintf(log_msg, sizeof(log_msg), " used public blocks: %llu", nwos_used_public_blocks);
    nwos_log(log_msg);

#ifndef PUBLIC_MODE
    /* don't set read_only flag if in PUBLIC_MODE */
    read_only = (type == PUBLIC || type == READ_ONLY);


    /***************************************************/
    /* if a private objects file was specified open it */
    /***************************************************/

    if (nwos_private_path != NULL)
    {
	if (read_only)
	{
	    private_file_desc = open(nwos_private_path, O_RDONLY);
	}
	else
	{
	    private_file_desc = open(nwos_private_path, O_RDWR);
	}

	if (private_file_desc < 0)
	{
	    nwos_perror(nwos_private_path);
	    exit(1);
	}

	lock.l_type = read_only ? F_RDLCK : F_WRLCK;
	lock.l_whence = SEEK_SET;
	lock.l_start = 0;
	lock.l_len = 0;

	if (fcntl(private_file_desc, F_SETLK, &lock) != 0)
	{
	    nwos_perror(nwos_private_path);
	    exit(1);
	}

	snprintf(log_msg, sizeof(log_msg), " private_path: %s", nwos_private_path);
	nwos_log(log_msg);

	if (compressed_file)
	{
	    index_compressed_file();   /* build an index into the compressed file */

	    if (lseek(private_file_desc, 0LL, SEEK_SET) < 0)
	    {
		nwos_perror("rewinding after indexing");
		exit(1);
	    }
	}

	if (read(private_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
	{
	    snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_private_path);
	    nwos_perror(log_msg);
	    exit(1);
	}

	error_msg = nwos_load_private_data(buffer, sizeof(buffer), compressed_file);

	if (error_msg != NULL)
	{
	    fprintf(stderr, "%s: %s\n", error_msg, nwos_private_path);
	    exit(1);
	}

	snprintf(log_msg, sizeof(log_msg), "  total private chunks: %u  used: %u  blocks: %llu", nwos_total_private_chunks, nwos_used_private_chunks, nwos_used_private_blocks);
	nwos_log(log_msg);
    }
#endif
}


/****************/
/* Open status. */
/****************/

bool nwos_disk_io_is_read_only()
{
    return read_only;
}

bool nwos_disk_io_is_read_write()
{
    return !read_only;
}

bool nwos_disk_io_is_public_only()
{
    return private_file_desc == 0;
}

bool nwos_disk_io_is_compressed_file(void)
{
    return nwos_storage_index != NULL;
}


/*********************************************/
/* Function to terminate the disk I/O module */
/* Returns true if disk was modified         */
/*********************************************/

bool nwos_terminate_disk_io()
{
    uint8 buffer[FILE_BLOCK_SIZE];
    char log_msg[256];
    bool disk_was_modified = false;

    if (modified)
    {
	memset(buffer, 0, sizeof(buffer));

#ifdef PUBLIC_MODE
	nwos_store_public_data(buffer, sizeof(buffer));
	snprintf(log_msg, sizeof(log_msg), "  used public blocks: %llu", nwos_used_public_blocks);
#else
	nwos_store_private_data(buffer, sizeof(buffer), nwos_archive_is_file() | total_chunks_modified);
	snprintf(log_msg, sizeof(log_msg), "  total private chunks: %u  used: %u  blocks: %llu", nwos_total_private_chunks, nwos_used_private_chunks, nwos_used_private_blocks);
#endif
	nwos_log(log_msg);

#ifdef PUBLIC_MODE
	if (lseek(public_file_desc, (off_t)0, SEEK_SET) < 0)
#else
	if (lseek(private_file_desc, (off_t)0, SEEK_SET) < 0)
#endif
	{
	    nwos_perror(nwos_private_path);
	    exit(1);
	}

#ifdef PUBLIC_MODE
	if (write(public_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
#else
	if (write(private_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
#endif
	{
	    nwos_perror(nwos_private_path);
	    exit(1);
	}

	modified = false;
	total_chunks_modified = false;
	disk_was_modified = true;
    }
    else
    {
	nwos_log("  No changes made");
    }

    if (private_file_desc > 0 && close(private_file_desc) < 0)
    {
	nwos_perror(nwos_private_path);
    }

    private_file_desc = 0;
    nwos_private_path = NULL;


    if (public_file_desc > 0 && close(public_file_desc) < 0)
    {
	nwos_perror(nwos_public_path);
    }

    public_file_desc = 0;

    return disk_was_modified;
}

#ifndef PUBLIC_MODE

/******************************************************/
/* This function reads the chunk info table from disk */
/******************************************************/

void nwos_read_chunk_info(void* chunk_info, size_t chunk_info_size_in_bytes)
{
    char log_msg[128];

    if (lseek(private_file_desc, FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	nwos_perror("chunk index");
	exit(1);
    }

    if (read(private_file_desc, nwos_chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
    {
	snprintf(log_msg, sizeof(log_msg), "reading chunk info from: %s", nwos_private_path);
	nwos_perror(log_msg);
	exit(1);
    }
}


/*****************************************************/
/* This function writes the chunk info table to disk */
/*****************************************************/

void nwos_write_chunk_info(void* chunk_info, size_t chunk_info_size_in_bytes)
{
    char log_msg[128];

    if (lseek(private_file_desc, (off_t)FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	nwos_perror(nwos_private_path);
	exit(1);
    }

    if (write(private_file_desc, nwos_chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
    {
	snprintf(log_msg, sizeof(log_msg), "writing chunk info to: %s", nwos_private_path);
	nwos_perror(log_msg);
	exit(1);
    }
}

#endif


/**********************************************************************************************/
/* This function returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */
/**********************************************************************************************/

void nwos_write_empty_chunk(uint32 chunk_index)
{
    uint32 offset;
    static uint8* chunk;

    if (nwos_archive_is_file())
    {
	assert(chunk_index == nwos_total_private_chunks);
    }
    else
    {
	assert(chunk_index < nwos_total_private_chunks);
    }

    if (chunk == NULL)   /* first time this has been called, allocate memory */
    {
	chunk = malloc(CHUNK_SIZE);

	if (chunk == NULL)
	{
	    nwos_perror("allocate memory for creating new chunk");
	    exit(1);
	}

	memset(chunk, 0, CHUNK_SIZE);
    }

    offset = nwos_block_offset_to_chunks + (chunk_index * BLOCKS_IN_CHUNK);

    if (lseek(private_file_desc, (off_t)offset << 8, SEEK_SET) < 0)
    {
	nwos_perror(nwos_private_path);
	exit(1);
    }

    if (write(private_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	nwos_perror(nwos_private_path);
	exit(1);
    }

    /* if the archve is a file, this expanded it, increase the counts */

    if (nwos_archive_is_file())
    {
	nwos_total_private_chunks++;
    }
}


void nwos_write_bit_map(int chunk_index, void* map)
{
    uint32 chunk;
    char log_msg[80];

    assert(chunk_index < nwos_used_private_chunks);

    chunk = nwos_block_offset_to_chunks + chunk_index * BLOCKS_IN_CHUNK;

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map lseek chunk:%08x", chunk);
	nwos_perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map write chunk:%08x", chunk);
	nwos_perror(log_msg);
	exit(1);
    }

    modified = true;
}


void nwos_read_bit_map(int chunk_index, void* map)
{
    uint32 chunk;
    char log_msg[80];

    assert(chunk_index < nwos_used_private_chunks);

    chunk = nwos_block_offset_to_chunks + chunk_index * BLOCKS_IN_CHUNK;

    if (lseek(private_file_desc, (off_t)chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map lseek chunk:%08x", chunk);
	nwos_perror(log_msg);
	exit(1);
    }

    if (read(private_file_desc, map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map write chunk:%08x", chunk);
	nwos_perror(log_msg);
	exit(1);
    }
}


#ifndef PUBLIC_MODE

//static float chunk_usage(Bit_Map_Cache_Entry* entry)
//{
//    float result;

//    result = (float)chunk_info[entry->index].used / (float)USABLE_BLOCKS_PER_CHUNK;

//    assert(0.0f <= result && result <= 1.0f);

//    return result;
//}
#endif



/****************************************************************/
/* Function to read a block from storage given the block offset */
/****************************************************************/

bool nwos_read_block_from_offset(Archive_Type archive, int64 offset, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    int file_desc;

    if (archive == Public_Archive || nwos_disk_io_is_compressed_file())   /* it is the public file or a compressed file */
    {
	assert(offset > 0);
    }
    else                                                         /* normal private storage */
    {
	assert(offset > nwos_block_offset_to_chunks);
    }

    assert(archive == Public_Archive || archive == Private_Archive);

    assert(sizeof(offset) == sizeof(off_t));    /* make sure nothing strange has happened */

    if (archive == Public_Archive)
    {
	file_desc = public_file_desc;
    }
    else
    {
	file_desc = private_file_desc;
    }

    if (lseek(file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(ref_str, sizeof(ref_str), "nwos_read_block_from_offset lseek offset:%x%08x", offset_upr, offset_lwr);

	nwos_perror(ref_str);
	exit(1);
    }

    return read(file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;
}


/****************************************/
/* Function to write a block to storage */
/****************************************/

bool nwos_write_block_to_offset(Archive_Type archive, int64 offset, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];


    assert(offset > nwos_block_offset_to_chunks);

    assert(sizeof(offset) == sizeof(off_t));    /* make sure nothing strange has happened */

#ifdef PUBLIC_MODE
    assert(archive == Public_Archive);

    if (lseek(public_file_desc, offset, SEEK_SET) < 0)
#else
    assert(archive == Private_Archive);

    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
#endif
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "nwos_write_block_to_offset lseek offset:%x%08x", offset_upr, offset_lwr);

	nwos_perror(log_msg);
	exit(1);
    }

    modified = true;

#ifdef PUBLIC_MODE
    return write(public_file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;
#else
    return write(private_file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;
#endif
}


/* This function is for the archive-resize program */

void nwos_change_total_chunks(uint32 new_chunks)
{
    assert(new_chunks >= nwos_used_private_chunks);

    nwos_total_private_chunks = new_chunks;

    modified = true;
    total_chunks_modified = true;
}


