/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-12-03 11:36:21 -0700 (Sat, 03 Dec 2011) $
--   $Revision: 4979 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <ctype.h>
#include <limits.h>    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "error.h"
#include "log.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "user_config.h"


static bool config_file_has_been_read;

static struct path_table
{
    char* name;
    char* env;
    char* default_path;
    char* path;
    size_t path_length;     /* if non-zero path was malloc'd */
} path_table[] = {
    { "public",  "OBJECTIFY_PUBLIC_PATH",   DEFAULT_PUBLIC_PATH,   NULL, 0 },
    { "private", "OBJECTIFY_PRIVATE_PATH",  DEFAULT_PRIVATE_PATH,  NULL, 0 },
    { "log",     "OBJECTIFY_LOG_FILE_PATH", NULL,                  NULL, 0 },
    { "backup",  "OBJECTIFY_BACKUP_PATH",   NULL,                  NULL, 0 },
};

#define PATH_TABLE_SIZE (sizeof(path_table) / sizeof(struct path_table))


/* get the user's home directory */

const char* get_users_home_dir(const char* path)
{
    static char* result = NULL;
    char msg[128];

    if (result == NULL)
    {
	result = getenv("HOME");

	if (result == NULL)
	{
	    snprintf(msg, sizeof(msg), "WARNING: $HOME environment variable not defined, cannot resolve path: %s", path);

	    nwos_log(msg);
	    fprintf(stderr, "%s\n", msg);
	}
    }

    return result;
}


static void read_configuration_file()
{
    FILE* fp = NULL;
    char config_path[PATH_MAX];
    char buffer[1024];
    int line_number = 0;
    char *p;
    int i;

    assert(DEFAULT_SYSTEM_CONFIG_PATH[0] == '/');

    assert(DEFAULT_USER_CONFIG_PATH[0] == '/' || (DEFAULT_USER_CONFIG_PATH[0] == '~' && DEFAULT_USER_CONFIG_PATH[1] == '/'));

    if (!config_file_has_been_read)
    {
	config_file_has_been_read = true;

	/* check all environment variables first */

	for (i = 0; i < PATH_TABLE_SIZE; i++)
	{
	    path_table[i].path = getenv(path_table[i].env);
	}	    

	/* then read the file */

#ifndef HTTP
	/* first try to open a users config file */
	if (*DEFAULT_USER_CONFIG_PATH == '~')
	{
	    strlcpy(config_path, get_users_home_dir(DEFAULT_USER_CONFIG_PATH), sizeof(config_path));
	    strlcat(config_path, DEFAULT_USER_CONFIG_PATH+1, sizeof(config_path));
	}
	else
	{
	    strlcpy(config_path, DEFAULT_USER_CONFIG_PATH, sizeof(config_path));
	}

	fp = fopen(config_path, "r");
#endif

	/* if that failed try to open the system config file */
	if (fp == NULL)
	{
	    strlcpy(config_path, DEFAULT_SYSTEM_CONFIG_PATH, sizeof(config_path));
	    fp = fopen(config_path, "r");
	}

	if (fp != NULL)
	{
	    while (fgets(buffer, sizeof(buffer), fp) != NULL)
	    {
		line_number++;

		p = strchr(buffer, '\n');

		if (p == NULL)    /* line too long */
		{
		    if (buffer[0] != '#')
		    {
			snprintf(buffer, sizeof(buffer), "WARNING: config file '%s' line %d is too long, ignored",
				 config_path, line_number);

			nwos_log(buffer);
			fprintf(stderr, "%s\n", buffer);
		    }

		    while (fgets(buffer, sizeof(buffer), fp) != NULL && strchr(buffer, '\n') == NULL); /* skip over */
		}
		else
		{
		    *p = '\0';    /* eliminate new line character */

		    for (p = buffer; *p != '\0' && *p != '#'; p++)   /* skip over leading spaces */
		    {
			assert(p < buffer + sizeof(buffer));

			if (isspace(*p))
			{
			    *p++ = '\0';

			    while (*p != '\0' && isspace(*p)) p++;   /* skip over any extra whitespace */

			    for (i = 0; i < PATH_TABLE_SIZE; i++)
			    {
				if (strcmp(path_table[i].name, buffer) == 0) break;
			    }

			    if (i < PATH_TABLE_SIZE)   /* found it */
			    {
				if (path_table[i].path == NULL)
				{
				    path_table[i].path_length = strlen(p) + 1;

				    path_table[i].path = malloc(path_table[i].path_length);

				    if (path_table[i].path == NULL)
				    {
					nwos_perror("allocating memory for configuration path");
					exit(1);
				    }

				    strlcpy(path_table[i].path, p, path_table[i].path_length);
				}
			    }
			    else
			    {
				snprintf(buffer, sizeof(buffer), "WARNING: config file '%s' line %d is invalid, ignored",
					 config_path, line_number);

				nwos_log(buffer);
				fprintf(stderr, "%s\n", buffer);
			    }

			    break;
			}
		    }
		}
	    }

	    fclose(fp);
	}

	/* finally fill in any null values with the default */

	for (i = 0; i < PATH_TABLE_SIZE; i++)
	{
	    if (path_table[i].path == NULL)
	    {
		path_table[i].path = path_table[i].default_path;
	    }
	}	    

    }
}

static const char* get_path(char* name)
{
    int i;
    const char* home;
    char* combined;
    size_t combined_length;


    for (i = 0; i < PATH_TABLE_SIZE; i++)
    {
	if (strcmp(path_table[i].name, name) == 0) break;
    }

    assert(i < PATH_TABLE_SIZE);

    if (path_table[i].path == NULL)
    {
	read_configuration_file();     /* read the environment variables and the config file */
    }

    if (path_table[i].path != NULL && path_table[i].path[0] == '~')
    {
	home = get_users_home_dir(path_table[i].path);

	if (home == NULL)
	{
	    if (path_table[i].path_length != 0)  /* path was malloc'd */
	    {
		free(path_table[i].path);
	    }

	    path_table[i].path = NULL;
	}
	else
	{
	    combined_length = strlen(home) + 1 + strlen(path_table[i].path) + 1;
	    combined = malloc(combined_length);

	    strlcpy(combined, home, combined_length);
	    if (path_table[i].path[1] != '/')
	    {
		strlcat(combined, "/", combined_length);
	    }
	    strlcat(combined, path_table[i].path + 1, combined_length);

	    if (path_table[i].path_length != 0)  /* path was malloc'd */
	    {
		free(path_table[i].path);   /* better free it */
	    }

	    path_table[i].path = combined;
	    path_table[i].path_length = combined_length;
	}
    }

    return path_table[i].path;
}


const char* nwos_get_public_objects_path()
{
    return get_path("public");
}


const char* nwos_get_private_objects_path()
{
    return get_path("private");
}


const char* nwos_get_log_file_path()
{
    return get_path("log");
}


const char* nwos_get_backup_directory_path()
{
    return get_path("backup");
}


void nwos_set_private_objects_path(const char* path)
{
    int i;

    if (strcmp(nwos_get_private_objects_path(), path) == 0) return;   /* If that is what it is already, no need to proceed */

    /* NOTE: the above call also runs the normal stuff so it won't run after we set the private path and write over it. */

    for (i = 0; i < PATH_TABLE_SIZE; i++)
    {
	if (strcmp(path_table[i].name, "private") == 0) break;
    }

    assert(i < PATH_TABLE_SIZE);

    if (path_table[i].path_length != 0)
    {
	free(path_table[i].path);
    }

    path_table[i].path = (char*)path;
}
