/*
--          This file is part of the New World OS and Objectify projects
--   Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-07-06 21:33:05 -0600 (Wed, 06 Jul 2011) $
--   $Revision: 4912 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the objectify.c file which this
--   file was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bit_map.h"             /* define nwos_block_used() */
#include "class_definition.h"    /* define nwos_reference_list_class_ref */
#include "crc32.h"
#include "gen_id.h"
#include "log.h"
#include "objects.h"
#include "security.h"
#include "storage.h"


#define MAX_TEMPORARY_OBJECTS 1024

static void* temporary_objects[MAX_TEMPORARY_OBJECTS];



/* Defined in file.c, not for general usage */

extern void nwos_update_file_001_object_to_current(void* object, size_t size);
extern void nwos_update_file_002_object_to_current(void* object, size_t size);


/*************************/
/* Verification routines */
/*************************/

#if 0
static void check_common_header(ObjRef* ref, CommonHeader* common)
{
  /* could check time stamp is valid here?  anything else? */
}
#endif


static bool object_header_checksum_matches(ObjectHeader* obj_header, uint8 stored[4])
{
    uint8 computed[4];

    nwos_crc32_calculate((uint8*)obj_header, sizeof(*obj_header), computed);

//    return (stored[0] == computed[0] && stored[1] == computed[1] && stored[2] == computed[2] && stored[3] == computed[3]);
    return (memcmp(stored, computed, sizeof(computed)) == 0);
}


static bool check_object_data(ObjRef* ref, uint8* object, size_t total_size, uint8 stored[4])
{
    uint8 computed[4];
    size_t data_size;

    data_size = total_size - sizeof(EveryObject);

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)object + sizeof(EveryObject), data_size, computed);

    return memcmp(computed, stored, sizeof(computed)) == 0;
}


/*********************************/
/* Routines to read objects, etc */
/*********************************/

#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


bool nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size)
{
    EveryObject*  header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;
    bool use_old_decryption = false;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    snprintf(log_msg, sizeof(log_msg), "nwos_read_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    assert(size > sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    if (nwos_reference_type(ref) == Temporary_Reference)
    {
	i = (int) (nwos_ref_to_word(ref) - MINIMUM_TEMPORARY_REFERENCE);

	assert(0 <= i && i < MAX_TEMPORARY_OBJECTS);

	memcpy(object, temporary_objects[i], size);

	header = (EveryObject*) object;

	if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* could be old encryption */
	{
	    return false;
	}
    }
    else
    {
	memset(ivec, 0, sizeof(ivec));

	bytes_remaining = size;

	ptr_obj = object;

	if (!nwos_read_block_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, seq)) return false;

	assert(sizeof(ObjRef) == 4);

	header = (EveryObject*) &extra;

	/* check_common_header(ref, &header->common);     -- this doesn't do anything right now */

	if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* could be old encryption */
	{
#ifdef PUBLIC_MODE
	    return false;
#else
	    memset(ivec, 0, sizeof(ivec));
	    if (!nwos_read_block_from_disk_and_old_decrypt(ref, &extra, sizeof(extra), ivec, seq)) return false;

	    if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* neither decryption worked */
	    {
		return false;
	    }

	    use_old_decryption = true;
#endif
	}

	*ptr_obj++ = extra.flags[0];
	*ptr_obj++ = extra.flags[1];
	*ptr_obj++ = extra.flags[2];
	*ptr_obj++ = extra.flags[3];
	*ptr_obj++ = extra.id.id[0];
	*ptr_obj++ = extra.id.id[1];
	*ptr_obj++ = extra.id.id[2];
	*ptr_obj++ = extra.id.id[3];

	while (bytes_remaining > FILE_BLOCK_SIZE)
	{
	    for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	    {
		*ptr_obj++ = extra.data[i];
		bytes_remaining--;
	    }

	    /* save the reference because the extra block is going to get written over, we don't want it to change */
	    copy_reference(&next_ref, &extra.next_block);

	    seq++;

	    if (use_old_decryption)
	    {
		if (!nwos_read_block_from_disk_and_old_decrypt(&next_ref, &extra, sizeof(extra), ivec, seq)) return false;
	    }
	    else
	    {
		if (!nwos_read_block_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, seq)) return false;
	    }
	}

	assert(bytes_remaining <= FILE_BLOCK_SIZE);

	for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];
    }

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

#ifndef USE_PREDEFINED_STRUCTS
#ifndef PUBLIC_MODE
    /* check to see if this is a file object (which has an older revision) */
    if (nwos_reference_type(ref) == Private_Reference && is_same_object(&header->common.class_definition, nwos_get_file_001_reference()))
    {
	if (check_object_data(ref, (uint8*)object, sizeof(C_struct_File_001), header->common.data_chksum))
	{
	    nwos_update_file_001_object_to_current(object, size);
	    return true;
	}
    }
    else if (nwos_reference_type(ref) == Private_Reference && is_same_object(&header->common.class_definition, nwos_get_file_002_reference()))
    {
	if (check_object_data(ref, (uint8*)object, sizeof(C_struct_File_002), header->common.data_chksum))
	{
	    nwos_update_file_002_object_to_current(object, size);
	    return true;
	}
    }
    else
#endif
#endif
    {
	if (check_object_data(ref, (uint8*)object, size, header->common.data_chksum))
	{
	    return true;
	}
    }

    return false;
}


bool nwos_read_variable_sized_object_from_disk(ObjRef* ref, void* object, size_t max_size, size_t (*size_function)(void*))
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;
//    size_t xyzzy;
//    char msg[128];
    bool use_old_decryption = false;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    snprintf(log_msg, sizeof(log_msg), "nwos_read_variable_size_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    memset(ivec, 0, sizeof(ivec));

    ptr_obj = object;

    if (!nwos_read_block_from_disk_and_decrypt(ref, &extra, sizeof(extra), ivec, seq)) return false;

    assert(sizeof(ObjRef) == 4);

    header = (EveryObject*) &extra;

    /* check_common_header(ref, &header->common);     -- this doesn't do anything right now */

    if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* could be old encryption */
    {
#ifdef PUBLIC_MODE
	return false;
#else
	memset(ivec, 0, sizeof(ivec));
	if (!nwos_read_block_from_disk_and_old_decrypt(ref, &extra, sizeof(extra), ivec, seq)) return false;

	if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* neither decryption worked */
	{
	    return false;
	}

	use_old_decryption = true;
#endif
    }

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    bytes_remaining = (*size_function)(&extra);
    //xyzzy = nwos_get_object_size(&extra);

    //    assert(xyzzy == 0 || bytes_remaining == xyzzy);
    //if (xyzzy != 0 && bytes_remaining != xyzzy)
    //{
    //	snprintf(msg, sizeof(msg), "Warning: size_function: %zd  get_object_size: %zd", bytes_remaining, xyzzy);
    //	nwos_log(msg);
    //	printf("%s\n", msg);
    //}

    assert(bytes_remaining <= max_size);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;


	if (use_old_decryption)
	{
	    if (!nwos_read_block_from_disk_and_old_decrypt(&next_ref, &extra, sizeof(extra), ivec, seq)) return false;
	}
	else
	{
	    if (!nwos_read_block_from_disk_and_decrypt(&next_ref, &extra, sizeof(extra), ivec, seq)) return false;
	}
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    return check_object_data(ref, (uint8*)object, (*size_function)(object), header->common.data_chksum);
}


void nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header)
{
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(ref));

    memset(ivec, 0, sizeof(ivec));
    assert(nwos_read_block_from_disk_and_decrypt(ref, header, sizeof(*header), ivec, 0));

    /* check_common_header(ref, &header->common); -- this currently doesn't do anything */

    if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* could be old encryption */
    {
#ifdef PUBLIC_MODE
	printf("\n");
	fflush(stdout);
	fprintf(stderr, "Unable to decrypt object headers: %08x\n", nwos_ref_to_word(ref));
	nwos_terminate_objectify();
	exit(1);
#else
	memset(ivec, 0, sizeof(ivec));
	assert(nwos_read_block_from_disk_and_old_decrypt(ref, header, sizeof(*header), ivec, 0));

	if (!object_header_checksum_matches(&header->object, header->common.header_chksum))   /* neither decryption worked */
	{
	    printf("\n");
	    fflush(stdout);
	    fprintf(stderr, "Unable to decrypt object headers: %08x\n", nwos_ref_to_word(ref));
	    exit(1);
	}
#endif
    }

#if 0
    printf("read header path: %s\n", path);
    printf("read header pointer: %p\n", header);
    {
      int i;
      for (i = 0; i < sizeof(EveryObject); i++) printf("%02x%c", *((uint8*)header+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif
}


/**********************************/
/* Routines to write objects, etc */
/**********************************/

void nwos_write_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    memset(ivec, 0, sizeof(ivec));


    assert(sizeof(extra) == FILE_BLOCK_SIZE);

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(&((EveryObject*)object)->common.class_definition) == Public_Reference);
#else
    if (nwos_reference_type(&((EveryObject*)object)->common.class_definition) != Private_Reference)  /* if not private, it has to be creating root */
    {
	static ObjRef public_root_object_class_def;
	if (is_void_reference(&public_root_object_class_def))   /* only read the public root object once */
	{
	    C_struct_Root root_obj;
	    ObjRef root_obj_ref;
	    root_obj_ref.id[0] = 0;
	    root_obj_ref.id[1] = 0;
	    root_obj_ref.id[2] = 0;
	    root_obj_ref.id[3] = 1;
	    assert(nwos_read_object_from_disk(&root_obj_ref, &root_obj, sizeof(root_obj)));
	    copy_reference(&public_root_object_class_def, &root_obj.header.common.class_definition);
	}
	assert(is_same_object(&((EveryObject*)object)->common.class_definition, &public_root_object_class_def));
    }
    assert(nwos_block_used(ref));      /* assert bit in map was set by generate id */
#endif

#ifdef VERIFY_BLOCK_IS_EMPTY
    /* NOTE: this was before using the bit map to verify that the block wasn't already */
    /* used, now we assume the bit map is correct even if the block is non-zero */
    assert(!nwos_read_block(ref, block) || is_void_reference((ObjRef*)&block[4]));   /* make sure block is zero */
#endif

    ptr_obj = (uint8*) object;

    assert(ptr_obj[0] == 0 && ptr_obj[1] == 0 && ptr_obj[2] == 0 && ptr_obj[3] == 0);

    ptr_obj += 4;

    assert(is_same_object(ref, (ObjRef*)ptr_obj));

    ptr_obj += 4;

    bytes_remaining = size;

    extra.flags[0] = 0;
    extra.flags[1] = 0;
    extra.flags[2] = 0;
    extra.flags[3] = 0;

    copy_reference(&extra.id, ref);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    extra.data[i] = *ptr_obj++;
	    bytes_remaining--;
	}

	nwos_generate_new_id(&extra.next_block);

	nwos_write_block_to_disk_and_encrypt(&extra.id, &extra, sizeof(extra), ivec, seq);

	copy_reference(&extra.id, &extra.next_block);

	seq++;
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) extra.data[i] = *ptr_obj++;

    nwos_write_block_to_disk_and_encrypt(&extra.id, &extra, bytes_remaining, ivec, seq);
}


void nwos_overwrite_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 ivec[IVEC_SIZE];
    uint8 block[FILE_BLOCK_SIZE];

    assert(size <= FILE_BLOCK_SIZE);        /* this function cannot handle objects larger than one block yet!! */
    assert(nwos_read_block(ref, block));    /* verify that there is already an object there */

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(&((EveryObject*)object)->common.class_definition) == Public_Reference);
#else
    assert(nwos_reference_type(&((EveryObject*)object)->common.class_definition) == Private_Reference);
    assert(nwos_block_used(ref));      /* assert bit in map was set by generate id */
#endif

    memset(ivec, 0, sizeof(ivec));

    nwos_write_block_to_disk_and_encrypt(ref, object, size, ivec, 0);
}


#ifndef PUBLIC_MODE
void nwos_remove_object(ObjRef* ref)
{
    /* this function should read the object and erase all blocks, functions should not call this to erase a block, they should call erase_block */

    bool this_needs_to_be_fixed_to_handle_multiple_block_objects = false;

    assert(this_needs_to_be_fixed_to_handle_multiple_block_objects);

    nwos_erase_block(ref);
}
#endif


void nwos_write_public_object_to_disk(ObjRef* ref, void* object, size_t size)
{
    uint8 block[FILE_BLOCK_SIZE];
//    int i;

    assert(size <= FILE_BLOCK_SIZE);
    assert((ref->id[0] & 0xF0) == 0);

    memset(block, 0, sizeof(block));

    memcpy(block, object, size);

    nwos_write_block(ref, block);

//    printf("write_public_object: %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
}


/***************/
/* Class stuff */
/***************/

/*--------------------------------------------------------------------------------*/
/* Get an object's class reference, without updating to the latest class revision */
/*--------------------------------------------------------------------------------*/

void nwos_get_object_class_without_update(ObjRef* obj, ObjRef* object_class)
{
    EveryObject header;
    uint8 ivec[IVEC_SIZE];
    int i;

    assert(!is_void_reference(obj));

    if (nwos_reference_type(obj) == Temporary_Reference)
    {
	i = (int) (nwos_ref_to_word(obj) - MINIMUM_TEMPORARY_REFERENCE);

	assert(0 <= i && i < MAX_TEMPORARY_OBJECTS);

	memcpy(&header, temporary_objects[i], sizeof(header));
    }
    else
    {
	memset(ivec, 0, sizeof(ivec));

	assert(nwos_read_block_from_disk_and_decrypt(obj, &header, sizeof(header), ivec, 0));

	/* check_common_header(obj, &header.common);  -- this doesn't currently do anything */

	/* if it is not a reference list check the header checksum */

	if (!is_same_object(&header.common.class_definition, &nwos_reference_list_class_ref))
	{
	    if (!object_header_checksum_matches(&header.object, header.common.header_chksum))   /* could be old encryption */
	    {
#ifdef PUBLIC_MODE
		printf("\n");
		fflush(stdout);
		fprintf(stderr, "Unable to decrypt object class without update: %08x\n", nwos_ref_to_word(obj));
		nwos_terminate_objectify();
		exit(1);
#else
		memset(ivec, 0, sizeof(ivec));
		assert(nwos_read_block_from_disk_and_old_decrypt(obj, &header, sizeof(header), ivec, 0));

		if (!is_same_object(&header.common.class_definition, &nwos_reference_list_class_ref))
		{
		    if (!object_header_checksum_matches(&header.object, header.common.header_chksum))   /* neither decryption worked */
		    {
			printf("\n");
			fflush(stdout);
			fprintf(stderr, "Unable to decrypt object class without update: %08x\n", nwos_ref_to_word(obj));
			nwos_terminate_objectify();
			exit(1);
		    }
		}
#endif
	    }
	}
    }

    copy_reference(object_class, &header.common.class_definition);
}


/*-------------------------------------------------------------------------*/
/* Get an object's class reference and update to the latest class revision */
/*-------------------------------------------------------------------------*/

void nwos_get_object_class(ObjRef* obj, ObjRef* object_class)
{
    nwos_get_object_class_without_update(obj, object_class);

#ifndef USE_PREDEFINED_STRUCTS
    if (nwos_reference_type(obj) == Private_Reference && is_same_object(object_class, nwos_get_file_001_reference()))
    {
	copy_reference(object_class, nwos_get_file_class_reference());
    }
    else if (nwos_reference_type(obj) == Private_Reference && is_same_object(object_class, nwos_get_file_002_reference()))
    {
	copy_reference(object_class, nwos_get_file_class_reference());
    }
#endif
}


void nwos_fill_in_common_header(CommonHeader* common, ObjRef* ref, ObjRef* class_definition_ref)
{
    memset(common, 0, sizeof(CommonHeader));
    memcpy(&common->id, ref, sizeof(ObjRef));
    nwos_get_time_stamp(common->creation_time);
    memcpy(&common->class_definition, class_definition_ref, sizeof(common->class_definition));
}


void* nwos_malloc_temporary_object(size_t size, ObjRef* temp_ref)
{
    int i;

    for (i = 0; i < MAX_TEMPORARY_OBJECTS; i++)
    {
	if (temporary_objects[i] == NULL) break;
    }

    assert(i < MAX_TEMPORARY_OBJECTS);

    temporary_objects[i] = nwos_malloc(size);

    nwos_word_to_ref(MINIMUM_TEMPORARY_REFERENCE + (uint32)i, temp_ref);

//    printf("i: %08x  temp_ref: %08x\n", MINIMUM_TEMPORARY_REFERENCE + (uint32)i, nwos_ref_to_word(temp_ref) );

    return temporary_objects[i];
}



static void* common_free_temporary_object(ObjRef* ref)
{
    int i;
    uint32 uint32ref = nwos_ref_to_word(ref);
    void* object;

//    printf("uint32ref: %08x\n", uint32ref);

    assert(MINIMUM_TEMPORARY_REFERENCE <= uint32ref && uint32ref <= MAXIMUM_TEMPORARY_REFERENCE);

    i = (int) (uint32ref - MINIMUM_TEMPORARY_REFERENCE);

    assert(0 <= i && i < MAX_TEMPORARY_OBJECTS);

    object = temporary_objects[i];

    temporary_objects[i] = NULL;

    nwos_free(object);

    return object;   /* This value MUST only be used for verification in the nwos_free_temporary_object function!! */
}


void nwos_free_temporary_object_from_reference(ObjRef* ref)
{
    common_free_temporary_object(ref);
}


void nwos_free_temporary_object(void* object)
{
    void* verify;

    verify = common_free_temporary_object(&((EveryObject*)object)->common.id);

    assert(verify == object);
}


