/*
--          This file is part of the New World OS and Objectify projects
--   Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2011-02-20 21:23:53 -0700 (Sun, 20 Feb 2011) $
--   $Revision: 4870 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the objectify.c file which this
--   file was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <limits.h>                    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bit_map.h"
#include "chunk_info.h"                /* define nwos_hash_ref */
#include "class_definition.h"
#include "crc32.h"
#include "disk_io.h"
#include "log.h"
#include "reference_list.h"
#include "security.h"


#undef VERIFY_LAST_BLOCK_POINTER   /* disable tests that verify the last block pointer is correct */


#define MAX_REFS_IN_REF_LIST (((FILE_BLOCK_SIZE - sizeof(CommonHeader)) / sizeof(ObjRef)) - 1)
#define MAX_REFS_IN_SIDECAR  (((FILE_BLOCK_SIZE - 12) / sizeof(ObjRef)) - 1)

typedef struct ref_list_extra_block {
    struct ref_list_extra_block* next_block_ptr;
    int dirty;
    ObjRef id;
    uint8 checksum[4];
    ObjRef refs[MAX_REFS_IN_SIDECAR];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_Extra_Block;

typedef struct {
    struct ref_list_extra_block* next_block_ptr;
    ReferenceList list;
    ObjRef refs[MAX_REFS_IN_REF_LIST];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_First_Block;


#define REF_LIST_CACHE_SIZE 64

typedef struct {
  ObjRef ref;
  int num_refs;
  Ref_List_First_Block* first_block;
  Ref_List_Extra_Block* last_block;        /* null if only one block in list */
  Sorted_Reference_List*  sorted_list;
  int age;
} Block_Cache;


static int ref_list_tick = 0;

static Block_Cache ref_list_cache[REF_LIST_CACHE_SIZE];


static bool ref_list_checksum_matches(uint8* data, size_t data_size, uint8 stored[4])
{
    uint8 computed[4];

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)data, data_size, computed);

    return stored[0] == computed[0] && stored[1] == computed[1] && stored[2] == computed[2] && stored[3] == computed[3];
}


static void read_ref_list_into_cache(Block_Cache* cache)
{
    uint8 ivec[IVEC_SIZE];
    Ref_List_Extra_Block* prev_ptr;            /* points to the previous block */
    Ref_List_Extra_Block* next_ptr;            /* points to the next block */
    int i;
    int seq;
    bool use_old_decryption = false;
    int save_num_refs;

    assert(sizeof(Ref_List_First_Block) == FILE_BLOCK_SIZE + sizeof(void*) + IVEC_SIZE);
    assert(sizeof(Ref_List_Extra_Block) == FILE_BLOCK_SIZE + sizeof(void*) + IVEC_SIZE);

    cache->first_block = nwos_malloc(sizeof(Ref_List_First_Block));
    assert(cache->first_block != NULL);

    memset(ivec, 0, sizeof(ivec));
    memcpy(cache->first_block->ivec, ivec, sizeof(cache->first_block->ivec));

    cache->first_block->next_block_ptr = NULL;

    cache->last_block = NULL;
    cache->sorted_list = NULL;

    while (1)
    {
	memset(ivec, 0, sizeof(ivec));

	if (use_old_decryption)
	{
	    assert(nwos_read_block_from_disk_and_old_decrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0));
	}
	else
	{
	    assert(nwos_read_block_from_disk_and_decrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0));
	}

	if (nwos_reference_type(&cache->ref) == Public_Reference || is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref))
	{
	    cache->num_refs = 0;

	    while (cache->num_refs < MAX_REFS_IN_REF_LIST)
	    {
		if (is_void_reference(&cache->first_block->refs[cache->num_refs])) break;
		cache->num_refs++;
	    }

	    if (ref_list_checksum_matches((uint8*) cache->first_block->refs, (cache->num_refs + 1) * sizeof(ObjRef), cache->first_block->list.common_header.data_chksum))
	    {
		break;   /* found correct decryption, exit loop */
	    }
	}

	if (nwos_reference_type(&cache->ref) == Public_Reference || use_old_decryption)
	{
	    printf("\n");
	    fflush(stdout);
	    fprintf(stderr, "Unable to decrypt reference list: %08x\n", nwos_ref_to_word(&cache->ref));
	    exit(1);
	}
	else
	{
	    use_old_decryption = true;
	}
    }


    /* use the flags as a pointer to the next block */
    prev_ptr = (Ref_List_Extra_Block*) cache->first_block;

    if (cache->num_refs < MAX_REFS_IN_REF_LIST)
    {
	prev_ptr->next_block_ref.word = 0;
    }

    assert(prev_ptr->next_block_ptr == NULL);

    seq = 1;    /* use different sequence tables for each block in turn */

    while (!is_void_reference(&prev_ptr->next_block_ref))    /* more blocks in this list */
    {
	next_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	assert(next_ptr != NULL);

	next_ptr->next_block_ptr = NULL;

	memcpy(next_ptr->ivec, ivec, sizeof(next_ptr->ivec));  /* save this ivec in case this block has to be written back to disk */

	save_num_refs = cache->num_refs;

	while (1)
	{
	    memcpy(ivec, next_ptr->ivec, sizeof(ivec));  /* get the saved ivec back */

	    if (use_old_decryption)
	    {
		assert(nwos_read_block_from_disk_and_old_decrypt(&prev_ptr->next_block_ref, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq));
	    }
	    else
	    {
		assert(nwos_read_block_from_disk_and_decrypt(&prev_ptr->next_block_ref, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq));
	    }

	    cache->num_refs = save_num_refs;

	    for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
	    {
		if (is_void_reference(&next_ptr->refs[i])) break;
		cache->num_refs++;
	    }

	    if (ref_list_checksum_matches((uint8*) next_ptr->refs, (i + 1) * sizeof(ObjRef), next_ptr->checksum))
	    {
		break;
	    }

	    if (nwos_reference_type(&cache->ref) == Public_Reference || !use_old_decryption)
	    {
		printf("\n");
		fflush(stdout);
		fprintf(stderr, "Unable to decrypt reference list: %08x  seq: %d\n", nwos_ref_to_word(&cache->ref), seq);
		nwos_terminate_objectify();
		exit(1);
	    }

	    use_old_decryption = false;   /* could have encountered a new one */
	}

	if (i < MAX_REFS_IN_SIDECAR)    /* the link will be garbage, make it void */
	{
	    next_ptr->next_block_ref.word = 0;
	}
	else
	{
	    assert(!is_same_object(&prev_ptr->next_block_ref, &next_ptr->next_block_ref));
	}

	prev_ptr->next_block_ptr = next_ptr;

	prev_ptr = next_ptr;

	cache->last_block = prev_ptr;        /* save it in cache in case this is the last block */

	seq++;
    }
}



static void write_reference_list(Block_Cache* cache)
{
    uint8 ivec[IVEC_SIZE];
    Ref_List_Extra_Block *block = NULL;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* prev_ptr = NULL;
    Ref_List_Extra_Block* next_ptr = NULL;
    ObjRef ref;
    int i;
    int seq;
    bool encrypted;

    encrypted = (nwos_reference_type(&cache->ref) == Private_Reference);

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	if (cache->first_block->list.common_header.flags == 0xffffffff)    /* block is dirty, needs to be written to disk */
	{
	    for (i = cache->num_refs + 1; i < MAX_REFS_IN_REF_LIST; i++)
	    {
		if (encrypted)
		{
		    cache->first_block->refs[i].word = (random() << 1) ^ (random() >> 1);
		}
		else
		{
		    cache->first_block->refs[i].word = 0;
		}
	    }

	    if (encrypted && cache->num_refs < MAX_REFS_IN_REF_LIST)
	    {
		cache->first_block->next_block_ref.word = (random() << 1) ^ (random() >> 1);
	    }
	    else
	    {
		cache->first_block->next_block_ref.word = 0;
	    }

	    nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				 (cache->num_refs + 1) * sizeof(ObjRef),               /* include void ptr or next_block_ref */
				 cache->first_block->list.common_header.data_chksum);

	    block = (Ref_List_Extra_Block*)cache->first_block;     /* flag that it needs to be written */

	    seq = 0;
	}
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);  /* the next block pointer should not be null */

	running_total = MAX_REFS_IN_REF_LIST;
	prev_ptr = (Ref_List_Extra_Block*)cache->first_block;

	seq = 0;

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    next_ptr = prev_ptr->next_block_ptr;

	    assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));

		if (next_ptr->dirty == -1)    /* block is dirty, needs to be written to disk */
		{

		    for (i = refs_remaining + 1; i < MAX_REFS_IN_SIDECAR; i++)
		    {
			if (encrypted)
			{
			    next_ptr->refs[i].word = (random() << 1) ^ (random() >> 1);
			}
			else
			{
			    next_ptr->refs[i].word = 0;
			}
		    }

		    if (encrypted && refs_remaining < MAX_REFS_IN_SIDECAR)
		    {
			next_ptr->next_block_ref.word = (random() << 1) ^ (random() >> 1);    /* fill next with random junk */
		    }
		    else
		    {
			next_ptr->next_block_ref.word = 0;    /* mark the end */
		    }

		    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
					 (refs_remaining + 1) * sizeof(ObjRef),        /* include void ptr or next_block_ref */
					 next_ptr->checksum);

		    block = next_ptr;     /* flag that it needs to be written */
		}

		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr != NULL);
		assert(next_ptr->next_block_ptr != NULL); /* the next block pointer should not be null */

		running_total += MAX_REFS_IN_SIDECAR;
	    }
	    
	    prev_ptr = next_ptr;

	    seq++;
	}

	assert(running_total == cache->num_refs);
	assert(cache->last_block == prev_ptr);
    }

    if (block != NULL)    /* it needs to be written */
    {
	assert(block->dirty == 0xffffffff);

	/* write object expects these to be null right now */
	block->dirty = 0;

	copy_reference(&ref, &block->id);

	/* printf("writing block: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]); */

	memcpy(ivec, block->ivec, sizeof(ivec));  /* make a copy of the ivec so the stored one isn't overwritten */

	nwos_write_block_to_disk_and_encrypt(&ref, &block->dirty, FILE_BLOCK_SIZE, ivec, seq);
    }
}


static void free_reference_list(Block_Cache* cache)
{
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* prev_ptr = NULL;
    Ref_List_Extra_Block* next_ptr = NULL;

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);
	assert(cache->first_block->list.common_header.flags != 0xffffffff);    /* block should not be dirty */

	nwos_free(cache->first_block);
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);  /* the next block pointer should not be null */

	running_total = MAX_REFS_IN_REF_LIST;
	prev_ptr = (Ref_List_Extra_Block*)cache->first_block;

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    next_ptr = prev_ptr->next_block_ptr;

	    assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	    nwos_free(prev_ptr);   /* won't be needing this anymore */

	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));
		assert(next_ptr->dirty != -1);    /* no blocks should be dirty dirty */

		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL); /* the next block pointer should not be null */
		running_total += MAX_REFS_IN_SIDECAR;
	    }
	    
	    prev_ptr = next_ptr;
	}

	nwos_free(next_ptr);

	assert(running_total == cache->num_refs);
	assert(cache->last_block == prev_ptr);
    }

    /* just in case, nuke these */
    cache->first_block = NULL;
    cache->last_block = NULL;
    memset(&cache->ref, 0, sizeof(cache->ref));

    if (cache->sorted_list != NULL)
    {
	nwos_free(cache->sorted_list);
	cache->sorted_list = NULL;
    }
}


static Block_Cache* find_ref_list_in_cache(ObjRef* ref)
{
    int i;
    Block_Cache* result;

    assert(!is_void_reference(ref));

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (is_same_object(&ref_list_cache[i].ref, ref)) break;
    }

    if (i < REF_LIST_CACHE_SIZE)   /* found it */
    {
	result = &ref_list_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = ref_list_cache;

	for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
	{
	    if (is_void_reference(&ref_list_cache[i].ref))
	    {
		result = &ref_list_cache[i];
		break;
	    }

	    if (ref_list_cache[i].age < result->age)
	    {
		result = &ref_list_cache[i];
	    }
	}

	if (i == REF_LIST_CACHE_SIZE)    /* didn't find an empty one, write the oldest one out */
	{
	    write_reference_list(result);
	}

	memcpy(&result->ref, ref, sizeof(result->ref));

	read_ref_list_into_cache(result);
    }

    ref_list_tick++;
    result->age = ref_list_tick;

    return result;
}


#ifndef USE_PREDEFINED_STRUCTS

static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


static size_t get_file_path_object_size(void* file_path_obj)
{
    return sizeof(C_struct_File_Path) + (((C_struct_File_Path*)file_path_obj)->count);
}


static bool less_than(ObjRef* left_ref, ObjRef* right_ref)
{
    bool result = false;
    bool no_compare_function_defined_for_class = false;
    ObjRef left_class_ref;
    ObjRef right_class_ref;
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    int32 public_class;

    nwos_get_object_class(left_ref, &left_class_ref);
    nwos_get_object_class(right_ref, &right_class_ref);

    assert(is_same_object(&left_class_ref, &right_class_ref));


    if (nwos_reference_type(left_ref) == Temporary_Reference)       /* can't use left to decide */
    {
	assert(nwos_reference_type(right_ref) != Temporary_Reference);    /* can't compare two temporaries */ 

	if (nwos_reference_type(right_ref) == Public_Reference)
	{
	    public_class = nwos_ref_to_word(&right_class_ref);
	}
	else
	{
	    assert(nwos_read_variable_sized_object_from_disk(&right_class_ref, kludge, sizeof(kludge), &get_class_object_size));
	    public_class = nwos_ref_to_word(&class_def_obj_ptr->header.object.clone_of);
	}
    }
    else
    {
	if (nwos_reference_type(left_ref) == Public_Reference)
	{
	    public_class = nwos_ref_to_word(&left_class_ref);
	}
	else
	{
	    assert(nwos_read_variable_sized_object_from_disk(&left_class_ref, kludge, sizeof(kludge), &get_class_object_size));
	    public_class = nwos_ref_to_word(&class_def_obj_ptr->header.object.clone_of);
	}
    }

    switch (public_class)
    {
      case NWOS_FILE_PATH_PUBLIC_REF:
	{
	    uint8 left_obj[sizeof(C_struct_File_Path) + PATH_MAX];
	    uint8 right_obj[sizeof(C_struct_File_Path) + PATH_MAX];
	    nwos_read_variable_sized_object_from_disk(left_ref, &left_obj, sizeof(left_obj), &get_file_path_object_size);
	    nwos_read_variable_sized_object_from_disk(right_ref, &right_obj, sizeof(right_obj), &get_file_path_object_size);
	    result = nwos_file_path_less_than((C_struct_File_Path*)left_obj, (C_struct_File_Path*)right_obj);
	}
	break;

      case NWOS_FILE_PUBLIC_REF:
	{
	    C_struct_File left_obj;
	    C_struct_File right_obj;
	    nwos_read_object_from_disk(left_ref, &left_obj, sizeof(left_obj));
	    nwos_read_object_from_disk(right_ref, &right_obj, sizeof(right_obj));
	    result = nwos_file_less_than(&left_obj, &right_obj);
	}
	break;

      case NWOS_MD5SUM_PUBLIC_REF:
	{
	    C_struct_MD5sum left_obj;
	    C_struct_MD5sum right_obj;
	    nwos_read_object_from_disk(left_ref, &left_obj, sizeof(left_obj));
	    nwos_read_object_from_disk(right_ref, &right_obj, sizeof(right_obj));
	    result = nwos_md5_less_than(&left_obj, &right_obj);
	}
	break;

      case NWOS_SHA1SUM_PUBLIC_REF:
	{
	    C_struct_SHA1sum left_obj;
	    C_struct_SHA1sum right_obj;
	    nwos_read_object_from_disk(left_ref, &left_obj, sizeof(left_obj));
	    nwos_read_object_from_disk(right_ref, &right_obj, sizeof(right_obj));
	    result = nwos_sha1_less_than(&left_obj, &right_obj);
	}
	break;

      case NWOS_SHA256SUM_PUBLIC_REF:
	{
	    C_struct_SHA256sum left_obj;
	    C_struct_SHA256sum right_obj;
	    nwos_read_object_from_disk(left_ref, &left_obj, sizeof(left_obj));
	    nwos_read_object_from_disk(right_ref, &right_obj, sizeof(right_obj));
	    result = nwos_sha256_less_than(&left_obj, &right_obj);
	}
	break;

      case NWOS_SHA512SUM_PUBLIC_REF:
	{
	    C_struct_SHA512sum left_obj;
	    C_struct_SHA512sum right_obj;
	    nwos_read_object_from_disk(left_ref, &left_obj, sizeof(left_obj));
	    nwos_read_object_from_disk(right_ref, &right_obj, sizeof(right_obj));
	    result = nwos_sha512_less_than(&left_obj, &right_obj);
	}
	break;

      default:
	assert(no_compare_function_defined_for_class);
	break;
    }

    return result;
}


static void insert_reference_into_sorted_list(ObjRef* ref, Block_Cache* cache)
{
    ObjRef object_class;
    int lwr;
    int upr;
    int mid;
    int i;


    nwos_get_object_class(ref, &object_class);

    if (is_same_object(&object_class, &cache->sorted_list->class))
    {
	/* first make sure there is room for more */

	assert(cache->sorted_list->count <= cache->sorted_list->capacity);

	if (cache->sorted_list->capacity == cache->sorted_list->count)
	{
	    cache->sorted_list->capacity = cache->sorted_list->capacity + 1024;

	    cache->sorted_list = nwos_realloc(cache->sorted_list, sizeof(Sorted_Reference_List) + (cache->sorted_list->capacity * sizeof(ObjRef)));
	}

	lwr = 1;
	mid = 0;
	upr = cache->sorted_list->count;

	while (lwr <= upr)
	{
	    mid = (upr + lwr) / 2 - 1;    /* mid == zero based index */

	    if (less_than(ref, &cache->sorted_list->references[mid]))
	    {
		upr = mid;
	    }
	    else if (mid + 1 == cache->sorted_list->count)    /* greater than or equal and at the end */
	    {
		mid = mid + 1;
		break;
	    }
	    else if (less_than(ref, &cache->sorted_list->references[mid + 1]))
	    {
		mid = mid + 1;
		break;
	    }
	    else
	    {
		lwr = mid + 2;
	    }
	}

	if (cache->sorted_list->count == 0)
	{
	    assert(mid == 0);
	}
	else
	{
	    assert(0 <= mid && mid <= cache->sorted_list->count);
	}

	for (i = cache->sorted_list->count; i > mid; i--) copy_reference(&cache->sorted_list->references[i], &cache->sorted_list->references[i-1]);

	copy_reference(&cache->sorted_list->references[mid], ref);

	cache->sorted_list->count++;

#ifdef VERIFY_SORT
	for (i = 0; i < cache->sorted_list->count - 1; i++)
	{
	    assert(!less_than(&cache->sorted_list->references[i+1], &cache->sorted_list->references[i]));
	}
#endif
    }
}
#endif


static size_t reference_list_size(Block_Cache* cache)
{
    return sizeof(ReferenceList) + (cache->num_refs * sizeof(ObjRef));
}


/* WARNING: this only works on reference lists */

size_t nwos_reference_list_size(ObjRef* ref)
{
    Block_Cache* cache;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    return reference_list_size(cache);
}


static void read_reference_list_from_disk(Block_Cache* cache, ReferenceList* object, size_t size)
{
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;

    assert(size >= sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));

    /* copy the first block (which is a reference list object */

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs])); /* this should always be null */

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
	assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef));

	running_total = MAX_REFS_IN_REF_LIST;
	next_ptr = *(Ref_List_Extra_Block**)cache->first_block;        /* get pointer to first sidecar */

	assert(is_same_object(&cache->first_block->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));
		memcpy(&object->references[running_total], next_ptr->refs, refs_remaining * sizeof(ObjRef));
		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL);  /* the next block pointer should not be null */
		memcpy(&object->references[running_total], next_ptr->refs, MAX_REFS_IN_SIDECAR * sizeof(ObjRef));
		running_total += MAX_REFS_IN_SIDECAR;

		/* next pointer ref matches */
		assert(is_same_object(&next_ptr->next_block_ref, &next_ptr->next_block_ptr->id));
	    }

	    next_ptr = next_ptr->next_block_ptr;
	}
	assert(running_total == cache->num_refs);
    }
}


void nwos_read_reference_list_from_disk(ObjRef* ref, ReferenceList* object, size_t size)
{
    Block_Cache* cache;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    read_reference_list_from_disk(cache, object, size);
}


void nwos_create_reference_list_with_existing_id(ObjRef* for_obj, ObjRef* ref_list)
{
    Block_Cache* cache;
    int i;

    assert(!is_void_reference(ref_list));

    cache = ref_list_cache;

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (is_void_reference(&ref_list_cache[i].ref))
	{
	    cache = &ref_list_cache[i];
	    break;
	}

	if (ref_list_cache[i].age < cache->age)
	{
	    cache = &ref_list_cache[i];
	}
    }

    if (i == REF_LIST_CACHE_SIZE)    /* didn't find an empty one, write the oldest one out */
    {
	write_reference_list(cache);
    }

    copy_reference(&cache->ref, ref_list);

    ref_list_tick++;
    cache->age = ref_list_tick;

    cache->first_block = nwos_malloc(sizeof(Ref_List_First_Block));

    assert(cache->first_block != NULL);

    memset(cache->first_block, 0, sizeof(Ref_List_First_Block));  /* zero it out */

    cache->last_block = NULL;

    cache->num_refs = 0;

    memset(cache->first_block->ivec, 0, sizeof(cache->first_block->ivec));

    nwos_fill_in_common_header(&cache->first_block->list.common_header, &cache->ref, &nwos_reference_list_class_ref);

    cache->first_block->list.common_header.flags = 0xffffffff;  /* mark it as dirty */

    copy_reference(&cache->first_block->list.common_header.id, &cache->ref);


#if 0
    printf("sizeof: %u chksum: %02x %02x %02x %02x\n", sizeof(ReferenceList),
	   rl.common_header.chksum[0], rl.common_header.chksum[1], rl.common_header.chksum[2], rl.common_header.chksum[3]);
#endif

    assert(cache->num_refs == 0);
    assert(cache->first_block != NULL);
    assert(cache->first_block->next_block_ptr == NULL);
    assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
}


void nwos_create_reference_list(ObjRef* for_obj, ObjRef* ref_list)
{
    nwos_generate_new_id(ref_list);

    nwos_create_reference_list_with_existing_id(for_obj, ref_list);
}


void nwos_add_to_references(ObjRef* ref, ObjRef* obj)
{
    EveryObject header;
    ObjRef class_ref;

    /* first verify we are didn't pass in a reference list as the object */
    /* I.E. meant to call add_to_reference_list instead. */

    nwos_get_object_class_without_update(obj, &class_ref);
    assert(!is_same_object(&class_ref, &nwos_reference_list_class_ref));

    nwos_read_object_headers_from_disk(obj, &header);

    nwos_add_to_reference_list(ref, &header.object.references);
}



/* Verify the last_block pointer and offset are correct, only used in testing.  */

#ifdef VERIFY_LAST_BLOCK_POINTER

static void verify_last_block_pointer(Block_Cache* cache, int remaining, int calc_seq)
{
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;
    int seq;

    assert(cache->num_refs > MAX_REFS_IN_REF_LIST);    /* there is more than one block */
    assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
    assert(cache->last_block != NULL);                          /* the last block pointer should not be null */
    assert(cache->last_block->next_block_ptr == NULL);          /* but it's next block pointer should be */
    assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

    next_ptr = cache->first_block->next_block_ptr;        /* get pointer to first sidecar */
    assert(is_same_object(&cache->first_block->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

    seq = 1;

    running_total = MAX_REFS_IN_REF_LIST;

    refs_remaining = cache->num_refs - running_total;

    while (refs_remaining > MAX_REFS_IN_SIDECAR)    /* read more blocks */
    {
	assert(running_total < cache->num_refs);

	assert(next_ptr->next_block_ptr != NULL);  /* the next block pointer should not be null */

	/* next pointer ref matches */
	assert(is_same_object(&next_ptr->next_block_ref, &next_ptr->next_block_ptr->id));

	next_ptr = next_ptr->next_block_ptr;

	seq++;

	running_total += MAX_REFS_IN_SIDECAR;

	refs_remaining = cache->num_refs - running_total;
    }

    assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
    assert(is_void_reference(&next_ptr->refs[refs_remaining]));

    assert(next_ptr == cache->last_block);                         /* verify last_block is correct */
    assert(running_total == ((cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR) * MAX_REFS_IN_SIDECAR + MAX_REFS_IN_REF_LIST);

    assert(refs_remaining == remaining);
    assert(seq == calc_seq);
}
#endif


/* Add a new reference to the reference list specified (ref_list is the id of the reference list itself). */

void nwos_add_to_reference_list(ObjRef* ref, ObjRef* ref_list)
{
#ifdef LOG_REFERENCE_LIST
    char log_msg[128];
#endif
    Block_Cache* cache;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;
    int block_num;
    uint8 ivec[IVEC_SIZE];
    int seq;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
    assert(nwos_reference_type(ref_list) == Public_Reference);
#else
    assert(nwos_reference_type(ref) == Private_Reference);
    assert(nwos_reference_type(ref_list) == Private_Reference);
#endif

    cache = find_ref_list_in_cache(ref_list);

    /* first make sure it is a reference list */

    assert(is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref));

#ifdef LOG_REFERENCE_LIST
    snprintf(log_msg, sizeof(log_msg), "nwos_add_to_reference_list - object: %02x%02x%02x%02x - ref list: %02x%02x%02x%02x - num_refs: %d", 
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3],
	    ref_list->id[0], ref_list->id[1], ref_list->id[2], ref_list->id[3],
	    cache->num_refs);
    nwos_log(log_msg);
#endif

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	if (cache->num_refs < MAX_REFS_IN_REF_LIST)    /* there is room for more */
	{
	    memcpy(&cache->first_block->refs[cache->num_refs], ref, sizeof(ObjRef));
	    cache->num_refs++;
	    memset(&cache->first_block->refs[cache->num_refs], 0, sizeof(ObjRef));
	    cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */
	}
	else       /* no room for more, must expand to 2 blocks */
	{
	    assert(cache->num_refs == MAX_REFS_IN_REF_LIST);

	    nwos_generate_new_id(&cache->first_block->refs[MAX_REFS_IN_REF_LIST]);    /* id of new list */

	    memset(ivec, 0, sizeof(ivec));

	    cache->first_block->list.common_header.flags = 0;    /* remove dirty flag */

	    nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				 (cache->num_refs + 1) * sizeof(ObjRef),               /* include void ptr or next_block_ref */
				 cache->first_block->list.common_header.data_chksum);

	    nwos_write_block_to_disk_and_encrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0);

	    next_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	    assert(next_ptr != NULL);

	    cache->first_block->next_block_ptr = next_ptr;

	    next_ptr->next_block_ptr = NULL;

	    memcpy(next_ptr->ivec, ivec, sizeof(next_ptr->ivec));

#ifdef LOG_REFERENCE_LIST
	    snprintf(log_msg, sizeof(log_msg), "add_reference - first_block ref: %02x%02x%02x%02x obj: %p  next_block ref: %02x%02x%02x%02x obj: %p",
		   cache->ref.id[0], cache->ref.id[2], cache->ref.id[2], cache->ref.id[3], cache->first_block,
		   cache->first_block->next_block_ref.id[0], cache->first_block->next_block_ref.id[1], 
		   cache->first_block->next_block_ref.id[2], cache->first_block->next_block_ref.id[3], 
		   cache->first_block->next_block_ptr);
	    nwos_log(log_msg);
#endif

	    next_ptr->dirty = -1;     /* mark it as dirty */
	    memcpy(&next_ptr->id, &cache->first_block->next_block_ref, sizeof(ObjRef));
	    memcpy(&next_ptr->refs[0], ref, sizeof(ObjRef));
	    next_ptr->refs[1].word = 0;              /* void reference to mark the end of the list */

	    cache->last_block = next_ptr;

	    cache->num_refs++;
	}
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
	assert(cache->last_block != NULL);                          /* the last block pointer should not be null */
	assert(cache->last_block->next_block_ptr == NULL);          /* but it's next block pointer should be */
	assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

	next_ptr = cache->last_block;                         /* go right to the last block */

	block_num = (cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR;
	seq = block_num + 1;
	running_total = MAX_REFS_IN_REF_LIST + block_num * MAX_REFS_IN_SIDECAR;

	assert(running_total < cache->num_refs);
	assert(cache->num_refs - running_total <= MAX_REFS_IN_SIDECAR);

	refs_remaining = cache->num_refs - running_total;

	assert(refs_remaining <= MAX_REFS_IN_SIDECAR);
	assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&next_ptr->refs[refs_remaining]));

#ifdef VERIFY_LAST_BLOCK_POINTER
	verify_last_block_pointer(cache, refs_remaining, seq);     /* verify last_block is correct */
#endif

	if (refs_remaining < MAX_REFS_IN_SIDECAR)    /* there is room for more */
	{
	    memcpy(&next_ptr->refs[refs_remaining], ref, sizeof(ObjRef));
	    next_ptr->refs[refs_remaining + 1].word = 0;  /* set the next ref to null */
	    next_ptr->dirty = -1;                   /* mark it as dirty */
	}
	else       /* no room for more, must add new block */
	{
	    assert(refs_remaining == MAX_REFS_IN_SIDECAR);

	    nwos_generate_new_id(&next_ptr->next_block_ref);    /* id of new list */

	    next_ptr->dirty = 0;   /* remove dirty flag */

	    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
				 (refs_remaining + 1) * sizeof(ObjRef),        /* include void ptr or next_block_ref */
				 next_ptr->checksum);

	    memcpy(ivec, next_ptr->ivec, sizeof(ivec));

	    nwos_write_block_to_disk_and_encrypt(&next_ptr->id, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq);

	    next_ptr->next_block_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	    assert(next_ptr->next_block_ptr != NULL);

	    next_ptr->next_block_ptr->next_block_ptr = NULL;

	    memcpy(next_ptr->next_block_ptr->ivec, ivec, sizeof(next_ptr->next_block_ptr->ivec));

#ifdef LOG_REFERENCE_LIST
	    snprintf(log_msg, sizeof(log_msg), "add_reference - next_block ref: %02x%02x%02x%02x obj: %p  next_block ref: %02x%02x%02x%02x obj: %p",
		   next_ptr->id.id[0], next_ptr->id.id[2], next_ptr->id.id[2], next_ptr->id.id[3], 
		   next_ptr,
		   next_ptr->next_block_ref.id[0], next_ptr->next_block_ref.id[1], 
		   next_ptr->next_block_ref.id[2], next_ptr->next_block_ref.id[3], 
		   next_ptr->next_block_ptr);
	    nwos_log(log_msg);
#endif

	    next_ptr->next_block_ptr->dirty = -1;     /* mark it as dirty */
	    memcpy(&next_ptr->next_block_ptr->id, &next_ptr->next_block_ref, sizeof(ObjRef));
	    memcpy(&next_ptr->next_block_ptr->refs[0], ref, sizeof(ObjRef));
	    next_ptr->next_block_ptr->refs[1].word = 0;              /* void reference to mark the end of the list */

	    cache->last_block = next_ptr->next_block_ptr;
	}

	cache->num_refs++;

	assert(running_total + refs_remaining + 1 == cache->num_refs);
    }

#ifndef USE_PREDEFINED_STRUCTS
    if (cache->sorted_list != NULL)
    {
	insert_reference_into_sorted_list(ref, cache);
    }
#endif
}


#ifndef PUBLIC_MODE

void nwos_remove_from_references(ObjRef* ref, ObjRef* obj)
{
    EveryObject header;
    ObjRef class_ref;

    /* first verify we are didn't pass in a reference list as the object */
    /* I.E. meant to call remove_from_reference_list instead. */

    nwos_get_object_class(obj, &class_ref);
    assert(!is_same_object(&class_ref, &nwos_reference_list_class_ref));

    nwos_read_object_headers_from_disk(obj, &header);

    nwos_remove_from_reference_list(ref, &header.object.references);
}



/*-------------------------------------------------------------------------------------------------------
 Remove a reference from the reference list specified (ref_list is the id of the reference list itself).
 
  Since the first block of a reference list is different than the subsequent blocks,
  there are some special cases which arise:

    1. There is only one block.
    2. There are two blocks, but the second block only has 1 entry
    3. There are multiple blocks but the entry to be deleted is in the first.
    4. There are two blocks, the entry is in the first, and the second only has one entry.

  After we are past the first blocker there are four cases:

    5. The entry is in the last block, no other blocks need to be modified.
    6. The entry is the only entry in the last block, the last block needs to be eliminated.
    7. The entry is not in the last block, blocks from the entry to the end need to be written.
    8. The entry is not in the last, and the last block has only one entry and needs to be removed.
-------------------------------------------------------------------------------------------------------*/

void nwos_remove_from_reference_list(ObjRef* ref, ObjRef* ref_list)
{
    char log_msg[128];
    Block_Cache* cache;
    Ref_List_Extra_Block* next_ptr;
    Ref_List_Extra_Block* prev_ptr = NULL;
    int count;
    int i;
    int seq;
    uint8 ivec[IVEC_SIZE];

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
    assert(nwos_reference_type(ref_list) == Public_Reference);
#else
    assert(nwos_reference_type(ref) == Private_Reference);
    assert(nwos_reference_type(ref_list) == Private_Reference);
#endif

    cache = find_ref_list_in_cache(ref_list);

    /* first make sure it is a reference list */

    assert(is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref));

    snprintf(log_msg, sizeof(log_msg), "nwos_remove_from_reference_list - object: %02x%02x%02x%02x - ref list: %02x%02x%02x%02x - num_refs: %d", 
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3],
	    ref_list->id[0], ref_list->id[1], ref_list->id[2], ref_list->id[3],
	    cache->num_refs);
    nwos_log(log_msg);

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* case #1 - there is only one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	cache->num_refs--;

	if (!is_same_object(ref, &cache->first_block->refs[cache->num_refs]))    /* not the last one in the list */
	{
	    for (i = 0; i < cache->num_refs; i++)
	    {
		if (is_same_object(ref, &cache->first_block->refs[i])) break;
	    }
		
	    assert(i < cache->num_refs);    /* make sure it was in the list */

	    while ((i + 1) < MAX_REFS_IN_REF_LIST && !is_void_reference(&cache->first_block->refs[i+1]))
	    {
		copy_reference(&cache->first_block->refs[i], &cache->first_block->refs[i+1]);
		i++;
	    }

	    assert(i == cache->num_refs);
	}

	void_reference(&cache->first_block->refs[cache->num_refs]);
	cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */
    }
    else   /* cases 2 through 8 - more than one block */
    {
	assert(!is_void_reference(&cache->first_block->next_block_ref));
	assert(cache->last_block != NULL);

	for (i = 0; i < MAX_REFS_IN_REF_LIST; i++)
	{
	    if (is_same_object(ref, &cache->first_block->refs[i])) break;
	}
		
	next_ptr = cache->first_block->next_block_ptr;
	assert(next_ptr != NULL);         /* the next block pointer should not be null */

	count = i;

	if (i < MAX_REFS_IN_REF_LIST)  /* case 2 or 3 - is in the first and there are blocks after it */
	{
	    assert(is_same_object(ref, &cache->first_block->refs[i]));

	    while (i + 1 < MAX_REFS_IN_REF_LIST)
	    {
		copy_reference(&cache->first_block->refs[i], &cache->first_block->refs[i + 1]);

		count++;
		i++;
	    }

	    assert (i == MAX_REFS_IN_REF_LIST - 1);

	    copy_reference(&cache->first_block->refs[i], &next_ptr->refs[0]);
	    count++;

	    if (cache->num_refs == (MAX_REFS_IN_REF_LIST + 1)) /* case 2 - don't need extra block anymore */
	    {
		assert(next_ptr == cache->last_block);

		nwos_clear_bit_in_map(nwos_hash_ref(&cache->first_block->next_block_ref));
		void_reference(&cache->first_block->next_block_ref);

		cache->first_block->next_block_ptr = NULL;
		cache->last_block = NULL;

		nwos_free(next_ptr);
		next_ptr = NULL;    /* flag that we don't need to move anymore */

		cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */

		/* should not need to change ivec for first block, should always be zero */
	    }
	    else   /* case 3 - is in first block, and we need to deal with following blocks */
	    {
		nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				     (MAX_REFS_IN_REF_LIST + 1) * sizeof(ObjRef),  /* include next_block_ref */
				     cache->first_block->list.common_header.data_chksum);

		memcpy(ivec, cache->first_block->ivec, sizeof(ivec));  /* should be all zeros */
		nwos_write_block_to_disk_and_encrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0);

		memcpy(next_ptr->ivec, ivec, sizeof(ivec));

		i = 0;
		seq = 1;
	    }
	}
	else if (cache->num_refs == (MAX_REFS_IN_REF_LIST + 1))  /* case 4, two blocks with 1 entry in 2nd */
	{
	    assert(next_ptr == cache->last_block);
	    assert(is_same_object(ref, &next_ptr->refs[0]));
	    assert(is_void_reference(&next_ptr->refs[1]));

	    nwos_clear_bit_in_map(nwos_hash_ref(&cache->first_block->next_block_ref));
	    void_reference(&cache->first_block->next_block_ref);

	    cache->first_block->next_block_ptr = NULL;
	    cache->last_block = NULL;

	    nwos_free(next_ptr);
	    next_ptr = NULL;    /* flag that we don't need to move anymore */

	    cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */

	    /* should not need to change ivec for first block, should always be zero */
	}
	else   /* cases 5 through 8 */
	{
	    seq = 1;

	    while (true)
	    {
		assert (next_ptr != NULL);

		for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
		{
		    assert(!is_void_reference(&next_ptr->refs[i]));

		    if (is_same_object(ref, &next_ptr->refs[i])) break;

		    count++;
		}

		if (i < MAX_REFS_IN_SIDECAR) break;

		prev_ptr = next_ptr;

		next_ptr = next_ptr->next_block_ptr;

		seq++;
	    }
	}	

	if (next_ptr != NULL)   /* cases 3 and 5-8 */
	{
	    assert(i < MAX_REFS_IN_SIDECAR);

	    if (i == 0 && is_void_reference(&next_ptr->refs[1]))  /* case 6 - only entry in this block */
	    {
		assert(next_ptr == cache->last_block);
		assert(is_same_object(ref, &next_ptr->refs[0]));
		assert(prev_ptr != NULL);
		assert(prev_ptr->next_block_ptr == next_ptr);
		assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));

		nwos_clear_bit_in_map(nwos_hash_ref(&next_ptr->id));
		void_reference(&prev_ptr->next_block_ref);

		prev_ptr->next_block_ptr = NULL;

		nwos_free(next_ptr);

		prev_ptr->dirty = -1;

		cache->last_block = prev_ptr;
	    }
	    else if (next_ptr->next_block_ptr == NULL)   /* case 5 - just need to remove it from the last block */
	    {
		assert(next_ptr == cache->last_block);

		while (!is_void_reference(&next_ptr->refs[i]))
		{
		    copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

		    count++;
		    i++;
		}

		next_ptr->dirty = -1;     /* mark it as dirty */
	    }
	    else   /* case 7 or 8 */
	    {
		while (next_ptr->next_block_ptr != NULL)
		{
		    while ((i + 1) < MAX_REFS_IN_SIDECAR)
		    {
			copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

			count++;
			i++;
		    }

		    copy_reference(&next_ptr->refs[i], &next_ptr->next_block_ptr->refs[0]);

		    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
					 (MAX_REFS_IN_SIDECAR + 1) * sizeof(ObjRef),  /* include next_block_ref */
					 next_ptr->checksum);

		    memcpy(ivec, next_ptr->ivec, sizeof(ivec));
		    nwos_write_block_to_disk_and_encrypt(&next_ptr->id, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq);

		    memcpy(next_ptr->next_block_ptr->ivec, ivec, sizeof(ivec));

		    prev_ptr = next_ptr;

		    next_ptr = next_ptr->next_block_ptr;

		    seq++;

		    i = 0;
		}

		assert(prev_ptr->next_block_ptr == next_ptr);
		assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));
		assert(next_ptr == cache->last_block);

		if (!is_void_reference(&next_ptr->refs[0]) && is_void_reference(&next_ptr->refs[1])) /* case 8 - only one entry in last block */
		{
		    assert(prev_ptr != NULL);

		    nwos_clear_bit_in_map(nwos_hash_ref(&prev_ptr->next_block_ref));
		    void_reference(&prev_ptr->next_block_ref);

		    nwos_free(prev_ptr->next_block_ptr);
		    prev_ptr->next_block_ptr = NULL;

		    prev_ptr->dirty = -1;     /* mark it as dirty */

		    cache->last_block = prev_ptr;
		}
		else  /* case 7 - just need to finish last block */
		{
		    assert(!is_void_reference(&next_ptr->refs[i + 1]));

		    while (!is_void_reference(&next_ptr->refs[i + 1]))
		    {
			copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

			count++;
			i++;
		    }

		    void_reference(&next_ptr->refs[i]);

		    next_ptr->dirty = -1;
		}
	    }

	    next_ptr = NULL;

#ifdef VERIFY_LAST_BLOCK_POINTER
	    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)
	    {
		assert(cache->last_block == NULL);
	    }
	    else
	    {
		int block_num;
		int running_total;
		int refs_remaining;

		assert(cache->last_block != NULL);

		block_num = (cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR;
		seq = block_num + 1;
		running_total = MAX_REFS_IN_REF_LIST + block_num * MAX_REFS_IN_SIDECAR;

		refs_remaining = cache->num_refs - running_total;

		verify_last_block_pointer(cache, refs_remaining, seq);     /* verify we didn't screw up the last_block pointer */
	    }
#endif
	}

	assert(next_ptr == NULL);

	printf("num_refs: %d  count: %d\n", cache->num_refs, count);

	cache->num_refs--;
    }
}
#endif


/* flush any reference lists that are dirty out to disk */

void nwos_flush_dirty_ref_lists()
{
    int i;

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (!is_void_reference(&ref_list_cache[i].ref))
	{
	    write_reference_list(&ref_list_cache[i]);
	}
    }
}


void nwos_free_ref_lists()
{
    int i;

    nwos_flush_dirty_ref_lists();

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (!is_void_reference(&ref_list_cache[i].ref))
	{
	    free_reference_list(&ref_list_cache[i]);
	}
    }
}


ReferenceList* nwos_malloc_reference_list(ObjRef* ref)
{
    size_t ref_list_size;
    ReferenceList* ref_list;

    ref_list_size = nwos_reference_list_size(ref);

    ref_list = nwos_malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(ref, ref_list, ref_list_size);

    ref_list->common_header.num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    return ref_list;
}


/* This creates a reference list of all objects of a class including both public and private */

ReferenceList* nwos_malloc_class_reference_list(char* class_name)
{
    size_t ref_list_size;
    ReferenceList* ref_list;
    ReferenceList* public_ref_list;
    ReferenceList* private_ref_list = NULL;
    C_struct_Class_Definition class_def;
    ObjRef public_ref;
    ObjRef private_ref;
    ObjRef obj_class_ref;
    int i;
    int num_refs = 0;

    assert(nwos_find_public_class_definition(class_name, &public_ref));
    nwos_read_class_definition(&public_ref, &class_def);
    public_ref_list = nwos_malloc_reference_list(&class_def.header.object.references);
    ref_list_size = public_ref_list->common_header.num_refs;

    if (nwos_find_private_class_definition(class_name, &private_ref))
    {
	nwos_read_class_definition(&private_ref, &class_def);
	private_ref_list = nwos_malloc_reference_list(&class_def.header.object.references);
	ref_list_size += private_ref_list->common_header.num_refs;
    }

    assert(ref_list_size > 0);

    ref_list = nwos_malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    if (private_ref_list != NULL)
    {
	for (i = 0; i < private_ref_list->common_header.num_refs; i++)
	{
	    nwos_get_object_class(&private_ref_list->references[i], &obj_class_ref);   /* find out what kind of object it is */

	    if (is_same_object(&obj_class_ref, &nwos_private_class_definition_class_ref))   /* it is a class definition object */
	    {
		copy_reference(&ref_list->references[num_refs], &private_ref_list->references[i]);
		num_refs++;
	    }
	}

	nwos_free_reference_list(private_ref_list);
	private_ref_list = NULL;
    }

    for (i = 0; i < public_ref_list->common_header.num_refs; i++)
    {
	nwos_get_object_class(&public_ref_list->references[i], &obj_class_ref);   /* find out what kind of object it is */

	if (is_same_object(&obj_class_ref, &nwos_public_class_definition_class_ref))   /* it is a class definition object */
	{
	    copy_reference(&ref_list->references[num_refs], &public_ref_list->references[i]);
	    num_refs++;
	}
    }

    nwos_free_reference_list(public_ref_list);
    public_ref_list = NULL;

    ref_list->common_header.num_refs = num_refs;

    return ref_list;
}


void nwos_free_reference_list(ReferenceList* ref_list)
{
    nwos_free(ref_list);
}


#ifndef USE_PREDEFINED_STRUCTS

Sorted_Reference_List* nwos_get_sorted_reference_list(ObjRef* list_ref, ObjRef* class_ref)
{
    Block_Cache* cache;
    ReferenceList* ref_list;
    int i;

    assert(!is_void_reference(list_ref));

    cache = find_ref_list_in_cache(list_ref);

    if (cache->sorted_list == NULL || !is_same_object(&cache->sorted_list->class, class_ref))
    {
	if (cache->sorted_list == NULL)
	{
	    cache->sorted_list = nwos_malloc(sizeof(Sorted_Reference_List) + (cache->num_refs * sizeof(ObjRef)));
	}

	copy_reference(&cache->sorted_list->class, class_ref);

	cache->sorted_list->capacity = cache->num_refs;
	cache->sorted_list->count = 0;

	ref_list = nwos_malloc_reference_list(list_ref);

	for (i = 0; i < ref_list->common_header.num_refs; i++)
	{
	    insert_reference_into_sorted_list(&ref_list->references[i], cache);
	}

	nwos_free_reference_list(ref_list);
    }

    return cache->sorted_list;
}


int nwos_find_object_in_sorted_reference_list(Sorted_Reference_List* list, ObjRef* temp_ref)
{
    int lwr;
    int upr;
    int mid = 0;

    lwr = 1;

    upr = list->count;

    while (lwr <= upr)
    {
	mid = (upr + lwr) / 2;

	if (less_than(temp_ref, &list->references[mid - 1]))
	{
	    upr = mid - 1;
	}
	else if (less_than(&list->references[mid - 1], temp_ref))
	{
	    lwr = mid + 1;
	}
	else
	{
	    mid = mid - 1;  /* adjust to 0 based index */
	    break;
	}
    }

    if (lwr <= upr)
    {
	assert(0 <= mid && mid < list->count);
	assert(!less_than(temp_ref, &list->references[mid]) && !less_than(&list->references[mid], temp_ref));

	while (mid > 0 && !less_than(&list->references[mid-1], temp_ref)) mid--;    /* find the first one if more than one */

	return mid;
    }

    return -1;   /* not found */
}


#endif


