#!/usr/bin/perl -w
#
# Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


=head1 NAME

offmirror-Packages - convert Debian 'Packages' files to offmirror list

=head1 SYNOPSIS

offmirror-Packages file ... | -

=head1 DESCRIPTION

This program parses Debian 'Packages' files and output to stdout
in offmirrror list form.

=head1 OPTIONS

=over 4

=item file ...

List of 'Packages' files to convert or '-' for stdin

=back

=head1 SEE ALSO

L<offmirror(1)>

L<offmirror-Sources(1)>

=cut


require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006
use integer;

use FindBin;
use lib "$FindBin::Bin/../lib";
use OffMirror::FileAttrRecord;


my $progname = 'offmirror-Packages';

sub usage {
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname file ... | -

For details, see man page.
EOF
	exit(1);
}


usage("no Packages files specified") if scalar(@ARGV) == 0;

my @package_files = @ARGV;

local $/="\n\n";

my ($filename, $size, $md5sum);

foreach my $file (@package_files)
{
    local *IN;

    if ($file ne '-')
    {
	$file =~ s/^(.*\.(gz|z|Z|zip))$/gzip -dc < $1|/
	    or
	$file =~ s/^(.*\.(bz2|bz))$/bzip2 -dc < $1|/
	    or
	$file =~ s/^(.*)$/< $1/;

	open IN, "$file"
	    or die "Can't open file $file: $!";
    }
    else
    {
	open IN, "<&STDIN"
	    or die die "Can't redirect stdin: $!";
    }

    while (<IN>)
    {
	($filename)=m/^Filename:\s+(.*)/im;

	#($architecture)=m/^Architecture:\s+(.*)/im;
	#	next if (!$arches{$architecture});

	($size)=m/^Size:\s+(\d+)/im;

	($md5sum)=m/^MD5sum:\s+([A-Za-z0-9]+)/im;

	die "error parsing $file"
	    unless $filename and $size and $md5sum;

	my $r = OffMirror::FileAttrRecord::make_record_from(
				$size,
				$md5sum,
				$filename);

	print ( $r->make_record_line($filename, 'l', 1) . "\n" );
    }
}
