#!/usr/bin/perl -w
#
# Copyright (C) 2003,2004,2006 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA


=head1 NAME

offmirror-test - test offline files tree mirroring utility

=head1 SYNOPSIS

offmirror-test basedir

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program creates and populates source and destination test directories
under the basedir and runs the offmirror programs to test.

=head1 SEE ALSO

L<offmirror(1)>

=cut

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use File::Basename;

use FindBin;
use lib "$FindBin::Bin/../lib";
use OffMirror::ChangeAttr;
use OffMirror::FileList;


my $progname = 'offmirror-test';

sub usage
{
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: [fakeroot] $progname basedir

For details, see man page.
EOF
	exit(1);
}


my $basedir = shift @ARGV or  usage("no basedir specified");
              shift @ARGV and usage("extra parameter specified");

$basedir =~ s|(.*)/$|$1|;

my @cleanlist = ();

my $srcdir = $basedir . '/src'; push @cleanlist, $srcdir;
my $dstdir = $basedir . '/dst'; push @cleanlist, $dstdir;

my $srclist = $srcdir.'.offlist'; push @cleanlist, $srclist;
my $dstlist = $dstdir.'.offlist'; push @cleanlist, $dstlist;

my $tarfile = $basedir . '/src-dst.tar.gz'; push @cleanlist, $tarfile;

my $dstverlist     = "$dstlist.v"     ; push @cleanlist, $dstverlist;
my $dstverdifflist = "$dstlist.v.diff"; push @cleanlist, $dstverdifflist;


umask 0; # to allow to create files/dirs with any permissions


sub my_mkdir($$)
{
    my ($dir, $mode) = @_;

    mkdir($dir, $mode)
	or die "can't mkdir $dir: $!";
}

sub my_symlink($$)
{
    my ($old, $new) = @_;

    symlink($old, $new)
	or die "can't symlink $old, $new: $!";
}

sub my_system($@)
{
    system(@_) == 0
	or die "can't run command: @_: $?";
}


sub my_clean_up()
{
    my_system("rm -rf @cleanlist");
}


$SIG{INT} = $SIG{TERM} = $SIG{__DIE__} = \&my_clean_up;


my_clean_up();


my_mkdir($srcdir, 0755);
my_mkdir($dstdir, 0755);

my_mkdir("$srcdir/copy-me", 0700);
my_mkdir("$srcdir/copy-me/1", 0700);
my_mkdir("$srcdir/copy-me/1/2", 0700);
my_mkdir("$srcdir/copy-me/1/2/3", 0700);

my $badname = "$srcdir/";
for (my $i=1; $i < ord('/'); $i++)
{
    $badname .= chr($i);
}

local *FH;
open  (FH, '> '. $badname)
    or die "can't open badname: $!";
print FH '1234567890'
    or die "can't print to badname: $!";
close FH
    or die "can't close badname: $!";


my_system("echo 6543210 >$srcdir/copy-me/file");
my_system("echo 6543210 >$srcdir/copy-me/1/file");
my_system("echo 6543210 >$srcdir/copy-me/1/2/file");
my_system("echo 6543210 >$srcdir/copy-me/1/2/3/file");

my_symlink("../file", "$srcdir/copy-me/1/2/3/link");


my_mkdir("$dstdir/delete-me", 0700);
my_mkdir("$dstdir/delete-me/1", 0700);
my_mkdir("$dstdir/delete-me/1/2", 0700);
my_mkdir("$dstdir/delete-me/1/2/3", 0700);

my_system("echo 6543210 >$dstdir/delete-me/1/file");
my_system("echo 6543210 >$dstdir/delete-me/1/2/file");
my_system("echo 6543210 >$dstdir/delete-me/1/2/3/file");

my_system("echo 12345 >$srcdir/file"); my_utime("$srcdir/file", 10000000);
my_system("echo 12365 >$dstdir/file"); my_utime("$dstdir/file", 10000000);

my_system("mknod $srcdir/fifo p -m 07700");

$> == 0 and my_system("mknod $srcdir/chrdev c 11 11 -m 07700");
$> == 0 and my_system("mknod $srcdir/blkdev b 11 33 -m 07700");


my_system("$FindBin::Bin/offmirror-list $srcdir --pretty >$srclist");
my_system("$FindBin::Bin/offmirror-list $dstdir --pretty >$dstlist");

my_system("$FindBin::Bin/offmirror-tar   $srclist $dstlist $tarfile");
my_system("$FindBin::Bin/offmirror-untar $srclist $dstlist $tarfile");

#+ verify
my_system("$FindBin::Bin/offmirror-list $dstdir --pretty >$dstverlist");
my_system("$FindBin::Bin/offmirror-diff $srclist $dstverlist --pretty >$dstverdifflist");

my $verlist = OffMirror::FileList::list_from_file($dstverdifflist);
my $items = keys %{$verlist->{'list'}};

if ( $items != 0 )
{
    print STDERR "$items file entries are different, test failed\n";
    exit 1;
}
#- verify

my_clean_up();


exit 0;


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 COPYRIGHT

Copyright (C) 2003,2004,2006 Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut
