#!/usr/bin/perl -w
#
# Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA


=head1 NAME

offmirror-tar - archive files on source host needed to copy to destination.

=head1 SYNOPSIS

offmirror-tar src-list dst-list tar-file [options]

=head1 DESCRIPTION

This program is part of files tree offline mirroring utility package,
which is used to synchronize file trees without any direct connection
between them.

This program produce tar archive of source-side files needed to copy
to destination mirror.

=cut

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use Getopt::Long;

use File::Basename;
use Cwd 'abs_path';

use FindBin;
use lib "$FindBin::Bin/../lib";
use OffMirror::FileList;

my $progname = 'offmirror-tar';

sub usage
{
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname src-list dst-list tar-file		\
			[-tar-program=/usr/gnu/bin/tar]	\
			[-show-cmd] [-ignore-tar-errors] [-verbose] [-help]

For details, see man page.
EOF
	exit(1);
}


=head1 OPTIONS

=over 4

=item src-list[.gz|.bz2|.Z|.zip]

Specifies file with source files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item dst-list[.gz|.bz2|.Z|.zip]

Specifies file with destination files tree list.

The file can be compressed with gzip or bzip2 or compress or
zip (single file only).

=item tar-file.tar[.gz|.bz2]

Specifies file name for the tar archive.
Name can have suffix '.gz' or '.bz2'
which implied proper tar compression option.

=item -tar-program=/usr/gnu/bin/tar

Specifies tar program to run.

=item -show-cmd

Show running command line.

=item -ignore-tar-errors

Ignore tar errors.
Workaround for cygwin' tar bug which always returns 1.

=item -verbose

List to STDERR files processed

=item -help

Display a usage summary.

=back

=head1 SEE ALSO

L<offmirror(1)>

L<offmirror-list(1)>

L<offmirror-untar(1)>

=cut


my $tar_prog='tar';
my $tar_opts='';
my $show_cmd;
my $verbose;
my $ignore_tar_errors = undef;
my $help;

GetOptions(
	'tar-program=s'	=> \$tar_prog,
	'show-cmd'	=> \$show_cmd,
	'ignore-tar-errors' => \$ignore_tar_errors,
	'verbose'	=> \$verbose,
	'help'		=> \$help
) or usage;

usage if $help;

my $src_list = shift @ARGV or  usage("no src-list file specified");
my $dst_list = shift @ARGV or  usage("no dst-list file specified");
my $tar_file = shift @ARGV or  usage("no tar-file file specified");
               shift @ARGV and usage("extra parameter specified");

my $tar_zopt='';
if    ( $tar_file =~ m/gz$/ )
{
    $tar_zopt = '--gzip';
}
elsif ( $tar_file =~ m/bz2$/ )
{
    $tar_zopt = '--bzip2';
}
elsif ( $tar_file =~ m/\.tar$/ )
{
    $tar_zopt = '';
}
else
{
    die "unknown tar file suffix: $tar_file";
}

$tar_opts .= '--verbose' if $verbose;


my $basedir = OffMirror::FileList::basedir_from_file($src_list);

$tar_file = abs_path(dirname($tar_file)) . '/' . basename($tar_file);


sub clean
{
    if ( length($tar_file) > 0 )
    {
	print STDERR "cleaning $tar_file ..."; $|=1;
	my $rc = system("rm -rf $tar_file");
	print STDERR (($rc==0) ? 'ok' : $!). "\n";
    }
    exit 1;
}

$SIG{INT} = $SIG{TERM} = $SIG{__DIE__} = \&clean;


my $   diffcmd = "$FindBin::Bin/offmirror-diff $src_list $dst_list";
my $    tarcmd = "(cd $basedir && ".
		 "$tar_prog -cf $tar_file -b1 $tar_zopt $tar_opts".
		 " --null -T-". # keep "--null -T-" at end of the tar cmd.
		 ")";
my $tarlistcmd = "$FindBin::Bin/offmirror-difflist2tar";

my $cmd = "$diffcmd | $tarlistcmd | $tarcmd";

print "$cmd\n" if $show_cmd;

if ( system("$cmd") != 0 )
{
    my $err = $?;
    unless ($ignore_tar_errors)
    {
	system "rm -f $tar_file";
	die "failed: $err\n";
    }
    else
    {
	warn "failed: $err\n";
    }
}

exit 0;


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 COPYRIGHT

Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut
