#
# Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA

package OffMirror::FileAttrRecord;

require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006

use integer;
use Carp;

use File::Stat::Bits;
use File::Stat::ModeString;

use OffMirror::User;
use OffMirror::Misc;


BEGIN
{
    use constant          NODIGEST => scalar 'digest';
    use constant       FIELDS_LIST => qw( action mode type major minor user group
				     size mtime digest );
    use constant   NUM_FIELDS_LIST => qw( major minor size mtime );
    use constant      ACTIONS_LIST => qw( l c d a );	# list, copy, delete, attr copy
    use constant      ACTION_CHARS => 'lcda';
    use constant  COMPARISON_CHARS => scalar 'ca=';	# copy, attr copy, are equal

    use constant   USERGROUP_RE => '[=a-zA-Z0-9_-]+';

    use constant        RECORD_FMT => scalar "%s %s %u %u %s %s %u %u %s %s";
    use constant PRETTY_RECORD_FMT => scalar "%s %s %3u %3u %-8s %-8s %11u %10u %32s %s";
    use constant        RECORD_RE  => scalar
	'^(['.ACTION_CHARS."])\\s+($MODE_STRING_RE)"
	.'\s+(\d+)\s+(\d+)\s+('.USERGROUP_RE.')\s+('.USERGROUP_RE.')\s+(\d+)\s+(\d+)\s+(\w+)\s+(.+)$';


    use vars qw($VERSION);
    $VERSION = do { my @r = (q$Revision: 1.67 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };


    #+ use MD5 || Digest::MD5
    use vars qw( $md5_module );
    my @md5_modules = qw(Digest::MD5 MD5);
    foreach my $nm ( @md5_modules )
    {
	eval "use $nm";
	if (! $@)
	{
            $md5_module = $nm;
            last;
        }
    }
    die "Cannot find any MD5 module from: @md5_modules\n" if $@;
    #warn "md5_module: $md5_module\n";
    #- use MD5 || Digest::MD5
}


# accessor
# $val = $r->field('field_name')
sub field($$)
{
    my ($this, $field) = @_;
    return $this->{'_'.$field};
}


my $strmodetype = sub ($)
{
    return substr($_[0], 0, 1);
};

# ctor
# objref = OffMirror::FileAttrRecord::stat_file( $fname [, $md5sums] )
# returns objref, '?' in file type on unsupported file type
sub stat_file($;$)
{
    my ($name, $md5sums) = @_;

    my $md5 = $md5_module->new;
    my $digest;


    my (undef, undef, $st_mode, undef, $st_uid, $st_gid, $st_rdev, $st_size,
	undef, $st_mtime) = lstat($name) or croak "Can't lstat \'$name\': $!";

    my $mode = mode_to_string($st_mode);
    my $type = &$strmodetype($mode);

    if ( $md5sums )
    {
	if ( $st_size != 0 )
	{
	    if    ($type eq '-')	# regular file
	    {
		local *HANDLE;
		open  (HANDLE, $name)
		    or croak "Can't open \'$name\': $!";

		$md5->addfile(*HANDLE);

		close(HANDLE);

		$digest = $md5->hexdigest;
	    }
	    elsif ($type eq 'l')
	    {
		my $link = readlink($name)
		    or croak "Can't read symlink \'$name\': $!";

		$md5->add($link);

		$digest = $md5->hexdigest;
	    }
	    else # other type
	    {
		$digest = NODIGEST;
	    }
	}
	else	# size == 0
	{
	    $digest = NODIGEST;
	}
    }
    else	# ! $md5sums
    {
	$digest = NODIGEST;
    }


    my ($major, $minor) = ($type eq 'c' or $type eq 'b')
	? dev_split($st_rdev) : (0,0);

    my $self =
    {
	_action => 'l',
	_mode   => $mode,
	_type   => $type,
	_major	=> $major,
	_minor	=> $minor,
	_user	=> scalar get_usr_name($st_uid),
	_group	=> scalar get_grp_name($st_gid),
	_size   => $st_size,
	_mtime  => $st_mtime,
	_digest => $digest
    };

    return bless($self, __PACKAGE__);
}


# ctor
# $objref = OffMirror::FileAttrRecord::make_record_from( $size, $md5sum, $filename )
sub make_record_from($$$)
{
    my ($size, $md5sum, $filename) = @_;

    my $self =
    {
	_action => 'l',
	_mode   => '-rw-r--r--',
	_type	=> '-',
	_major	=> 0,
	_minor	=> 0,
	_user	=> 0,
	_group	=> 0,
	_size   => $size,
	_mtime  => 0,
	_digest => $md5sum
    };

    bless($self, __PACKAGE__);
}


# ctor
# OffMirror::FileAttrRecord::parse_record_line( $line )
# returns list (objref, fname) or (undef, errortext) on parse failure
sub parse_record_line($)
{
    my $line = $_[0];

    my $record_re = RECORD_RE;
    die "invalid record format at line $., stopped"
	unless $line =~ /$record_re/o;

    my ($action,$mode,$major,$minor,$user,$group,$size,$mtime,$digest,$name) =
	split(/\s+/,$line,10);

    my $self =
    {
	_action => $action,
	_mode   => $mode,
	_type	=> &$strmodetype($mode),
	_major	=> $major,
	_minor	=> $minor,
	_user	=> decode_bad_name($user),
	_group	=> decode_bad_name($group),
	_size   => $size,
	_mtime  => $mtime,
	_digest => $digest
    };

    bless($self, __PACKAGE__);

    return ( $self, decode_bad_name($name) );
}


# ( $this, $fname, $action [,$pretty] )
# returns formatted line
sub make_record_line($$$;$)
{
    my ($this, $fname, $action, $pretty) = @_;

    $this->{_action} = $action;

    sprintf (	$pretty ? PRETTY_RECORD_FMT : RECORD_FMT,
		$this->{_action},
		$this->{_mode  },
		$this->{_major },
		$this->{_minor },
		encode_bad_name($this->{_user }),
		encode_bad_name($this->{_group}),
		$this->{_size  },
		$this->{_mtime },
		$this->{_digest},
		encode_bad_name($fname)
	    );
}


# compare with other record.
# returns flag (COMPARISON_CHARS):
#	'=' - no differences found
#	'c' - contents is differ, needs to copy entirely;
#	'a' - attributes are differ, needs to copy attributes only
sub compare_records($$)
{
    my ($this, $other) = @_;

    if ( $this->{_type} ne $other->{_type} )
    {
	return 'c';
    }

    # types are equal
    my $type = $this->{_type};

    if ( ($type eq '-' or $type eq 'l') and
	 $this->{_size} != $other->{_size} )
    {
	return 'c';
    }

    # types and sizes are equal

    if ( $this->{_digest} ne NODIGEST and $other->{_digest} ne NODIGEST )
    {	# have digest
	if ( $this->{_digest} eq $other->{_digest} )
	{
	    return 'a'
		if ( $type ne 'l' and $this->{_mtime} != $other->{_mtime} ) or
		   $this->{_mode } ne $other->{_mode } or
		   $this->{_major} != $other->{_major} or
		   $this->{_minor} != $other->{_minor} or
		   $this->{_user } ne $other->{_user } or
		   $this->{_group} ne $other->{_group};
	}
	else
	{
	    return 'c';
	}
    }
    else	# have no digest
    {
	return 'c'
	    if ($type ne 'l' and $type ne 'd') and
		$this->{_mtime} != $other->{_mtime};
	return 'a'
	    if	$this->{_mode } ne $other->{_mode } or
		$this->{_major} != $other->{_major} or
		$this->{_minor} != $other->{_minor} or
		$this->{_user } ne $other->{_user } or
		$this->{_group} ne $other->{_group};
    }

    return '=';	# all equal
}


1;


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 COPYRIGHT

Copyright (C) 2003,2004,2006,2008 Dmitry Fedorov <dm.fedorov@gmail.com>

=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut
