/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PluginModule.h"
#include <iostream>
#include <cstdlib>
#include "NodeControlWindow.h"
#include "NodeModel.h"
#include "PatchWindow.h"
#include "OmPort.h"
#include "OmPatchBayArea.h"
#include "PatchController.h"
#include "Controller.h"
#include "OmGtk.h"
#include "OmGtkApp.h"
#include "MetadataModel.h"
using std::cerr; using std::cout; using std::endl;

using namespace PatchBay;


namespace OmGtk {
	
PluginModule::PluginModule(OmPatchBayArea* patch_bay,
                           NodeModel* nm, PatchController* patch_controller)
: OmModule(patch_bay, nm, patch_controller),
  m_dialog(NULL)
{
	if (nm->plugin_info()->type() == PluginInfo::Internal
			&& nm->plugin_info()->plug_label() == "midi_control_in") {
		Gtk::Menu::MenuList& items = m_menu.items();
		items.push_back(Gtk::Menu_Helpers::MenuElem("Learn",
			sigc::mem_fun(this, &PluginModule::menu_learn)));
	}
}


PluginModule::~PluginModule()
{
	if (m_dialog != NULL) {
		m_dialog->hide();
		delete m_dialog;
	}
}


void
PluginModule::add_port(PortModel* pm, bool resize_to_fit)
{
	Port* port = new OmPort(this, m_patch_controller, pm);

	port->signal_event().connect(
		sigc::bind<Port*>(sigc::mem_fun(m_patch_bay, &OmPatchBayArea::port_clicked), port));
	
	Module::add_port(port, resize_to_fit);

	if (m_dialog != NULL)
		m_dialog->add_port(pm, true);
}


void
PluginModule::control_change(const ControlModel* const cm)
{
	//std::cerr << "[PluginModule] Control change." << std::endl;

	m_node_model->control_change(cm);
	if (m_dialog != NULL)
		m_dialog->set_control(cm->port_path(), cm->value());
}


void
PluginModule::metadata_update(const MetadataModel* const mm)
{
	if (mm->path() == m_node_model->path()) {
		if (mm->key() == "module-x") {
			float x = atof(mm->value().c_str());
			move_to(x, property_y().get_value());
		} else if (mm->key() == "module-y") {
			float y = atof(mm->value().c_str());
			move_to(property_x().get_value(), y);
		} else {
			//cerr << "[OmModule] Unknown metadata key " << mm->key() << endl;
		}
		m_node_model->set_metadata(mm->key(), mm->value());
	}
}


void
PluginModule::port_metadata_update(const MetadataModel* const mm)
{
	if (m_dialog != NULL) {
		if (mm->key() == "user-min")
			m_dialog->set_range_min(mm->path(), atof(mm->value().c_str()));
		else if (mm->key() == "user-max")
			m_dialog->set_range_max(mm->path(), atof(mm->value().c_str()));
		//else
	//		cerr << "[PluginModule::port_metadata_update] Unknown metadata key " << mm->key() << endl;
	} else if (ClientPathParser::parent(mm->path()) == m_node_model->path()) {
		string port_name = ClientPathParser::name(mm->path());
		PortModel* pm = m_node_model->port_model(port_name);
		if (pm != NULL) {
			if (mm->key() == "user-min")
				pm->user_min(atof(mm->value().c_str()));
			else if (mm->key() == "user-max")
				pm->user_max(atof(mm->value().c_str()));
			//else
	//			cerr << "[PluginModule::port_metadata_update] Unknown metadata key " << mm->key() << endl;
		} else {
			//cerr << "[PluginModule::port_metadata_update] Can not find port " << mm->path() << endl;
		}
	}
}


void
PluginModule::show_dialog()
{
      //std::cerr << "[PluginModule] Show dialog\n";
      if (m_dialog == NULL) {
              Glib::RefPtr<Gnome::Glade::Xml> xml = app->new_glade_reference("node_control_win");
              xml->get_widget_derived("node_control_win", m_dialog);
              m_dialog->init(m_patch_controller, m_node_model);
      }
      m_dialog->show();
      m_dialog->raise();
}


void
PluginModule::menu_remove()
{
	controller->remove_node(m_node_model->path());
}


void
PluginModule::menu_learn()
{
	controller->midi_learn(m_node_model->path());
}


} // namespace OmGtk
