/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef ALSADRIVER_H
#define ALSADRIVER_H

#include <alsa/asoundlib.h>
#include <string>
#include "List.h"
#include "Queue.h"
#include <list>
using std::list;
using std::string;

namespace Om {

class InputPort;
class Node;
class SetControlEvent;
class MidiLearnEvent;
class Patch;


/** Representation of an ALSA MIDI port, associated with a Patch.
 *
 * MIDI events are actually handled in this class, the AlsaDriver reads events
 * and distributes them to the appropriate AlsaPort.
 *
 * \ingroup engine
 */
class AlsaPort
{
public:
	AlsaPort(snd_seq_t* seq, Patch* patch);
	~AlsaPort();

	void event(snd_seq_event_t* const ev);
		
	void build_dssi_events_array(const samplecount block_start, const samplecount block_end);

	void set_midi_learn_event(MidiLearnEvent* const ev) { m_midi_learn_event = ev; }
	
	Patch* patch()   { return m_patch; }
	int    port_id() { return m_port_id; }

	friend class Patch;

private:
	void note_on(const snd_seq_ev_note_t& note);
	void note_off(const snd_seq_ev_note_t& note);
	void control(const snd_seq_ev_ctrl_t& ctrl);
	
	snd_seq_t* m_seq;
	Patch*     m_patch;
	int        m_port_id;

	Queue<snd_seq_event_t> m_dssi_events;
	snd_seq_event_t*       m_dssi_events_array;
	size_t                 m_dssi_events_size;

	MidiLearnEvent* m_midi_learn_event;
};



/** Alsa MIDI driver.
 *
 * This driver reads Alsa MIDI events and dispatches them to the appropriate
 * AlsaPort for processing.
 *
 * \ingroup engine
 */
class AlsaDriver
{
public:
	AlsaDriver();
	
	void activate();
	void deactivate();
	void enable()  { m_is_enabled = true; }
	void disable() { m_is_enabled = false; }
	
	void set_midi_learn_event(MidiLearnEvent* const ev);

	void build_dssi_events_arrays(const samplecount block_start, const samplecount block_end);
	
	void add_port(Patch* p);
	void remove_port(Patch* p);

private:
	list<AlsaPort*> m_ports; // FIXME: remove std::list usage
	
	// MIDI thread
	static void* process_midi_in(void* me);

	snd_seq_t*   m_seq_handle;
	pthread_t    m_process_thread;
	bool         m_is_enabled;
	static bool  m_midi_thread_exit_flag;
};


} // namespace Om

#endif // ALSADRIVER_H
