/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "DisconnectPortEvent.h"
#include <iostream>
#include "Om.h"
#include "OmApp.h"
#include "Maid.h"
#include "List.h"
#include "Node.h"
#include "Connection.h"
#include "DisconnectionEvent.h"
#include "Port.h"
#include "Array.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Patch.h"
#include "OSCSender.h"
#include "util.h"


using std::cerr; using std::endl;

namespace Om {


DisconnectPortEvent::DisconnectPortEvent(Request request, const string& port_path)
: SlowEvent(request),
  m_port_path(port_path),
  m_patch(NULL),
  m_port(NULL),
  m_process_order(NULL),
  m_succeeded(true),
  m_lookup(true)
{
}


DisconnectPortEvent::DisconnectPortEvent(Port* port)
: SlowEvent(NIL_REQUEST),
  m_port_path(""),
  m_patch((port->node() == NULL) ? NULL : port->node()->parent()),
  m_port(port),
  m_process_order(NULL),
  m_succeeded(true),
  m_lookup(false)
{
	cerr << "DisconnectPortEvent()\n";
}


DisconnectPortEvent::~DisconnectPortEvent()
{
	for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
		delete (*i);
}


void
DisconnectPortEvent::prepare()
{
	// cerr << "Preparing disconnection event...\n";
	
	if (m_lookup) {
		m_patch = om->path_parser()->find_patch(
			om->path_parser()->parent(om->path_parser()->parent(m_port_path)));
	
		if (m_patch == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		
		m_patch->node_remove_mutex().soft_lock();
		
		m_port = om->path_parser()->find_port(m_port_path);
		
		if (m_port == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
	}

	if (m_patch == NULL) {
		m_succeeded = false;
		SlowEvent::prepare();
		return;
	}
	
	Connection* c = NULL;
	for (List<Connection*>::const_iterator i = m_patch->connections().begin(); i != m_patch->connections().end(); ++i) {
		c = (*i);
		if ((c->src_port() == m_port || c->dst_port() == m_port) && !c->pending_disconnection()) {
			DisconnectionEvent* ev = new DisconnectionEvent(c->src_port(), c->dst_port());
			ev->prepare();
			m_disconnection_events.push_back(new ListNode<DisconnectionEvent*>(ev));
			c->pending_disconnection(true);
		}
	}

	cout << "[DisconnectPortEvent::prepare]\n";

	m_succeeded = true;
	SlowEvent::prepare();	
}


void
DisconnectPortEvent::execute(uint sample_offset)
{
	if (m_succeeded) {
		for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
			(*i)->execute(sample_offset);
	}
	
	SlowEvent::execute(sample_offset);
}


void
DisconnectPortEvent::post_process()
{
	if (m_lookup)
		m_patch->node_remove_mutex().soft_unlock();

	if (m_succeeded) {
		m_request.respond_ok();
		for (List<DisconnectionEvent*>::iterator i = m_disconnection_events.begin(); i != m_disconnection_events.end(); ++i)
			(*i)->post_process();
	} else {
		m_request.respond_error("Unable to disconnect port.");
	}
}


} // namespace Om

