/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "NodeTree.h"
#include <cstdlib>
#include <iostream>
#include <cassert>

namespace Om {


/** FIXME: this is all in horrible need of a rewrite. */


/** Insert a node into the tree.
 *
 * This method is realtime safe.
 */
void
NodeTree::insert(TreeNode* const n)
{
	assert(n != NULL);
	assert(n->node() != NULL);
	assert(n->left_child() == NULL);
	assert(n->right_child() == NULL);
	assert(n->parent() == NULL);
	
	if (m_root == NULL) {
		m_root = n;
	} else {
		bool left = false; // which child to insert at
		bool right = false;
		TreeNode* i = m_root;
		while (true) {
			assert(i != NULL);
			assert(i->node() != NULL);
			if (n->node()->name() <= i->node()->name()) {
				if (i->left_child() == NULL) {
					left = true;
					break;
				} else {
					i = i->left_child();
				}
			} else {
				if (i->right_child() == NULL) {
					right = true;
					break;
				} else {
					i = i->right_child();
				}
			}
		}
		
		assert(i != NULL);
		assert(i->node() != NULL);
		assert(left || right);
		assert( ! (left && right) );
		
		if (left) {
			assert(i->left_child() == NULL);
			i->left_child(n);
		} else if (right) {
			assert(i->right_child() == NULL);
			i->right_child(n);
		} 
		n->parent(i);
	}
	++m_size;
}


/** Remove a node from the tree.
 *
 * Realtime safe, caller is responsible to delete returned value.
 *
 * IMPORTANT:  The TreeNode returned to be deleted is not necessarily
 * the TreeNode passed - DO NOT DELETE THE PASSED TREENODE AFTER THIS
 * CALL!  The passed treenode will also be modified (possibly containing
 * a different Node than at first).  Do not use it after this call, it
 * is invalid.
 */
TreeNode*
NodeTree::remove(TreeNode* const node)
{
	TreeNode* n         = node;
	TreeNode* swap      = NULL;
	Node*     temp_node = NULL;
	
	const Node* const remove_node = node->node(); // for error checking
	
	// Node is not even in tree
	if (node->parent() == NULL && m_root != node)
		return NULL;
	
	// FIXME: What if the node is in a different tree?  Check for this?
	
	assert(node != NULL);
	assert(node->node() != NULL);
	
	// n has two children
	if (n->left_child() != NULL && n->right_child() != NULL) {
		if (rand()%2)
			swap = m_find_largest(n->left_child());
		else
			swap = m_find_smallest(n->right_child());
		
		temp_node = swap->m_node;
		swap->m_node = n->m_node;
		n->m_node = temp_node;

		n = swap;
	}

	// be sure we swapped correctly (ie right node is getting removed)
	assert(n->node() == remove_node);
	
	// n now has at most one child
	assert(n->left_child() == NULL || n->right_child() == NULL);

	if (n->is_leaf()) {
		if (n->is_left_child())
			n->parent()->left_child(NULL);
		else if (n->is_right_child())
			n->parent()->right_child(NULL);
		
		if (m_root == n) m_root = NULL;
	} else {  // has a single child
		TreeNode* child = NULL;
		if (n->left_child() != NULL)
			child = n->left_child();
		else if (n->right_child() != NULL)
			child = n->right_child();
		else
			throw;

		assert(child != n);
		assert(child != NULL);
		assert(child->node() != NULL);
		assert(n->parent() != n);

		if (n->is_left_child()) {
			assert(n->parent() != child);
			n->parent()->left_child(child);
			child->parent(n->parent());
		} else if (n->is_right_child()) {
			assert(n->parent() != child);
			n->parent()->right_child(child);
			child->parent(n->parent());
		} else {
			child->parent(NULL);
		}
		if (m_root == n) m_root = child;	
	}
	
	// Be sure node is cut off completely
	assert(n->parent() == NULL || n->parent()->left_child() != n);
	assert(n->parent() == NULL || n->parent()->right_child() != n);
	assert(n->left_child() == NULL || n->left_child()->parent() != n);
	assert(n->right_child() == NULL || n->right_child()->parent() != n);
	assert(m_root != n);

	n->parent(NULL);
	n->left_child(NULL);
	n->right_child(NULL);

	--m_size;

	if (m_size == 0) m_root = NULL;

	// Be sure right node is being removed
	assert(n->node() == remove_node);
			
	return n;
}


Node*
NodeTree::find(const string& name) const
{
	TreeNode* tn = find_treenode(name);

	return (tn == NULL) ? NULL : tn->node();
}


TreeNode*
NodeTree::find_treenode(const string& name) const
{
	TreeNode* i = m_root;
	
	while (i != NULL) {
		if (name == i->node()->name())
			break;
		else if (name <= i->node()->name())
			i = i->left_child();
		else
			i = i->right_child();
	}

	return i;
}


/// Private ///
void
NodeTree::m_set_all_traversed_recursive(TreeNode* root, bool b)
{
	assert(root != NULL);
	
	// Preorder traversal
	root->node()->traversed(b);
	if (root->left_child() != NULL)
		m_set_all_traversed_recursive(root->left_child(), b);
	if (root->right_child() != NULL)
		m_set_all_traversed_recursive(root->right_child(), b);
}


/** Finds the smallest (key) node in the subtree rooted at "root"
 */
TreeNode*
NodeTree::m_find_smallest(TreeNode* root)
{
	TreeNode* r = root;

	while (r->left_child() != NULL)
		r = r->left_child();

	return r;
}


/** Finds the largest (key) node in the subtree rooted at "root".
 */
TreeNode*
NodeTree::m_find_largest(TreeNode* root)
{
	TreeNode* r = root;

	while (r->right_child() != NULL)
		r = r->right_child();

	return r;

}



//// Iterator Stuff ////



NodeTree::iterator::iterator(const NodeTree *tree, uint size)
: m_depth(-1),
  m_size(size),
  m_stack(NULL),
  m_tree(tree)
{
	if (size > 0)
		m_stack = new TreeNode*[size];
}


NodeTree::iterator::~iterator()
{
	delete[] m_stack;
}


Node*
NodeTree::iterator::operator*() const
{
	if (m_depth < 0)
		return NULL;
	else
		return m_stack[m_depth]->node();
}


void
NodeTree::iterator::operator++()
{
	if (m_depth < 0)
		return;

	TreeNode* tn = m_stack[m_depth];
	--m_depth;

	tn = tn->right_child();
	while (tn != NULL) {
		++m_depth;
		m_stack[m_depth] = tn;
		tn = tn->left_child();
	}
}


bool
NodeTree::iterator::operator!=(const NodeTree::iterator& iter) const
{
	// (DeMorgan's Law)
	return (m_tree != iter.m_tree || m_depth != iter.m_depth);
}


NodeTree::iterator
NodeTree::begin() const
{
	NodeTree::iterator iter(this, m_size);
	iter.m_depth = -1;
	
	TreeNode *ptr = m_root;
	while (ptr != NULL) {
		iter.m_depth++;
		iter.m_stack[iter.m_depth] = ptr;
		ptr = ptr->left_child();
	}

	return iter;
}


NodeTree::iterator
NodeTree::end() const
{
	NodeTree::iterator iter(this, 0);
	iter.m_depth = -1;

	return iter;
}

} // namespace Om

