/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OSCReceiver.h"
#include <iostream>
#include <string>
#include <lo/lo.h>
#include "Om.h"
#include "OmApp.h"
#include "OSCSender.h"
#include "JackDriver.h"
#include "AlsaDriver.h"
#include "ActivateEvent.h"
#include "DeactivateEvent.h"
#include "EnableEvent.h"
#include "DisableEvent.h"
#include "EnablePatchEvent.h"
#include "DisablePatchEvent.h"
#include "SetControlEvent.h"
#include "SetControlSlowEvent.h"
#include "ConnectionEvent.h"
#include "DisconnectionEvent.h"
#include "AddNodeEvent.h"
#include "RemoveNodeEvent.h"
#include "CreatePatchEvent.h"
#include "DestroyPatchEvent.h"
#include "SetMetadataEvent.h"
#include "GetMetadataEvent.h"
#include "GetControlEvent.h"
#include "GetAllObjectsEvent.h"
#include "SendPluginsEvent.h"
#include "LoadPluginsEvent.h"
#include "NoteOnEvent.h"
#include "NoteOffEvent.h"
#include "DSSIUpdateEvent.h"
#include "DSSIControlEvent.h"
#include "DSSIConfigureEvent.h"
#include "DSSIProgramEvent.h"
#include "MidiLearnEvent.h"
#include "DisconnectNodeEvent.h"

using std::cerr; using std::cout; using std::endl;

namespace Om {

/*! \page engine_osc_namespace Engine OSC Namespace Documentation
 *
 * \p These are the commands the engine recognizes.  A client can control every
 * aspect of the engine entirely with these commands. \n \n
 */


OSCReceiver::OSCReceiver(const char* const port)
: m_port(port)
{
	m_st = lo_server_thread_new(port, error_cb);
	
	if (m_st == NULL) {
		std::cerr << "Could not start OSC server.  Aborting." << std::endl;
		throw;
	} else {
		std::cout << "*** Started OSC server on port " << lo_server_thread_get_port(m_st) <<
			", " << lo_server_thread_get_url(m_st) << "***" << std::endl;
	}

	// For debugging, print all incoming OSC messages
	//lo_server_thread_add_method(m_st, NULL, NULL, generic_cb, NULL);

	// Commands
	lo_server_thread_add_method(m_st, "/om/ping", "i", ping_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/quit", "i", quit_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/register_client", "i", register_client_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/register_client", "isi", register_client_host_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/unregister_client", "i", unregister_client_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/load_plugins", "i", load_plugins_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/activate", "i", engine_activate_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/deactivate", "i", engine_deactivate_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/enable", "i", engine_enable_cb, this);
	lo_server_thread_add_method(m_st, "/om/engine/disable", "i", engine_disable_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/create_patch", "isi", create_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/destroy_patch", "is", destroy_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/enable_patch", "is", enable_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/disable_patch", "is", disable_patch_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/create_node", "issssi", patch_add_node_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/destroy_node", "is", patch_remove_node_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/connect", "iss", patch_connect_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/disconnect", "iss", patch_disconnect_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/disconnect_all", "is", patch_disconnect_all_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/set_port_value", "isf", set_port_value_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/set_port_value", "isif", set_port_value_voice_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/set_port_value_slow", "isf", set_port_value_slow_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/note_on", "isii", note_on_cb, this);
	lo_server_thread_add_method(m_st, "/om/synth/note_off", "isi", note_off_cb, this);
	lo_server_thread_add_method(m_st, "/om/midi/learn", "is", midi_learn_cb, this);
	
	lo_server_thread_add_method(m_st, "/om/metadata/request", "iss", metadata_get_cb, this);
	lo_server_thread_add_method(m_st, "/om/metadata/set", "isss", metadata_set_cb, this);
	
	// Queries
	lo_server_thread_add_method(m_st, "/om/request/plugins", "i", send_plugins_cb, this);
	lo_server_thread_add_method(m_st, "/om/request/all_objects", "i", send_all_objects_cb, this);
	lo_server_thread_add_method(m_st, "/om/request/port_value", "is", patch_request_control_cb, this);
	
	// DSSI support
	lo_server_thread_add_method(m_st, NULL, NULL, dssi_cb, this);

	lo_server_thread_add_method(m_st, NULL, NULL, unknown_cb, NULL);
}


OSCReceiver::~OSCReceiver()
{
	lo_server_thread_free(m_st);
}


void
OSCReceiver::activate()
{
	lo_server_thread_start(m_st);
	//lo_server_thread_broadcast(m_st, "om-synth");
}


lo_address
OSCReceiver::get_reply_addr(const lo_message msg)
{
	// Look up using URL?  Host/Port?  Does it matter?
	
	lo_address incoming_addr = lo_message_get_source(msg);
	const char* const incoming_hostname = lo_address_get_hostname(incoming_addr);
	const char* const incoming_port = lo_address_get_port(incoming_addr);
	
	ClientRecord* client_record = NULL;
	
	for (list<ClientRecord*>::iterator i = m_clients.begin(); i != m_clients.end(); ++i) {
		if (!strcmp((*i)->hostname().c_str(), incoming_hostname)
				&& !strcmp((*i)->port().c_str(), incoming_port)) {
			client_record = (*i);
			break;
		}
	}

	if (client_record != NULL) {
		return client_record->addr();
	} else {
		//std::cerr << "Unable to find client!  (Received message from unregistered client?)" << std::endl;
		//throw;
		//return incoming_addr; // Sensible default I guess?
		return NOBODY; // don't respond to anyone
	}
}


void
OSCReceiver::error_cb(int num, const char* msg, const char* path)
{
	std::cerr << "liblo server error " << num << " in path \"" << "\" - " << msg << std::endl;
}


/** \page engine_osc_namespace
 * <p> \b /om/ping - Sends a successful response to the given request id.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_ping_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;
	
	Request(addr, request_id).respond_ok();
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/quit - Terminates the engine.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_quit_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr = get_reply_addr(msg);
	const int request_id = argv[0]->i;

	Request(addr, request_id).respond_ok();
	
	om->set_quit_flag();

	return 0;  // handled
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/register_client - Registers a new client with the engine (using source address)
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_register_client_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	const int   request_id   =  argv[0]->i;
	
	lo_address incoming_addr = lo_message_get_source(msg);
	const char* const incoming_host = lo_address_get_hostname(incoming_addr);
	const char* const incoming_port = lo_address_get_port(incoming_addr);
	
	cout << "Registering client " << incoming_host << ":" << incoming_port << endl;

	bool found = false;
	for (list<ClientRecord*>::iterator i = m_clients.begin(); i != m_clients.end(); ++i)
		if (!strcmp((*i)->hostname().c_str(), incoming_host)
				&& !strcmp((*i)->port().c_str(), incoming_port))
			found = true;

	if (!found) {
		ClientRecord* cr = new ClientRecord(incoming_host, incoming_port);
		m_clients.push_back(cr);
		cout << "Added client " << incoming_host << ":" << incoming_port
			<< " to clients list (" << m_clients.size() << " clients)" << endl;
	} else {
		cout << "Client already registered." << endl;
	}
	
	Request(incoming_addr, request_id).respond_ok();
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/register_client - Registers a new client with the engine (using specified host/port)
 * \arg \b request-id (integer)
 * \arg \b host (string) - Hostname to send replies/notifications to
 * \arg \b port (integer) - Port to send replies/notifications to </p> \n \n
 */
int
OSCReceiver::m_register_client_host_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	const int         request_id =  argv[0]->i;
	const char* const host       = &argv[1]->s;
	const int         port       =  argv[2]->i;
	
	char port_str[8];
	snprintf(port_str, 8, "%d", port);
	
	cout << "Registering client " << host << ":" << port_str << endl;

	bool found = false;
	for (list<ClientRecord*>::iterator i = m_clients.begin(); i != m_clients.end(); ++i)
		if (!strcmp((*i)->hostname().c_str(), host)
				&& !strcmp((*i)->port().c_str(), port_str))
			found = true;

	if (!found) {
		ClientRecord* cr = new ClientRecord(host, port_str);
		m_clients.push_back(cr);
		cout << "Added client " << host << ":" << port_str
			<< " to clients list (" << m_clients.size() << " clients)" << endl;
	} else {
		cout << "Client already registered." << endl;
	}
	
	lo_address incoming_addr = lo_message_get_source(msg);
	Request(incoming_addr, request_id).respond_ok();
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/unregister_client - Unregisters a client so it will no longer receive messages from the engine.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_unregister_client_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	
	if (addr == NULL) {
		cerr << "WARNING: Unregistration request from unregistered client." << endl;
		return 0;
	}
	
	const char* const incoming_host = lo_address_get_hostname(addr);
	const char* const incoming_port = lo_address_get_port(addr);
	
	bool found = false;
	for (list<ClientRecord*>::iterator i = m_clients.begin(); i != m_clients.end(); ++i) {
		if (!strcmp((*i)->hostname().c_str(), incoming_host)
				&& !strcmp((*i)->port().c_str(), incoming_port)) {
			found = true;
			Request(addr, request_id).respond_ok();
			m_clients.erase(i);
			delete (*i);
			break;
		}
	}
	
	if (found) {
		cout << "Unregistered client.\n";
	} else {
		cerr << "ERROR: Unable to find client to unregister!" << std::endl;
		Request(addr, request_id).respond_error("Unable to find client to unregister.");
	}
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/load_plugins - Locates all available plugins, making them available for use.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_load_plugins_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;

	LoadPluginsEvent* ev = new LoadPluginsEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/activate - Activate the engine (MIDI, audio, everything)
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_engine_activate_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;

	ActivateEvent* ev = new ActivateEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	//om->activate();
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/deactivate - Deactivate the engine completely.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_engine_deactivate_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;

	DeactivateEvent* ev = new DeactivateEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	//om->deactivate();

	return 0;
}

/** \page engine_osc_namespace
 * <p> \b /om/engine/enable - Enable the engine's DSP processing
 * \arg \b request-id (integer) 
 *
 * \li The client that sends this command will receive an affirmative response,
 * but not an engine enabled notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_engine_enable_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;

	EnableEvent* ev = new EnableEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/engine/disable - Disable the engine's DSP processing
 * \arg \b request-id (integer) 
 * 
 * \li The client that sends this command will receive an affirmative response,
 * but not an engine disabled notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_engine_disable_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	const int  request_id = argv[0]->i;

	DisableEvent* ev = new DisableEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/create_patch - Creates a new, empty, toplevel patch.
 * \arg \b request-id  (integer)
 * \arg \b patch-path  (string) - Patch path (complete, ie /master/parent/new_patch)
 * \arg \b poly        (integer) - Patch's (internal) polyphony </p> \n \n
 */
int
OSCReceiver::m_create_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	const int   poly       =  argv[2]->i;

	CreatePatchEvent* ev = new CreatePatchEvent(Request(addr, request_id), patch_path, poly);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/destroy_patch - Destroys a patch and all nodes (etc) underneath it.
 * \arg \b request-id (integer)
 * \arg \b patch-path - Patch's path </p> \n \n
 */
int
OSCReceiver::m_destroy_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	
	DestroyPatchEvent* ev = new DestroyPatchEvent(Request(addr, request_id), patch_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/enable_patch - Enable DSP processing of a patch
 * \arg \b request-id (integer)
 * \arg \b patch-path - Patch's path
 * 
 * \li The client that sends this command will receive an affirmative response,
 * but not a patch enabled notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_enable_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	
	EnablePatchEvent* ev = new EnablePatchEvent(Request(addr, request_id), patch_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/disable_patch - Disable DSP processing of a patch
 * \arg \b request-id (integer)
 * \arg \b patch-path - Patch's path
 * 
 * \li The client that sends this command will receive an affirmative response,
 * but not a patch disabled notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_disable_patch_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	
	DisablePatchEvent* ev = new DisablePatchEvent(Request(addr, request_id), patch_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/create_node - Adds (loads) a node into a given patch.
 * \arg \b request-id (integer)
 * \arg \b node-path (string) - Full path of the new node (ie. /patch2/subpatch/newnode)
 * \arg \b type (string) - Plugin type ("LADSPA" or "Internal")
 * \arg \b lib-name (string) - Name of library where plugin resides (eg "cmt.so")
 * \arg \b plug-label (string) - Label (ID) of plugin (eg "sine_fcaa")
 * \arg \b poly (integer-boolean) - Whether node is polyphonic (0 = false, 1 = true) </p> \n \n
 */
int
OSCReceiver::m_patch_add_node_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;
	const char* type       = &argv[2]->s;
	const char* lib_name   = &argv[3]->s;
	const char* plug_label = &argv[4]->s;
	const int   poly       =  argv[5]->i;
	
	// FIXME Event-ize
	
	PluginInfo* info = new PluginInfo();
	info->set_type(type);
	info->lib_name(lib_name);
	info->plug_label(plug_label);

	if (poly != 0 && poly != 1) {
		Request(addr, request_id).respond_error("Invalid poly parameter in /patch/add_node");
		return 0;
	}

	AddNodeEvent* ev = new AddNodeEvent(Request(addr, request_id), node_path, info, (poly == 1));
	om->jack_driver()->push_slow_event(ev);

	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/destroy_node - Removes (destroys) a loaded node from a patch
 * \arg \b request-id (integer)
 * \arg \b node-path (string) - Full path of the node </p> \n \n
 */
int
OSCReceiver::m_patch_remove_node_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;
	
	RemoveNodeEvent* ev = new RemoveNodeEvent(Request(addr, request_id), node_path);
	om->jack_driver()->push_slow_event(ev);
		
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/connect - Connects two ports (must be in the same patch)
 * \arg \b request-id (integer)
 * \arg \b src-port-path (string) - Full path of source port
 * \arg \b dst-port-path (string) - Full path of destination port </p> \n \n
 */
int
OSCReceiver::m_patch_connect_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr          = get_reply_addr(msg);
	const int   request_id    =  argv[0]->i;
	const char* src_port_path = &argv[1]->s;
	const char* dst_port_path = &argv[2]->s;

	ConnectionEvent* ev = new ConnectionEvent(Request(addr, request_id), src_port_path, dst_port_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/disconnect - Disconnects two ports.
 * \arg \b request-id (integer)
 * \arg \b src-port-path (string) - Full path of source port
 * \arg \b dst-port-path (string) - Full path of destination port </p> \n \n
 */
int
OSCReceiver::m_patch_disconnect_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr          = get_reply_addr(msg);
	const int   request_id    =  argv[0]->i;
	const char* src_port_path = &argv[1]->s;
	const char* dst_port_path = &argv[2]->s;

	DisconnectionEvent* ev = new DisconnectionEvent(Request(addr, request_id), src_port_path, dst_port_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/disconnect_all - Disconnect all connections to/from a node.
 * \arg \b request-id (integer)
 * \arg \b node-path (string) - Full path of node. </p> \n \n
 */
int
OSCReceiver::m_patch_disconnect_all_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;

	DisconnectNodeEvent* ev = new DisconnectNodeEvent(Request(addr, request_id), node_path);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/set_port_value - Sets the value of a port for all voices (both AR and CR)
 * \arg \b request-id (integer)
 * \arg \b port-path (string) - Name of port
 * \arg \b value (float) - Value to set port to
 *
 * \li The client that sends this command will receive an affirmative response,
 * but not a control update notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_set_port_value_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* port_path  = &argv[1]->s;
	const float value      =  argv[2]->f;

	Request request(addr, request_id);
	
	SetControlEvent* ev = new SetControlEvent(request, port_path, value);
	om->jack_driver()->push_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/set_port_value - Sets the value of a port for a specific voice (both AR and CR)
 * \arg \b request-id (integer)
 * \arg \b port-path (string) - Name of port
 * \arg \b value (float) - Value to set port to
 *
 * \li The client that sends this command will receive an affirmative response,
 * but not a control update notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_set_port_value_voice_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* port_path  = &argv[1]->s;
	const int   voice      =  argv[2]->i;
	const float value      =  argv[3]->f;

	Request request(addr, request_id);
	
	SetControlEvent* ev = new SetControlEvent(request, voice, port_path, value);
	om->jack_driver()->push_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/set_port_value_slow - Sets the value of a port for all voices (as a SlowEvent)
 * \arg \b request-id (integer)
 * \arg \b port-path (string) - Name of port
 * \arg \b value (float) - Value to set port to
 *
 * \li This version exists so you can send it immediately after a SlowEvent it may depend on (ie a
 * node creation) and be sure it happens after the event (a normal set_control could beat the
 * slow event and arrive out of order). </p> \n \n
 */
int
OSCReceiver::m_set_port_value_slow_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* port_path  = &argv[1]->s;
	const float value      =  argv[2]->f;

	Request request(addr, request_id);
	
	SetControlSlowEvent* ev = new SetControlSlowEvent(request, port_path, value);
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/note_on - Triggers a note-on, just as if it came from MIDI
 * \arg \b request-id (integer)
 * \arg \b patch-path (string) - Patch of patch to send event to
 * \arg \b note-num (integer) - MIDI style note number
 * \arg \b velocity (integer) - MIDI style velocity </p> \n \n
 */
int
OSCReceiver::m_note_on_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	const int   note_num   =  argv[2]->i;
	const int   velocity   =  argv[3]->i;

	Request request(addr, request_id);
	
	NoteOnEvent* ev = new NoteOnEvent(request, patch_path, note_num, velocity);
	om->jack_driver()->push_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/synth/note_off - Triggers a note-off, just as if it came from MIDI
 * \arg \b request-id (integer)
 * \arg \b patch-path (string) - Patch of patch to send event to
 * \arg \b note-num (integer) - MIDI style note number </p> \n \n
 */
int
OSCReceiver::m_note_off_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* patch_path = &argv[1]->s;
	const int   note_num   =  argv[2]->i;

	Request request(addr, request_id);
	
	NoteOffEvent* ev = new NoteOffEvent(request, patch_path, note_num);
	om->jack_driver()->push_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/midi/learn - "Learn" a binding for a MIDI control node.
 * \arg \b request-id (integer)
 * \arg \b node-path (string) - Path of the MIDI control node to bind.
 */
int
OSCReceiver::m_midi_learn_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	const int   request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;

	Request request(addr, request_id);

	MidiLearnEvent* ev = new MidiLearnEvent(request, node_path);
	ev->prepare(); // FIXME: holds up the OSC thread
	om->alsa_driver()->set_midi_learn_event(ev);

	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/metadata/set - Sets a piece of metadata, associated with a synth-space object (node, etc)
 * \arg \b request-id (integer)
 * \arg \b object-path (string) - Full path of object to associate metadata with
 * \arg \b key (string) - Key (index) for new piece of metadata
 * \arg \b value (string) - Value of new piece of metadata
 *
 * \li The client that sends this command will receive an affirmative response,
 * but not a metadata update notification, to avoid a feedback loop. </p> \n \n
 */
int
OSCReceiver::m_metadata_set_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	int         request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;
	const char* key        = &argv[2]->s;
	const char* value      = &argv[3]->s;
	
	SetMetadataEvent* ev = new SetMetadataEvent(Request(addr, request_id),
		node_path, key, value);
	
	om->jack_driver()->push_slow_event(ev);

	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/metadata/request - Requests the engine send a piece of metadata, associated with a synth-space object (node, etc)
 * \arg \b request-id (integer)
 * \arg \b object-path (string) - Full path of object metadata is associated with
 * \arg \b key (string) - Key (index) for piece of metadata </p> \n \n
 */
int
OSCReceiver::m_metadata_get_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	int         request_id =  argv[0]->i;
	const char* node_path  = &argv[1]->s;
	const char* key        = &argv[2]->s;
	
	GetMetadataEvent* ev = new GetMetadataEvent(Request(addr, request_id),
		node_path, key);

	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/request/port_value - Requests the engine send the value of a port.
 * \arg \b request-id (integer)
 * \arg \b port-path (string) - Full path of port to send the value of </p> \n \n
 */
int
OSCReceiver::m_patch_request_control_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr       = get_reply_addr(msg);
	int         request_id =  argv[0]->i;
	const char* port_path  = &argv[1]->s;

	Request request(addr, request_id);
	
	GetControlEvent* ev = new GetControlEvent(Request(addr, request_id), port_path);
	om->jack_driver()->push_slow_event(ev);

	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/request/plugins - Requests the engine send a list of all known plugins.
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_send_plugins_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address addr       = get_reply_addr(msg);
	int        request_id = argv[0]->i;
	
	Request request(addr, request_id);

	SendPluginsEvent* ev = new SendPluginsEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


/** \page engine_osc_namespace
 * <p> \b /om/request/all_objects - Requests the engine send information about \em all objects (patches, nodes, etc)
 * \arg \b request-id (integer) </p> \n \n
 */
int
OSCReceiver::m_send_all_objects_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	lo_address  addr = get_reply_addr(msg);
	int         request_id =  argv[0]->i;

	GetAllObjectsEvent* ev = new GetAllObjectsEvent(Request(addr, request_id));
	om->jack_driver()->push_slow_event(ev);
	
	return 0;
}


int
OSCReceiver::m_dssi_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg)
{
	string node_path(path);

	if (node_path.substr(0, 5) != "/dssi")
		return 1;
	
	string command = node_path.substr(node_path.find_last_of("/")+1);
	node_path = node_path.substr(5); // chop off leading "/dssi/"
	node_path = node_path.substr(0, node_path.find_last_of("/")); // chop off command at end
	
	cout << "DSSI:  Got message " << command << " for node " << node_path << std::endl;

	SlowEvent* ev = NULL;
	
	if (command == "update" && !strcmp(types, "s"))
		ev = new DSSIUpdateEvent(NIL_REQUEST, node_path, &argv[0]->s);
	else if (command == "control" && !strcmp(types, "if"))
		ev = new DSSIControlEvent(NIL_REQUEST, node_path, argv[0]->i, argv[1]->f);
	else if (command == "configure" && ~strcmp(types, "ss"))
		ev = new DSSIConfigureEvent(NIL_REQUEST, node_path, &argv[0]->s, &argv[1]->s);
	else if (command == "program" && ~strcmp(types, "ii"))
		ev = new DSSIProgramEvent(NIL_REQUEST, node_path, argv[0]->i, argv[1]->i);

	if (ev != NULL)
		om->jack_driver()->push_slow_event(ev);
	else
		cerr << "[OSCReceiver] Unknown DSSI command received: " << path << endl;

	return 0;
}


//  Static Callbacks //


// Display incoming OSC messages (for debugging purposes)
int
OSCReceiver::generic_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* user_data)
{
    printf("Path:  %s\n", path);
    
	for (int i=0; i < argc; ++i) {
		printf("Arg %d: '%c'  ", i, types[i]);
		lo_arg_pp(lo_type(types[i]), argv[i]);
		printf("\n");
    }
    printf("\n");

	return 1;  // not handled
}


int
OSCReceiver::unknown_cb(const char* path, const char* types, lo_arg** argv, int argc, lo_message msg, void* user_data)
{
	std::cerr << "Unknown command, sending error.\n";

	string error_msg = "Unknown command: ";
	error_msg.append(path).append(" ").append(types);

	om->osc_sender()->send_error(error_msg);

	return 0;
}


} // namespace Om
