/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <cassert>
#include <cmath>
#include <iostream>
#include "Patch.h"
#include "PluginInfo.h"
#include "Port.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "OSCSender.h"
#include "AlsaDriver.h"
#include "Connection.h"
#include "InputNode.h"
#include "OutputNode.h"
#include "MidiInNode.h"
#include "Om.h"

namespace Om {


Patch::Patch(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size, uint internal_poly) 
: NodeBase(path, poly, parent, srate, buffer_size),
  m_internal_poly(internal_poly),
  m_process_order(NULL),
  m_alsa_port(NULL),
  m_process(false)
{
	m_plugin_info.type(PluginInfo::Patch);
	m_plugin_info.lib_path("");
	m_plugin_info.lib_name("");
	m_plugin_info.plug_label("om_patch");
	m_plugin_info.name("Om patch");

	m_num_ports = 0;

	//std::cerr << "Creating patch " << m_name << ", poly = " << poly
	//	<< ", internal poly = " << internal_poly << std::endl;
}


Patch::~Patch()
{
	deactivate();

	for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
		delete (*i);

	delete m_process_order;
}


void
Patch::activate()
{
	if (m_activated) {
		NodeBase::activate();
	
		for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
			(*i)->activate();
	}
}


void
Patch::deactivate()
{
	if (m_activated ) {
		NodeBase::deactivate();
	
		for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
			(*i)->deactivate();
	}
}


/** Run the patch for the specified number of frames.
 * 
 * Calls all Nodes in the order m_process_order specifies.
 */
inline void
Patch::run(size_t nframes)
{
	if (m_process_order == NULL)
		return;

	// Prepare all ports
	for (List<InputNode*>::iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		(*i)->external_port()->prepare_buffers();
	for (List<OutputNode*>::iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		(*i)->external_port()->prepare_buffers();

	// Run all nodes
	for (List<Node*>::iterator i = m_process_order->begin(); i != m_process_order->end(); ++i) {
		if ((*i) == NULL)
			return;
		else
			(*i)->run(nframes);
	}
}


Port* const
Patch::port(const string& port_name) const
{
	for (List<InputNode*>::const_iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		if ((*i)->external_port()->name() == port_name)
			return (*i)->external_port();
	
	for (List<OutputNode*>::const_iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		if ((*i)->external_port()->name() == port_name)
			return (*i)->external_port();

	std::cerr << "[Patch::port] Did not find port " << port_name << "!" << std::endl;

	return NULL;
}


/** Returns the number of ports.
 *
 * Needs to override the NodeBase implementation since a Patch's ports are really
 * just it's input and output nodes' ports.
 */
uint
Patch::num_ports() const
{
	return num_in_ports() + num_out_ports();
}


uint
Patch::num_in_ports() const
{
	uint count = 0;

	for (List<InputNode*>::const_iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		if ((*i) != NULL)
			count += (*i)->num_ports();
	
	return count;
}


uint
Patch::num_out_ports() const
{
	uint count = 0;	
	
	for (List<OutputNode*>::const_iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		if ((*i) != NULL)
			count += (*i)->num_ports();
	
	return count;
}


void
Patch::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_patch_to(addr, this);
}


void
Patch::send_deletion_messages(lo_address addr) const
{
	om->osc_sender()->send_patch_destruction_to(addr, m_path);
	
	if (m_parent != NULL) 
		om->osc_sender()->send_node_removal_to(addr, m_path);
}


// Patch specific stuff


void
Patch::add_node(TreeNode* tn)
{
	assert(tn != NULL);
	assert(tn->node() != NULL);
	assert(tn->node()->poly() == m_internal_poly || tn->node()->poly() == 1);
	
	m_nodes.insert(tn);
}


TreeNode*
Patch::remove_node(TreeNode* tn)
{
	const Node* const orig_node = tn->node();
	TreeNode* ret = m_nodes.remove(tn);
	assert(ret->node() == orig_node);

	if (tn->node()->plugin_info().type() == PluginInfo::Internal) {
		((InternalNode*)(tn->node()))->remove_from_patch(this);
	}

	return ret;
}



/** Remove a connection.  Realtime safe.
 */
ListNode<Connection*>* const
Patch::remove_connection(const string& src_port_path, const string& dst_port_path)
{
	bool found = false;
	ListNode<Connection*>* connection = NULL;
	for (List<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		if ((*i)->src_port()->path() == src_port_path &&
		    (*i)->dst_port()->path() == dst_port_path) {
			connection = m_connections.remove(i);
			found = true;
		}
	}

	if ( ! found) {
		std::cerr << "WARNING:  [Patch::remove_connection] Connection not found !" << std::endl;
	}

	return connection;
}


} // namespace Om
