/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PathParser.h"
#include "Om.h"
#include "Patch.h"
#include "Node.h"
#include "Port.h"
#include "List.h"

namespace Om {


/** Parse the name from a full path.
 *
 * ie name("/master/snare/osc") == osc
 */
string
PathParser::name(const string& path)
{
	string name = path;
	
	string::size_type last_slash = path.find_last_of("/");
	if (last_slash != string::npos)
		name = name.substr(last_slash+1);

	return name;
}


/** Parse the parent's path (ie chop the name off the end).
 */
string
PathParser::parent(const string& path)
{
	string parent = path;

	string::size_type last_slash = path.find_last_of("/");
	parent = parent.substr(0, last_slash);
	
	return parent;
}


/** Find the patch at the given path.
 */
Patch* const
PathParser::find_patch(const string& path) 
{
	for (List<Patch*>::iterator i = om->patches().begin(); i != om->patches().end(); ++i)
		if ((*i)->path() == path)
			return (*i);
	
	return NULL;
}


/** Find the node at the given path.
 */
Node* const
PathParser::find_node(const string& path) 
{
	Patch* parent = find_patch(PathParser::parent(path));
	Node* node = NULL;
	
	if (parent != NULL) {
		parent->node_remove_mutex().soft_lock();
		node = parent->nodes().find(PathParser::name(path));
		parent->node_remove_mutex().soft_unlock();
	}
		
	return node;
}


/** Find the port at the given path.
 */
Port* const
PathParser::find_port(const string& path) 
{
	Node* node = find_node(PathParser::parent(path));
	
	if (node != NULL) {
		return node->port(PathParser::name(path));
	} else {
		// maybe a port on a patch
		Patch* patch = find_patch(PathParser::parent(path));
		if (patch != NULL)
			return patch->port(PathParser::name(path));
	}
	
	return NULL;
}


/** Find the object at the given path.
 */
OmObject* const
PathParser::find_object(const string& path)
{
	// FIXME: this is implemented in the worst possible way
	OmObject* obj = NULL;
	obj = find_patch(path);
	if (obj == NULL)
		obj = find_node(path);
	if (obj == NULL)
		obj = find_port(path);

	return obj;
}


} // namespace Om
