/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "RemoveNodeEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "PluginInfo.h"
#include "InputNode.h"
#include "InputPort.h"
#include "OutputNode.h"
#include "OutputPort.h"
#include "Om.h"
#include "Array.h"
#include "DisconnectNodeEvent.h"
#include "DisconnectPortEvent.h"
#include "OmApp.h"
#include "OSCSender.h"
#include "Maid.h"
#include "JackDriver.h"
#include "SlowEvent.h"

namespace Om {


RemoveNodeEvent::RemoveNodeEvent(Request request, const string& path, bool lock_mutex)
: SlowEvent(request, true),
  m_path(path),
  m_patch(NULL),
  m_treenode(NULL),
  m_delete_treenode(NULL),
  m_disconnect_event(NULL),
  m_parent_disconnect_event(NULL),
  m_lock_mutex(lock_mutex)
{
	//cout << "Creating remove node event.." << std::endl;
}


RemoveNodeEvent::~RemoveNodeEvent()
{
	delete m_disconnect_event;
	delete m_parent_disconnect_event;
}


void
RemoveNodeEvent::prepare()
{
	m_patch = om->path_parser()->find_patch(om->path_parser()->parent(m_path));

	if (m_patch != NULL) {
		m_treenode = m_patch->nodes().find_treenode(om->path_parser()->name(m_path));
		//std::cerr << "Preparing remove node event..." << std::endl;
		if (m_treenode != NULL) {
			if (m_treenode->node()->providers()->size() != 0
					|| m_treenode->node()->dependants()->size() != 0) {
				m_disconnect_event = new DisconnectNodeEvent(m_treenode->node());
				m_disconnect_event->prepare();
			}
			const PluginInfo& pi = m_treenode->node()->plugin_info();
			if (pi.type() == PluginInfo::Internal) {
				Port* parent_port = NULL;
				if (pi.plug_label() == "control_input" || pi.plug_label() == "audio_input")
					parent_port = ((InputNode*)m_treenode->node())->external_port();
				else if (pi.plug_label() == "control_output" || pi.plug_label() == "audio_output")
					parent_port = ((OutputNode*)m_treenode->node())->external_port();
				if (parent_port != NULL) {
					m_parent_disconnect_event = new DisconnectPortEvent(parent_port);
					m_parent_disconnect_event->prepare();
				}
			}
						
			if (m_lock_mutex)
				m_patch->node_remove_mutex().hard_lock();
		}
	}

	SlowEvent::prepare();
}


void
RemoveNodeEvent::execute(uint sample_offset)
{
	//cout << "Executing remove node event...";

	if (m_treenode != NULL) {
		if (m_disconnect_event != NULL)
			m_disconnect_event->execute(sample_offset);
		if (m_parent_disconnect_event != NULL)
			m_parent_disconnect_event->execute(sample_offset);
		m_delete_treenode = m_patch->remove_node(m_treenode);
	}
	
	SlowEvent::execute(sample_offset);

	//cout << "done." << std::endl;
}


void
RemoveNodeEvent::post_process()
{
	om->jack_driver()->slow_event_queue()->signal_blocking_event_finished();
	if (m_patch == NULL) {
		m_request.respond_error("Could not find patch in /patch/remove_node");
	} else if (m_delete_treenode != NULL) {	
		m_request.respond_ok();
		if (m_disconnect_event != NULL)
			m_disconnect_event->post_process();
		if (m_parent_disconnect_event != NULL)
			m_parent_disconnect_event->post_process();
		//m_delete_treenode->node()->send_deletion_messages();
		om->osc_sender()->send_node_deletion_messages(m_delete_treenode->node());
		om->maid()->push(m_delete_treenode);
		om->maid()->push(m_delete_treenode->node());
		if (m_lock_mutex)
			m_patch->node_remove_mutex().hard_unlock();
	} else {
		m_request.respond_error("Unable to remove node.  (Is it connected?)");
	}
}


} // namespace Om
