/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PatchLibrarian.h"
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include "PatchModel.h"
#include "NodeModel.h"
#include "ConnectionModel.h"
#include "ControlMapModel.h"
#include "PortModel.h"
#include "PresetModel.h"
#include "Comm.h"
#include "PluginInfo.h"
#include "ClientPathParser.h"
#include <iostream>
#include <vector>
#include <utility> // for pair, make_pair
#include <cassert>
#include <string>
#include "unistd.h" // for usleep
#include <cstdlib>  // for atof
#include <cmath>
using Om::PluginInfo;

using std::string; using std::vector; using std::pair;

namespace LibOmClient {

/** Save a patch from a PatchModel to a filename.
 */
void
PatchLibrarian::save_patch(const PatchModel* patch_model, const string& filename)
{
	/* I am not a fan of this BAD_CAST nonsense at all, it's just what's in the
	 * libxml documentation.  Figure out what to replace it with.. */
	
	cout << "Saving patch " << patch_model->path() << " to " << filename << endl;

	NodeModel* nm = NULL;
	PatchModel* spm = NULL; // subpatch model
	
	xmlDocPtr  xml_doc = NULL;
    xmlNodePtr xml_root_node = NULL;
	xmlNodePtr xml_node = NULL;
	xmlNodePtr xml_child_node = NULL;
	xmlNodePtr xml_grandchild_node = NULL;
	
    xml_doc = xmlNewDoc(BAD_CAST "1.0");
    xml_root_node = xmlNewNode(NULL, BAD_CAST "patch");
    xmlDocSetRootElement(xml_doc, xml_root_node);

	const size_t temp_buf_length = 255;
	char temp_buf[temp_buf_length];
	
	if (patch_model->name() != "")
		xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "name", BAD_CAST patch_model->name().c_str());
	else
		xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "name", BAD_CAST filename.substr(0, filename.find('.')).c_str());
	
	snprintf(temp_buf, temp_buf_length, "%d", patch_model->poly());
	xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "polyphony", BAD_CAST temp_buf);

	/*xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "metadata", NULL);
	
	snprintf(temp_buf, temp_buf_length, "%d", patch_model->width());
	xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "window-width", BAD_CAST temp_buf);
		
	snprintf(temp_buf, temp_buf_length, "%d", patch_model->height());
	xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "window-height", BAD_CAST temp_buf);
	*/
	
	// Save nodes and subpatches

	for (NodeModelMap::const_iterator i = patch_model->nodes().begin(); i != patch_model->nodes().end(); ++i) {
		nm = i->second;
		
		if (nm->plugin_info()->type() == PluginInfo::Patch) {  // Subpatch
			spm = (PatchModel*)i->second;
			xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "subpatch", NULL);
			
			xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "name", BAD_CAST spm->name().c_str());
			
			string subpatch_filename;
			if (spm->filename() == "") {
				subpatch_filename = spm->name() + ".om";
				spm->filename(subpatch_filename);
			} else {
				subpatch_filename = spm->filename().substr(spm->filename().find_last_of("/")+1);;
				spm->filename(subpatch_filename);
			}
			
			xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "filename", BAD_CAST subpatch_filename.c_str());
			
			snprintf(temp_buf, temp_buf_length, "%d", spm->poly());
			xml_child_node = xmlNewChild(xml_node, NULL,  BAD_CAST "polyphony", BAD_CAST temp_buf);
			
			subpatch_filename = filename.substr(0, filename.find_last_of("/")) +"/"+ spm->filename();
			save_patch(spm, subpatch_filename);
		} else {  // Normal node
			xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "node", NULL);
			
			xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "name", BAD_CAST nm->name().c_str());
			
			if (nm->plugin_info() == NULL) break;
	
			xml_child_node = xmlNewChild(xml_node, NULL,  BAD_CAST "polyphonic",
				BAD_CAST ((nm->polyphonic()) ? "true" : "false"));
				
			xml_child_node = xmlNewChild(xml_node, NULL,  BAD_CAST "type",
				BAD_CAST nm->plugin_info()->type_string());
				
			xml_child_node = xmlNewChild(xml_node, NULL,  BAD_CAST "plugin-label",
				BAD_CAST (nm->plugin_info()->plug_label().c_str()));
	
			if (nm->plugin_info()->type() != Om::PluginInfo::Internal) {
				xml_child_node = xmlNewChild(xml_node, NULL,  BAD_CAST "library-name",
					BAD_CAST (nm->plugin_info()->lib_name().c_str()));
			}
		}
		
		// Write metadata
		for (map<string, string>::const_iterator i = nm->metadata().begin(); i != nm->metadata().end(); ++i) {
			xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST (*i).first.c_str(), BAD_CAST (*i).second.c_str());
		}

		PortModel* pm = NULL;
		// Write port metadata, if necessary
		for (list<PortModel*>::const_iterator i = nm->port_models().begin(); i != nm->port_models().end(); ++i) {
			pm = (*i);
			if (pm->user_min() != pm->min_val() || pm->user_max() != pm->max_val()) {
				xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "port", NULL);
				xml_grandchild_node = xmlNewChild(xml_child_node, NULL, BAD_CAST "name",
					BAD_CAST ClientPathParser::name(pm->path()).c_str());
				snprintf(temp_buf, temp_buf_length, "%f", pm->user_min());
				xml_grandchild_node = xmlNewChild(xml_child_node, NULL, BAD_CAST "user-min", BAD_CAST temp_buf);
				snprintf(temp_buf, temp_buf_length, "%f", pm->user_max());
				xml_grandchild_node = xmlNewChild(xml_child_node, NULL, BAD_CAST "user-max", BAD_CAST temp_buf);
			}	
		}
	}

	// Save connections
	
	const list<ConnectionModel*>& cl = patch_model->connections();
	const ConnectionModel* c = NULL;
	
	for (list<ConnectionModel*>::const_iterator i = cl.begin(); i != cl.end(); ++i) {
		c = (*i);
		xml_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "connection", NULL);
		xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "source-node",
			BAD_CAST c->src_node_name().c_str());
		xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "source-port",
			BAD_CAST c->src_port_name().c_str());
		xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "destination-node",
			BAD_CAST c->dst_node_name().c_str());
		xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "destination-port",
			BAD_CAST c->dst_port_name().c_str());
	}
	
    // Save control values (ie presets eventually, right now just current control vals)
	
	xmlNodePtr xml_preset_node = xmlNewChild(xml_root_node, NULL, BAD_CAST "preset", NULL);
	xml_node = xmlNewChild(xml_preset_node, NULL, BAD_CAST "name", BAD_CAST "default");

	PortModel* pm = NULL;
	for (NodeModelMap::const_iterator n = patch_model->nodes().begin(); n != patch_model->nodes().end(); ++n) {
		nm = n->second;
		for (PortModelList::const_iterator p = nm->port_models().begin(); p != nm->port_models().end(); ++p) {
			pm = *p;
			if (pm->is_control()) {
				float val = pm->value();
				xml_node = xmlNewChild(xml_preset_node, NULL, BAD_CAST "control",  NULL);
				xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "node-name",
					BAD_CAST nm->name().c_str());
				xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "port-name",
					BAD_CAST ClientPathParser::name(pm->path()).c_str());
				snprintf(temp_buf, temp_buf_length, "%f", val);
				xml_child_node = xmlNewChild(xml_node, NULL, BAD_CAST "value",
					BAD_CAST temp_buf);
			}
		}
	}
	
	xmlSaveFormatFileEnc(filename.c_str(), xml_doc, "UTF-8", 1); // last arg is pretty print

    xmlFreeDoc(xml_doc);
    xmlCleanupParser();
}


/** Load a patch in to the engine (and client) from a patch file.
 *
 * The name and poly from the passed PatchModel are used.  If the name is
 * the empty string, the name will be loaded from the file.  If the poly
 * is 0, it will be loaded from file.  Otherwise the given values will
 * be used.
 *
 * Returns the path of the newly created patch.
 */
string
PatchLibrarian::load_patch(PatchModel* pm)
{
	string filename = pm->filename();
	string patch_name = pm->name();
	uint poly = pm->poly();
	const PatchModel* parent = pm->parent();

	std::cerr << "[PatchLibrarian] Loading patch " << filename << "" << std::endl;

	const size_t temp_buf_length = 255;
	char temp_buf[temp_buf_length];
	
	string name = patch_name;

	bool load_name = (name == "");
	bool load_poly = (poly == 0);
	
	//int last_request_id = 0;
	
	xmlDocPtr doc = xmlParseFile(filename.c_str());

	if (doc == NULL ) {
		std::cerr << "Unable to parse patch file." << std::endl;
		return "";
	}

	xmlNodePtr cur = xmlDocGetRootElement(doc);

	if (cur == NULL) {
		std::cerr << "Empty document." << std::endl;
		xmlFreeDoc(doc);
		return "";
	}

	if (xmlStrcmp(cur->name, (const xmlChar*) "patch")) {
		std::cerr << "File is not an Om patch file, root node != patch" << std::endl;
		xmlFreeDoc(doc);
		return "";
	}

	xmlChar* key = NULL;
	cur = cur->xmlChildrenNode;
	string path;

	// Load Patch attributes
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
		
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"name"))) {
			if (load_name)
				name = (char*)key;
			
			if (parent != NULL) {
				path = parent->path() +"/"+ name;
			} else {
				path = "/"; path += name;
			}
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"polyphony"))) {
			if (load_poly)
				poly = atoi((char*)key);
		//} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"width"))) {
		//	pm->width(atoi((char*)key));
		//} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"height"))) {
		//	pm->height(atoi((char*)key));
		}
		
		xmlFree(key);
		key = NULL; // Avoid a (possible?) double free

		cur = cur->next;
	}

	pm->path(path);
	if (poly == 0) poly = 1;
	pm->poly(poly);

	// Wait until the patch is created or the node creations may fail
	int id = m_comm->get_next_request_id();
	m_comm->set_wait_response_id(id);
	m_comm->create_patch(pm, id);
	bool succeeded = m_comm->wait_for_response();

	// If creating the patch failed, bail out so we don't load all these nodes
	// into an already existing patch
	if (!succeeded)
		return "";
	
	m_comm->set_metadata(pm->path(), "filename",
		pm->filename().substr(pm->filename().find_last_of("/")+1));
	
	// Load nodes
	NodeModel* nm = NULL;
	cur = xmlDocGetRootElement(doc)->xmlChildrenNode;
	
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"node"))) {
			nm = parse_node(pm, doc, cur);
			m_comm->add_node(nm);
			m_comm->set_all_metadata(nm);
			for (list<PortModel*>::const_iterator j = nm->port_models().begin();
					j != nm->port_models().end(); ++j) {
				snprintf(temp_buf, temp_buf_length, "%f", (*j)->user_min());
				m_comm->set_metadata((*j)->path(), "user-min", temp_buf);
				snprintf(temp_buf, temp_buf_length, "%f", (*j)->user_max());
				m_comm->set_metadata((*j)->path(), "user-max", temp_buf);
			}
			nm = NULL;
		}
		usleep(1000);
		cur = cur->next;
	}

	// Load subpatches
	cur = xmlDocGetRootElement(doc)->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"subpatch"))) {
			load_subpatch(pm, doc, cur);
		}
		cur = cur->next;
	}
	
	// Load connections
	ConnectionModel* cm = NULL;
	cur = xmlDocGetRootElement(doc)->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"connection"))) {
			cm = parse_connection(pm, doc, cur);
			assert(cm != NULL);
			m_comm->connect(cm->src_port_path(), cm->dst_port_path());
		}
		usleep(1000);
		cur = cur->next;
	}
	
	
	// Load presets (control values)
	PresetModel* preset_model = NULL;
	cur = xmlDocGetRootElement(doc)->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"preset"))) {
			preset_model = parse_preset(pm, doc, cur);
			assert(preset_model != NULL);
			if (preset_model->name() == "default")
				m_comm->set_preset(pm->path(), preset_model);
		}
		cur = cur->next;
	}
	
	
	// Load random control values
	// (These are deprecated and not written anymore, because of presets.  This will go away soon)
	
	cur = xmlDocGetRootElement(doc)->xmlChildrenNode;
	while (cur != NULL) {
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"control"))) {
			xmlChar *key;
			xmlNodePtr node = cur->xmlChildrenNode;
	
			string node_name = "", port_name = "";
			float val = 0.0;
			
			while (node != NULL) {
				key = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
				
				if ((!xmlStrcmp(node->name, (const xmlChar*)"node-name"))) {
					node_name = (char*)key;
				} else if ((!xmlStrcmp(node->name, (const xmlChar*)"port-name"))) {
					port_name = (char*)key;
				} else if ((!xmlStrcmp(node->name, (const xmlChar*)"value"))) {
					val = atof((char*)key);
				}
				
				xmlFree(key);
				key = NULL; // Avoid a (possible?) double free
		
				node = node->next;
			}
			
			if (node_name == "" || port_name == "") {
				string msg = "Unable to parse control in patch file ( node = ";
				msg.append(node_name).append(", port = ").append(port_name).append(")");
				m_client_hooks->error(msg);
			} else {
				// FIXME: temporary compatibility, remove any slashes from port name
				// remove this soon once patches have migrated
				string::size_type slash_index;
				while ((slash_index = port_name.find("/")) != string::npos)
					port_name[slash_index] = '-';
				m_comm->set_control_slow(pm->path() +"/"+ node_name +"/"+ port_name, val);	
			}
		}
		cur = cur->next;
	}

	xmlFreeDoc(doc);
	xmlCleanupParser();

	m_comm->set_all_metadata(pm);

	m_comm->enable_patch(pm->path());

	string ret = pm->path();
	return ret;
}


/** Build a NodeModel given a pointer to a Node in a patch file.
 */
NodeModel*
PatchLibrarian::parse_node(const PatchModel* parent, xmlDocPtr doc, const xmlNodePtr node)
{
	NodeModel* nm = new NodeModel();
	PluginInfo* pi = new PluginInfo();

	xmlChar *key;
	xmlNodePtr cur = node->xmlChildrenNode;
	
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
		
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"name"))) {
			nm->path(parent->path() + "/" + (char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"polyphonic"))) {
			nm->polyphonic(!strcmp((char*)key, "true"));
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"type"))) {
			pi->set_type((const char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"library-name"))) {
			pi->lib_name((char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"plugin-label"))) {
			pi->plug_label((char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"port"))) {
			xmlNodePtr child = cur->xmlChildrenNode;
			
			PortModel* pm = new PortModel();
			
			while (child != NULL) {
				key = xmlNodeListGetString(doc, child->xmlChildrenNode, 1);
				
				if ((!xmlStrcmp(child->name, (const xmlChar*)"name"))) {
					pm->path(nm->path() +"/"+ (char*)key);
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"user-min"))) {
					pm->user_min(atof((char*)key));
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"user-max"))) {
					pm->user_max(atof((char*)key));
				}
				
				xmlFree(key);
				key = NULL; // Avoid a (possible?) double free
		
				child = child->next;
			}
			nm->add_port_model(pm);
		} else {  // Don't know what this tag is, add it as metadata
			if (key != NULL)
				nm->set_metadata((const char*)cur->name, (const char*)key);
		}
		xmlFree(key);
		key = NULL;

		cur = cur->next;
	}
	
	nm->plugin_info(pi);
	return nm;
}


void
PatchLibrarian::load_subpatch(PatchModel* parent, xmlDocPtr doc, const xmlNodePtr subpatch)
{
	xmlChar *key;
	xmlNodePtr cur = subpatch->xmlChildrenNode;
	
	PatchModel* pm = new PatchModel();
	pm->poly(1);
	pm->parent(parent);
	
	string base_path = parent->filename().substr(0, parent->filename().find_last_of("/")) + "/";
	
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
		
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"name"))) {
			if (parent == NULL)
				pm->path(string("/") + (const char*)key);
			else
				pm->path(parent->path() +"/"+ (const char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"polyphony"))) {
			pm->poly(atoi((const char*)key));
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"filename"))) {
			pm->filename(base_path + (const char*)key);
		} else {  // Don't know what this tag is, add it as metadata
			if (key != NULL)
				pm->set_metadata((const char*)cur->name, (const char*)key);
		}
		xmlFree(key);
		key = NULL;

		cur = cur->next;
	}
	
	load_patch(pm);
}


/** Build a ConnectionModel given a pointer to a connection in a patch file.
 */
ConnectionModel*
PatchLibrarian::parse_connection(const PatchModel* parent, xmlDocPtr doc, const xmlNodePtr node)
{
	//std::cerr << "[PatchLibrarian] Parsing connection..." << std::endl;

	xmlChar *key;
	xmlNodePtr cur = node->xmlChildrenNode;
	
	string source_node, source_port, dest_node, dest_port;
	
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);
		
		if ((!xmlStrcmp(cur->name, (const xmlChar*)"source-node"))) {
			source_node = (char*)key;
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"source-port"))) {
			source_port = (char*)key;
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"destination-node"))) {
			dest_node = (char*)key;
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"destination-port"))) {
			dest_port = (char*)key;
		}
		
		xmlFree(key);
		key = NULL; // Avoid a (possible?) double free

		cur = cur->next;
	}

	// FIXME: temporary compatibility, remove any slashes from port names
	// remove this soon once patches have migrated
	string::size_type slash_index;
	while ((slash_index = source_port.find("/")) != string::npos)
		source_port[slash_index] = '-';

	while ((slash_index = dest_port.find("/")) != string::npos)
		dest_port[slash_index] = '-';
	
	ConnectionModel* cm = new ConnectionModel(parent->path() +"/"+ source_node +"/"+ source_port,
		parent->path() +"/"+ dest_node +"/"+ dest_port);
	
	return cm;
}


/** Build a PresetModel given a pointer to a preset in a patch file.
 */
PresetModel*
PatchLibrarian::parse_preset(const PatchModel* patch, xmlDocPtr doc, const xmlNodePtr node)
{
	xmlNodePtr cur = node->xmlChildrenNode;
	xmlChar* key;

	PresetModel* pm = new PresetModel(patch->path());
	
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);

		if ((!xmlStrcmp(cur->name, (const xmlChar*)"name"))) {
			pm->name((char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"control"))) {
			xmlNodePtr child = cur->xmlChildrenNode;
	
			string node_name = "", port_name = "";
			float val = 0.0;
			
			while (child != NULL) {
				key = xmlNodeListGetString(doc, child->xmlChildrenNode, 1);
				
				if ((!xmlStrcmp(child->name, (const xmlChar*)"node-name"))) {
					node_name = (char*)key;
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"port-name"))) {
					port_name = (char*)key;
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"value"))) {
					val = atof((char*)key);
				}
				
				xmlFree(key);
				key = NULL; // Avoid a (possible?) double free
		
				child = child->next;
			}
			
			if (node_name == "" || port_name == "") {
				string msg = "Unable to parse control in patch file ( node = ";
				msg.append(node_name).append(", port = ").append(port_name).append(")");
				m_client_hooks->error(msg);
			} else {
				// FIXME: temporary compatibility, remove any slashes from port name
				// remove this soon once patches have migrated
				string::size_type slash_index;
				while ((slash_index = port_name.find("/")) != string::npos)
					port_name[slash_index] = '-';
				pm->add_control(node_name, port_name, val);
			}
		}
		xmlFree(key);
		key = NULL;
		cur = cur->next;
	}
	if (pm->name() == "") {
		m_client_hooks->error("Preset in patch file has no name.");
		pm->name("Unnamed");
	}

	return pm;
}


/** Build a ControlMapModel given a pointer to a <control-map> node in a patch file.
 */
/*
ControlMapModel*
PatchLibrarian::parse_control_map(xmlDocPtr doc, const xmlNodePtr node, const string& patch_name)
{
	xmlNodePtr cur = node->xmlChildrenNode;
	xmlChar* key;

	ControlMapModel* cm = new ControlMapModel(patch_name);
	
	while (cur != NULL) {
		key = xmlNodeListGetString(doc, cur->xmlChildrenNode, 1);

		if ((!xmlStrcmp(cur->name, (const xmlChar*)"name"))) {
			cm->name((char*)key);
		} else if ((!xmlStrcmp(cur->name, (const xmlChar*)"control"))) {
			xmlNodePtr child = cur->xmlChildrenNode;
	
			// FIXME: if this ever gets uncommented, remove the NANs!
			
			string node_name = "", port_name = "";
			int channel = -1, controller = -1;
			float min = NAN, max = NAN;
			
			while (child != NULL) {
				key = xmlNodeListGetString(doc, child->xmlChildrenNode, 1);
				
				if ((!xmlStrcmp(child->name, (const xmlChar*)"node-name"))) {
					node_name = (char*)key;
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"port-name"))) {
					port_name = (char*)key;
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"channel"))) {
					channel = atoi((char*)key);
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"controller"))) {
					controller = atoi((char*)key);
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"min"))) {
					min = atof((char*)key);
				} else if ((!xmlStrcmp(child->name, (const xmlChar*)"max"))) {
					max = atof((char*)key);
				}
				
				xmlFree(key);
				key = NULL; // Avoid a (possible?) double free
		
				child = child->next;
			}
			
			if (node_name == "" || port_name == "" || channel == -1 || controller == -1 || min == NAN || max == NAN) {
				m_client_hooks->error("Unable to parse midi binding in patch file.");
			} else {
				cm->add_midi_binding(node_name, port_name, channel, controller, min, max);
			}
		}
		xmlFree(key);
		key = NULL;
		cur = cur->next;
	}
	if (cm->name() == "") {
		m_client_hooks->error("ControlMap in patch file has no name.");
		cm->name("Unnamed");
	}

	return cm;
}*/

} // namespace LibOmClient
