/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OmGtk.h"
#include "OmGtkApp.h"
#include "PatchModel.h"
#include "PatchController.h"
#include "PatchWindow.h"
#include "Comm.h"
#include "GtkClientHooks.h"
#include "PatchLibrarian.h"
#include "Controller.h"

namespace OmGtk {


// LoadPatchEvent //

void
LoadPatchEvent::execute()
{ m_patch_librarian->load_patch(m_patch_model); }



// SavePatchEvent //


void
SavePatchEvent::execute()
{ m_patch_librarian->save_patch(m_patch_model, m_filename); }



// Controller //


Controller::Controller()
: m_client_hooks(new GtkClientHooks()),
  m_comm(new Comm(m_client_hooks)),
  m_patch_librarian(new PatchLibrarian(m_comm, m_client_hooks)),
  m_is_attached(false)
{
}


Controller::~Controller()
{
	delete m_patch_librarian;
	delete m_comm;
	delete m_client_hooks;
}


/** "Attach" to the Om engine.
 *
 * See documentation for Comm::attach.
 */
void
Controller::attach(const string& engine_host, int engine_port, int client_port)
{
	m_comm->attach(engine_host, engine_port, client_port);
	m_is_attached = true;
	activate_engine();
	enable_engine();
}


/** "Attach" to the Om engine.
 *
 * See documentation for Comm::attach.
 */
void
Controller::attach_url(const string& engine_url, int client_port)
{
	m_comm->attach_url(engine_url, client_port);
	m_is_attached = true;
	activate_engine();
	enable_engine();
}

void
Controller::register_client(const string& client_host, int client_port)
{
	m_comm->register_client(client_host, client_port);
	int id = m_comm->get_next_request_id();
	m_comm->set_wait_response_id(id);
	m_comm->load_plugins(id);
	m_comm->wait_for_response();
}

string Controller::engine_url() { return m_comm->engine_url(); }

void Controller::activate_engine()   { m_comm->activate(); }
void Controller::deactivate_engine() { m_comm->deactivate(); }
void Controller::enable_engine()     { m_comm->enable(); }
void Controller::disable_engine()    { m_comm->disable(); }

void Controller::quit() { m_comm->quit(); }

void
Controller::set_control(const string& port_path, float val)
	{ m_comm->set_control(port_path, val); }

void
Controller::set_control(const string& port_path, int voice, float val)
	{ m_comm->set_control(port_path, voice, val); }
	
void
Controller::request_control(const string& port_path)
	{ m_comm->request_control(port_path); }

void
Controller::add_node(NodeModel* nm)
{
	push_added_node(nm);
	m_comm->add_node(nm);
	char temp_buf[16];
	snprintf(temp_buf, 16, "%f", nm->x());
	m_comm->set_metadata(nm->path(), "module-x", temp_buf);
	snprintf(temp_buf, 16, "%f", nm->y());
	m_comm->set_metadata(nm->path(), "module-y", temp_buf);
}


void
Controller::remove_node(const string& node_path)
	{ m_comm->remove_node(node_path); }


void Controller::connect(const string& port1_path, const string& port2_path)
	{ m_comm->connect(port1_path, port2_path); }
	
void Controller::disconnect(const string& port1_path, const string& port2_path)
	{ m_comm->disconnect(port1_path, port2_path); }

void Controller::disconnect_all(const string& node_path)
	{ m_comm->disconnect_all(node_path); }

void
Controller::create_patch(PatchModel* pm)
{ 
	if (pm->parent() != NULL)
		push_added_patch(pm);

	//int id = m_comm->get_next_request_id();
	//m_comm->set_wait_response_id(id);
	m_comm->create_patch(pm);
	if (pm->parent() != NULL) {
	//	m_comm->wait_for_response();
		char temp_buf[16];
		snprintf(temp_buf, 16, "%f", pm->x());
		m_comm->set_metadata(pm->path(), "module-x", temp_buf);
		snprintf(temp_buf, 16, "%f", pm->y());
		m_comm->set_metadata(pm->path(), "module-y", temp_buf);
	}
	m_comm->enable_patch(pm->path());
}


void
Controller::destroy_patch(const string& path)
	{ m_comm->destroy_patch(path); }

void
Controller::enable_patch(const string& path)
	{ m_comm->enable_patch(path); }

void
Controller::disable_patch(const string& path)
	{ m_comm->disable_patch(path); }
	
void
Controller::midi_learn(const string& node_path)
	{ m_comm->midi_learn(node_path); }

void
Controller::set_metadata(const string& obj_path, const string& key, const string& value)
	{ m_comm->set_metadata(obj_path, key, value); }


void
Controller::request_plugins()
	{ m_comm->request_plugins(); }


void
Controller::request_all_objects()
	{ m_comm->request_all_objects(); }


void
Controller::load_patch(PatchModel* model)
{
	push_added_patch(model);
	m_thread.set_event(new LoadPatchEvent(m_patch_librarian, model));
	m_thread.signal();
}


void
Controller::save_patch(const PatchModel* model, const string& filename)
{
	m_thread.set_event(new SavePatchEvent(m_patch_librarian, model, filename));
	m_thread.signal();
}


/** Returns the added node with the given path and removes it from the cache.
 */
NodeModel*
Controller::yank_added_node(const string& path)
{
	NodeModel* nm = NULL;
	
	for (list<NodeModel*>::iterator i = m_added_nodes.begin(); i != m_added_nodes.end(); ++i) {
		if ((*i)->path() == path) {
			nm = *i;
			m_added_nodes.erase(i);
			break;
		}
	}

	return nm;
}


/** Returns the added patch with the given path and removes it from the cache.
 */
PatchModel*
Controller::yank_added_patch(const string& path)
{
	PatchModel* pm = NULL;
	
	for (list<PatchModel*>::iterator i = m_added_patches.begin(); i != m_added_patches.end(); ++i) {
		if ((*i)->path() == path) {
			pm = *i;
			m_added_patches.erase(i);
			break;
		}
	}

	return pm;
}


//////// ControllerThread //////////


ControllerThread::ControllerThread()
: m_controller_thread_exit_flag(false),
  m_event(NULL)
{
	pthread_mutex_init(&m_mutex, NULL);
	pthread_cond_init(&m_cond, NULL);
}


void
ControllerThread::launch()
{
	pthread_create(&m_thread, NULL, ControllerThread::thread_function, this);
}


void
ControllerThread::signal()
{
	pthread_mutex_lock(&m_mutex);
	pthread_cond_signal(&m_cond);
	pthread_mutex_unlock(&m_mutex);
}


void*
ControllerThread::thread_function(void* me)
{
	ControllerThread* ct = static_cast<ControllerThread*>(me);
	return ct->m_thread_function(NULL);
}


void*
ControllerThread::m_thread_function(void *)
{
	while ( ! m_controller_thread_exit_flag) {
		pthread_mutex_lock(&m_mutex);
		pthread_cond_wait(&m_cond, &m_mutex);
		ControllerEvent* ev = m_event;
		m_event = NULL;
		pthread_mutex_unlock(&m_mutex);
		
		ev->execute();
		delete ev;
	}

	pthread_exit(NULL);
	return NULL;
}


} // namespace OmGtk
