/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GtkClientHooks.h"
#include <gtkmm.h>
#include <string>
#include "GtkClientHooksEvents.h"
#include <sys/time.h>
#include <time.h>


namespace OmGtk {

GtkClientHooks::GtkClientHooks()
//: m_events(2048) // FIXME: size?
{
	pthread_mutex_init(&m_event_queue_mutex, NULL);
}


GtkClientHooks::~GtkClientHooks()
{
	pthread_mutex_destroy(&m_event_queue_mutex);
}


/** Process all queued events that came from the OSC thread.
 *
 * This function is to be called from the Gtk thread only.
 */
void
GtkClientHooks::process_events()
{
	// Maximum time to spend processing events, in ms
	static const float MAX_TIME = 20;
	
	pthread_mutex_lock(&m_event_queue_mutex);
	Event* ev = NULL;

	timeval start;
	gettimeofday(&start, NULL);
	timeval now;
	
	while ( ! m_events.empty()) {
		ev = m_events.front();
		m_events.pop();
		//ev = m_events.pop();

		// Unlock the mutex to avoid locking OSC thread while we execute
		pthread_mutex_unlock(&m_event_queue_mutex);
		
		ev->execute();
		delete ev;

		// Relock mutex to protect queue access
		pthread_mutex_lock(&m_event_queue_mutex);

		gettimeofday(&now, NULL);
		float ms_elapsed = ((now.tv_sec - start.tv_sec)
			+ (now.tv_usec - start.tv_usec) * 0.000001f) * 1000.0f;

		if (ms_elapsed > MAX_TIME)
			break;
	}
	
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push an error event onto the queue.
 */
void
GtkClientHooks::error(const string& msg)
{
	ErrorEvent* ev = new ErrorEvent(msg);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push an engine enabled event onto the queue.
 */
void
GtkClientHooks::engine_enabled()
{
	EngineEnabledEvent* ev = new EngineEnabledEvent();
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push an engine disabled event onto the queue.
 */
void
GtkClientHooks::engine_disabled()
{
	EngineDisabledEvent* ev = new EngineDisabledEvent();
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a new patch event onto the queue.
 */
void
GtkClientHooks::new_patch(PatchModel* pm)
{
	NewPatchEvent* ev = new NewPatchEvent(pm);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a new port event onto the queue.
 */
void
GtkClientHooks::new_port(PortModel* port_model)
{
	NewPortEvent* ev = new NewPortEvent(port_model);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a port removal event onto the queue.
 */
void
GtkClientHooks::port_removal(const string& path)
{
	PortRemovalEvent* ev = new PortRemovalEvent(path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a patch destruction event onto the queue.
 */
void
GtkClientHooks::patch_destruction(const string& path)
{
	PatchDestructionEvent* ev = new PatchDestructionEvent(path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a patch enabled event onto the queue.
 */
void
GtkClientHooks::patch_enabled(const string& path)
{
	PatchEnabledEvent* ev = new PatchEnabledEvent(path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a patch disabled event onto the queue.
 */
void
GtkClientHooks::patch_disabled(const string& path)
{
	PatchDisabledEvent* ev = new PatchDisabledEvent(path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a new node event onto the queue.
 */
void
GtkClientHooks::new_node(NodeModel* nm)
{
	NewNodeEvent* ev = new NewNodeEvent(nm);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a node removal event onto the queue.
 */
void
GtkClientHooks::node_removal(const string& path)
{
	NodeRemovalEvent* ev = new NodeRemovalEvent(path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a new connection event onto the queue.
 */
void
GtkClientHooks::connection(ConnectionModel* cm)
{
	ConnectionEvent* ev = new ConnectionEvent(cm);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a disconnection event onto the queue.
 */
void
GtkClientHooks::disconnection(const string& src_port_path, const string& dst_port_path)
{
	DisconnectionEvent* ev = new DisconnectionEvent(src_port_path, dst_port_path);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a metadata_update event onto the queue.
 */
void
GtkClientHooks::metadata_update(MetadataModel* mm)
{
	MetadataUpdateEvent* ev = new MetadataUpdateEvent(mm);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a control_change event onto the queue.
 */
void
GtkClientHooks::control_change(ControlModel* cm)
{
	ControlChangeEvent* ev = new ControlChangeEvent(cm);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a new_plugin event onto the queue.
 */
void
GtkClientHooks::new_plugin(PluginInfo* pi)
{
	NewPluginEvent* ev = new NewPluginEvent(pi);
	
	pthread_mutex_lock(&m_event_queue_mutex);
	m_events.push(ev);
	pthread_mutex_unlock(&m_event_queue_mutex);
}


} // namespace OmGtk
