/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OmGtkApp.h"
#include <cassert>
#include <string>
#include <libgnomecanvasmm.h>
#include <libglademm/xml.h>
#include <algorithm>
#include <fstream>
#include <pthread.h>
#include "OmGtk.h"
#include "PatchLibrarian.h"
#include "PatchController.h"
#include "LoadPluginWindow.h"
#include "PatchWindow.h"
#include "NewPatchWindow.h"
#include "LoadPatchWindow.h"
#include "Controller.h"
#include "GtkClientHooks.h"
using std::cerr; using std::cout; using std::endl;
using std::string;

namespace OmGtk {


OmGtkApp::OmGtkApp()
{
	Glib::RefPtr<Gnome::Glade::Xml> glade_xml;

	//m_settings_filename = getenv("HOME");
	//m_settings_filename += "/.omrc";
	//m_state_manager = new OmGtkStateManager();
	//m_state_manager->load(m_settings_filename);

	
	// Check for the .glade file in current directory
	m_glade_filename = "./om_gtk.glade";
	std::ifstream fs(m_glade_filename.c_str());
	if (fs.fail()) { // didn't find it, check PKGDATADIR
		fs.clear();
		m_glade_filename = PKGDATADIR;
		m_glade_filename += "/om_gtk.glade";
	
		fs.open(m_glade_filename.c_str());
		if (fs.fail()) {
			std::cerr << "Unable to find om_gtk.glade in current directory or " << PKGDATADIR << "." << std::endl;
			throw;
		}
	}

	try {
		glade_xml = Gnome::Glade::Xml::create(m_glade_filename);
	} catch(const Gnome::Glade::XmlError& ex) {
		std::cerr << ex.what() << std::endl;
		throw;
	}

	glade_xml->get_widget("main_win", m_main_window);
	glade_xml->get_widget_derived("new_patch_win", m_new_patch_window);
	glade_xml->get_widget_derived("load_patch_win", m_load_patch_window);
	glade_xml->get_widget("about_win", m_about_window);
	
	glade_xml->get_widget("main_file_open_patch_menuitem", m_menu_file_open_patch);
	glade_xml->get_widget("main_file_new_patch_menuitem", m_menu_file_new_patch);
	glade_xml->get_widget("main_file_quit_nokill_menuitem", m_menu_file_quit);
	glade_xml->get_widget("main_file_quit_and_kill_menuitem", m_menu_file_quit_and_kill);
	glade_xml->get_widget("help_about_menuitem", m_menu_help_about);
	
	glade_xml->get_widget("main_process_checkbutton", m_process_checkbutton);
	glade_xml->get_widget("about_close_button", m_about_close_button);
	glade_xml->get_widget("main_messages_textview", m_messages_textview);
	glade_xml->get_widget("main_patch_window_selector", m_patch_selector);
	glade_xml->get_widget("engine_error_dialog", m_engine_error_dialog);
	glade_xml->get_widget("engine_error_dialog_close_button", m_engine_error_close_button);
	
	m_main_window->show();

	m_patch_selector_list_store = Gtk::ListStore::create(m_patch_selector_columns);
	m_patch_selector->set_model(m_patch_selector_list_store);

	Gtk::CellRendererText* crt;
    m_patch_selector->pack_start(*manage(crt = new Gtk::CellRendererText()), true);
    m_patch_selector->add_attribute(crt->property_text(), m_patch_selector_columns.path_col);


	//m_main_window->resize(m_state_manager->get_window_size().x, m_state_manager->get_window_size().y);
	//m_main_window->move(m_state_manager->get_window_location().x, m_state_manager->get_window_location().y);

	// Idle callback, process events and whatnot
	// Gtk refreshes at priority G_PRIORITY_HIGH_IDLE+20, so do this stuff at a slightly
	// lower priority so added nodes will be drawn during patch loading etc.
	Glib::signal_timeout().connect(
		sigc::mem_fun(this, &OmGtkApp::idle_callback), 20, G_PRIORITY_HIGH_IDLE+20);
	
	m_menu_file_new_patch->signal_activate().connect(         sigc::mem_fun(this, &OmGtkApp::event_new_patch));
	m_menu_file_open_patch->signal_activate().connect(        sigc::mem_fun(this, &OmGtkApp::event_open_patch));
	m_menu_file_quit->signal_activate().connect(              sigc::mem_fun(this, &OmGtkApp::event_quit));
	m_menu_file_quit_and_kill->signal_activate().connect(     sigc::mem_fun(this, &OmGtkApp::event_quit_and_kill));
	m_process_checkbutton->signal_toggled().connect(          sigc::mem_fun(this, &OmGtkApp::event_process_toggled));
	m_menu_help_about->signal_activate().connect(             sigc::mem_fun(this, &OmGtkApp::event_show_about));
	m_patch_selector->signal_changed().connect(               sigc::mem_fun(this, &OmGtkApp::event_patch_selected));   
	m_about_close_button->signal_clicked().connect(           sigc::mem_fun(m_about_window, &Gtk::Window::hide));
	m_engine_error_close_button->signal_clicked().connect(sigc::mem_fun(m_engine_error_dialog, &Gtk::Dialog::hide));
	controller->request_plugins();
	controller->request_all_objects();
}


OmGtkApp::~OmGtkApp()
{
	//delete m_load_plugin_window;
}


Glib::RefPtr<Gnome::Glade::Xml>
OmGtkApp::new_glade_reference(const string& toplevel_widget = "")
{
	Glib::RefPtr<Gnome::Glade::Xml> glade_xml;

	try {
		if (toplevel_widget == "")
			glade_xml = Gnome::Glade::Xml::create(m_glade_filename);
		else
			glade_xml = Gnome::Glade::Xml::create(m_glade_filename, toplevel_widget.c_str());
	} catch(const Gnome::Glade::XmlError& ex) {
		std::cerr << ex.what() << std::endl;
		throw;
	}

	return glade_xml;
}


void
OmGtkApp::engine_enabled(bool e)
{
	m_process_checkbutton->set_active(e);
}


PatchWindow*
OmGtkApp::add_patch_window(PatchModel* pm, bool show)
{
	Glib::RefPtr<Gnome::Glade::Xml> xml = new_glade_reference();

	PatchWindow* pw = NULL;
	xml->get_widget_derived("patch_win", pw);
	assert(pw != NULL);
	pw->patch_model(pm);
	m_patch_windows[pm->path()] = pw;
	
	// Add to patch selector
	Gtk::TreeModel::iterator iter = m_patch_selector_list_store->append();
	Gtk::TreeModel::Row row = *iter;
	row[m_patch_selector_columns.path_col] = pm->path();
	row[m_patch_selector_columns.patch_window_col] = pw;
	//m_patch_selector->set_active(row);

	// Only show top level patches immediately
	if (show)
		pw->show();

	return pw;
}


void
OmGtkApp::destroy_patch_window(const string& path)
{
	/*vector<PatchWindow*>::iterator w = m_patch_windows.begin();
	for ( ; w != m_patch_windows.end(); ++w)
		if ((*w)->patch_controller()->model()->path() == path)
			break;
	*/
	map<string, PatchWindow*>::iterator w = m_patch_windows.find(path);
	
	if (w != m_patch_windows.end()) {
		m_patch_windows.erase(w);
		delete (*w).second;
	} else {
		cerr << "[OmGtkApp::destroy_patch_window] Unable to find patch window " << path << endl;
	}

	// Remove from patch selector
	Gtk::TreeModel::Children children = m_patch_selector_list_store->children();
	Gtk::TreeModel::Children::iterator c = children.begin();
	for ( ; c != children.end(); ++c)
		if ((*c)[m_patch_selector_columns.path_col] == path) break;

	if (c != children.end()) {
		m_patch_selector_list_store->erase(c);
	}
}


bool
OmGtkApp::idle_callback()
{
	assert(controller != NULL);
	assert(controller->client_hooks() != NULL);
	
	controller->client_hooks()->process_events();

	return true;
}


void
OmGtkApp::status_message(const string& msg)
{
	Glib::RefPtr<Gtk::TextBuffer> text_buf = m_messages_textview->get_buffer();
	text_buf->insert(text_buf->end(), "* ");
	text_buf->insert(text_buf->end(), msg);
	text_buf->insert(text_buf->end(), "\n");
}


void
OmGtkApp::error_message(const string& msg)
{
	Glib::RefPtr<Gtk::TextBuffer> text_buf = m_messages_textview->get_buffer();
	text_buf->insert(text_buf->end(), "ERROR: ");
	text_buf->insert(text_buf->end(), msg);
	text_buf->insert(text_buf->end(), "\n");
}


void
OmGtkApp::new_plugin(const PluginInfo* pi)
{
	LoadPluginWindow* lpw = NULL;
	
	// I don't like this, it's too slow (this function gets called a LOT of
	// times VERY fast on initial connection), but it's needed to avoid
	// duplicate plugins appearing :/
	// Update: fixed engine to not load a plugin twice, still possible to send them to this
	// client multiple times if something screws up though...
	/*for (list<const PluginInfo*>::iterator i = m_plugin_infos.begin(); i != m_plugin_infos.end(); ++i)
		if ((*i)->lib_name() == pi->lib_name() && (*i)->plug_label() == pi->plug_label()) {
			cerr << "Duplicate plugin received (" << pi->lib_name() << ":" << pi->plug_label() << endl;
			return;
		}
		*/
		
	m_plugin_infos.push_back(pi);
	for (map<string, PatchWindow*>::iterator i = m_patch_windows.begin(); i != m_patch_windows.end(); ++i) {
		lpw = (*i).second->load_plugin_window();
		if (lpw != NULL && lpw->has_shown())
			lpw->add_plugin(pi);
	}

	//cout << "Adding plugin " << pi->plug_label() << " (" << m_plugin_infos.size() << " plugins)" << endl;
}


/******** Event Handlers ************/



void
OmGtkApp::event_new_patch()
{
	m_new_patch_window->show();
}


void
OmGtkApp::event_open_patch()
{
	m_load_patch_window->show();
}


void
OmGtkApp::event_quit()
{
	m_main_window->hide();
}

void
OmGtkApp::event_quit_and_kill()
{
	controller->quit();
	m_main_window->hide();
}


void
OmGtkApp::event_process_toggled() 
{
	if (m_process_checkbutton->get_active()) {
		std::cerr << "Enabling.\n";
		controller->enable_engine();
	} else {
		std::cerr << "Disabling.\n";
		controller->disable_engine();
	}
}


void
OmGtkApp::event_patch_selected()
{
	Gtk::TreeModel::iterator active = m_patch_selector->get_active();
	Gtk::TreeModel::Row row = *active;
	PatchWindow* pw = row[m_patch_selector_columns.patch_window_col];
	if (pw != NULL) {
		pw->show();
		pw->raise();
	}
}


void
OmGtkApp::event_show_about()
{
	m_about_window->show();
}


} // namespace OmGtk

