/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PatchController.h"
#include "OmGtk.h"
#include "OmGtkApp.h"
#include "OmGtkStateManager.h"
#include "ClientPathParser.h"
#include "ConnectionModel.h"
#include "OmPatchBayArea.h"
#include "canvas/Module.h"
#include "PatchModel.h"
#include "Controller.h"
#include "PluginModule.h"
#include "SubpatchModule.h"
#include "DSSIModule.h"
#include "PatchWindow.h"
#include "NodeModel.h"
#include "OmModule.h"
#include "OmPort.h"
#include "MetadataModel.h"
#include "ControlModel.h"
#include "PluginInfo.h"
#include <cassert>
#include <cstdlib>


using std::cerr; using std::cout; using std::endl;
using Om::PluginInfo;

namespace OmGtk {


PatchController::PatchController(PatchModel* patch_model, Controller* controller)
: m_patch_model(patch_model),
  m_controller(controller)
{
}


/** Adds a node to the patch bay.
 */
void
PatchController::new_node(NodeModel* const nm)
{
	if (m_patch_model->get_node(nm->name()) != NULL) {
		// Patch already exists, ignore
		delete nm;
	} else {
		m_patch_model->add_node(nm);

		PluginModule* module = NULL;
		
		if (nm->plugin_info()->type() == Om::PluginInfo::DSSI)
			module = new DSSIModule(m_patch_bay, nm, this);
		else
			module = new PluginModule(m_patch_bay, nm, this);
	
		for (PortModelList::const_iterator i = nm->port_models().begin(); i != nm->port_models().end(); ++i)
			module->add_port(*i, false);
	
		module->resize();

		m_patch_bay->add_module(module);
	}
}


/** Adds a subpatch to the patch bay.
 */
void
PatchController::new_subpatch(PatchWindow* const pw)
{
	PatchModel* pm = pw->patch_controller()->model();
	
	if (m_patch_model->get_node(pm->name()) != NULL) {
		delete pm;
	} else {
		assert(pm->parent() != NULL);
		assert(pm->parent() == m_patch_model);
		m_patch_model->add_node(pm);
		
		NodeModel* nm2 = controller->yank_added_node(pm->path());
		if (nm2 != NULL) {
			pm->x(nm2->x());
			pm->y(nm2->y());
		}
		// FIXME: delete nm2?  

		SubpatchModule* module = new SubpatchModule(m_patch_bay, pw, this);
		module->property_x() = pm->x();
		module->property_y() = pm->y();
		m_patch_bay->add_module(module);
	}
}


/** Adds a port to a node in the patch bay.
 */
void
PatchController::new_port(PortModel* const pm)
{
	cerr << "[PatchController] (" << m_patch_model->path() << "): new port - "
		<< pm->path() << endl;
																			
	string parent_path = ClientPathParser::parent(pm->path());
	string node_name = ClientPathParser::name(parent_path);

	OmModule* m = (OmModule*)m_patch_bay->find_module(node_name);
	
	if (m == NULL) {
		cerr << "[PatchController::new_port] WARNING: " << parent_path << " does not exist." << endl;
	} else if (m->port(pm->name()) != NULL) {
		//cerr << "[PatchController::new_port] " << pm->path() << " already exists." << endl;
	} else {
		m->node_model()->add_port_model(pm);
		m->add_port(pm, true);
	}
}


/** Removed a port from a node in the patch bay.
 */
void
PatchController::remove_port(const string& path)
{
	string parent_path = path.substr(0, path.find_last_of("/"));
	string node_name = parent_path.substr(parent_path.find_last_of("/")+1);
	string port_name = path.substr(path.find_last_of("/")+1);
	
	OmModule* m = (OmModule*)m_patch_bay->find_module(node_name);
	
	if (m == NULL) {
		cerr << "[PatchController::new_port] WARNING: " << parent_path << " does not exist." << endl;
	} else if (m->port(port_name) == NULL) {
		cerr << "[PatchController::remove_port] " << path << " does not exist." << endl;
	} else {
		m->remove_port(port_name);
	}
}


void
PatchController::node_removal(const string& name)
{
	m_patch_bay->remove_module(name);
	m_patch_model->remove_node(name);
}


void
PatchController::connection(ConnectionModel* const cm)
{
	//std::cerr << "[PatchController] Connection: " << port1_name << " -> " << port2_name << std::endl;
	m_patch_bay->add_connection(cm->src_node_name(), cm->src_port_name(), cm->dst_node_name(), cm->dst_port_name());
	m_patch_model->add_connection(cm);
}



void
PatchController::disconnection(const string& src_port_path, const string& dst_port_path)
{
	m_patch_bay->remove_connection(
		ClientPathParser::name(ClientPathParser::parent(src_port_path)),
		ClientPathParser::name(src_port_path),
		ClientPathParser::name(ClientPathParser::parent(dst_port_path)),
		ClientPathParser::name(dst_port_path));
	m_patch_model->remove_connection(src_port_path, dst_port_path);
}


void
PatchController::metadata_update(MetadataModel* const mm)
{
	//cerr << "[PatchController] " << m_patch_model->path() << ": metadata update - "
	//	<< mm->key() << mm->path() << endl;

	OmModule* m = (OmModule*)m_patch_bay->find_module(
		ClientPathParser::name(mm->path()));

	if (m != NULL) {  // module
		m->metadata_update(mm);
	} else {  // maybe a port on a module...
		m = (OmModule*)m_patch_bay->find_module(
			ClientPathParser::name(ClientPathParser::parent(mm->path())));
		if (m != NULL)
			m->port_metadata_update(mm);
	}

	// Port on this patch
	
}


void
PatchController::control_change(ControlModel* const cm)
{
	//cerr << "[PatchController] Got control change: " << cm->port_path()
	//	<< " = " << cm->value() << endl;

	string parent_path = ClientPathParser::parent(cm->port_path());
	
	OmModule* m = (OmModule*)m_patch_bay->find_module(
		ClientPathParser::name(parent_path));

	if (m == NULL) {
		cerr << "[PatchController::control_change] Unable to find module " << parent_path << endl;
	} else {
		m->control_change(cm);
	}
}


void
PatchController::get_new_module_location(int& x, int& y)
{
	int scroll_x;
	int scroll_y;
	m_patch_bay->get_scroll_offsets(scroll_x, scroll_y);

	x = scroll_x + 20;
	y = scroll_y + 20;
}


} // namespace OmGtk
