/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PatchWindow.h"
#include <iostream>
#include <cassert>
#include <fstream>
#include "OmGtk.h"
#include "OmGtkStateManager.h"
#include "OmPatchBayArea.h"
#include "PatchController.h"
#include "LoadPluginWindow.h"
#include "PatchModel.h"
#include "PortModel.h"
#include "MetadataModel.h"
#include "AddSubpatchWindow.h"
#include "LoadSubpatchWindow.h"
#include "NodeControlWindow.h"
#include "Controller.h"
#include "ClientPathParser.h"
#include "ControlModel.h"
#include "OmPort.h"
#include "OmGtkApp.h"

namespace OmGtk {


PatchWindow::PatchWindow(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& xml)
: Gtk::Window(cobject),
  m_patch_controller(NULL),
  m_patch_bay(NULL),
  m_load_plugin_window(NULL),
  m_new_subpatch_window(NULL),
  m_control_window(NULL)
{
	property_visible() = false;

	xml->get_widget("patch_file_save_patch_menuitem", m_menu_file_save_patch);
	xml->get_widget("patch_file_save_patch_as_menuitem", m_menu_file_save_patch_as);
	xml->get_widget("patch_file_close_menuitem", m_menu_file_close);
	//xml->get_widget("patch_view_refresh_menuitem", m_menu_view_refresh);
	xml->get_widget("patch_view_control_window_menuitem", m_menu_view_control_window);
	xml->get_widget("patch_add_plugin_menuitem", m_menu_add_plugin);
	xml->get_widget("patch_add_new_subpatch_menuitem", m_menu_new_subpatch);
	xml->get_widget("patch_add_subpatch_from_file_menuitem", m_menu_load_subpatch);
	xml->get_widget("patch_canvas_scrolledwindow", m_canvas_scrolledwindow);
	xml->get_widget("patch_zoom_scale", m_zoom_slider);
	xml->get_widget("patch_destroy_button", m_destroy_button);
	xml->get_widget("patch_polyphony_label", m_polyphony_label);
	xml->get_widget("patch_process_checkbutton", m_process_checkbutton);
	
	xml->get_widget_derived("load_plugin_win", m_load_plugin_window);
	xml->get_widget_derived("new_subpatch_win", m_new_subpatch_window);
	xml->get_widget_derived("load_subpatch_win", m_load_subpatch_window);

	m_menu_file_save_patch->signal_activate().connect(    sigc::mem_fun(this, &PatchWindow::event_save_patch));
	m_menu_file_save_patch_as->signal_activate().connect( sigc::mem_fun(this, &PatchWindow::event_save_patch_as));
	m_menu_file_close->signal_activate().connect(         sigc::mem_fun(this, &PatchWindow::event_close));
	//m_menu_view_refresh->signal_activate().connect(       sigc::mem_fun(this, &PatchWindow::event_refresh));
	m_menu_view_control_window->signal_activate().connect(sigc::mem_fun(this, &PatchWindow::show_control_window));
	m_menu_add_plugin->signal_activate().connect(         sigc::mem_fun(this, &PatchWindow::event_show_add_plugin_win));
	m_menu_new_subpatch->signal_activate().connect(       sigc::mem_fun(this, &PatchWindow::event_show_new_subpatch_win));
	m_menu_load_subpatch->signal_activate().connect(      sigc::mem_fun(this, &PatchWindow::event_show_load_subpatch_win));
	m_zoom_slider->signal_value_changed().connect(        sigc::mem_fun(this, &PatchWindow::zoom_changed));
	m_destroy_button->signal_clicked().connect(           sigc::mem_fun(this, &PatchWindow::destroy_clicked));
	m_process_checkbutton->signal_toggled().connect(      sigc::mem_fun(this, &PatchWindow::process_toggled));
}


PatchWindow::~PatchWindow()
{
	if (m_control_window != NULL) {
		m_control_window->hide();
		delete m_control_window;
	}
	
	hide();
	
	delete m_patch_controller;
	delete m_patch_bay;
	delete m_load_plugin_window;
	delete m_new_subpatch_window;
	delete m_load_subpatch_window;
}


/** Sets the patch model for this window and initializes everything.
 *
 * This function MUST be called before using the window in any way!
 */
void
PatchWindow::patch_model(PatchModel* pm)
{
	m_patch_controller = new PatchController(pm, controller);
	m_patch_bay = new OmPatchBayArea(pm->path(), 1600*2, 1200*2);
	
	assert(m_load_plugin_window != NULL);
	assert(m_new_subpatch_window != NULL);
	assert(m_load_subpatch_window != NULL);

	m_load_plugin_window->patch_controller(m_patch_controller);
	m_new_subpatch_window->patch_controller(m_patch_controller);
	m_load_subpatch_window->patch_controller(m_patch_controller);

	m_canvas_scrolledwindow->add(*m_patch_bay);
	int width, height;
	get_size(width, height);
	m_patch_bay->scroll_to(
		((int)m_patch_bay->width() - width)/2,
		((int)m_patch_bay->height() - height)/2);

	m_patch_bay->show();
	m_canvas_scrolledwindow->show();

	m_patch_controller->patch_bay(m_patch_bay);
	assert(m_patch_controller->model() != NULL);

	/*m_patch_controller->model()->width(1600*2);
	m_patch_controller->model()->height(1200*2);*/

	Glib::ustring title = "Om - ";
	title += m_patch_controller->model()->path();
	property_title() = title;

	char txt[8];
	snprintf(txt, 8, "%d", pm->poly());
	m_polyphony_label->set_text(txt);

	m_process_checkbutton->set_active(true);
}


void
PatchWindow::set_control(ControlModel* cm)
{
	m_patch_controller->model()->control_change(cm);

	if (m_control_window != NULL)
		m_control_window->set_control(cm->port_path(), cm->value());
}


void
PatchWindow::new_port(PortModel* pm)
{
	if (m_patch_controller->model()->port_model(ClientPathParser::name(pm->path())) != NULL)
		return;

	m_patch_controller->model()->add_port_model(pm);
	if (m_control_window != NULL)
		m_control_window->add_port(pm, true);
}


void
PatchWindow::zoom_changed()
{
	float z = m_zoom_slider->get_value();
	m_patch_bay->zoom(z);
	//m_patch_state_manager->set_zoom(z);
}


void
PatchWindow::destroy_clicked()
{
	controller->destroy_patch(m_patch_controller->model()->path());
}


void
PatchWindow::process_toggled()
{
	if (m_process_checkbutton->get_active())
		controller->enable_patch(m_patch_controller->model()->path());
	else
		controller->disable_patch(m_patch_controller->model()->path());
}


void
PatchWindow::enabled(bool e)
{
	m_process_checkbutton->set_active(e);
}


void
PatchWindow::show_control_window()
{
	if (m_control_window == NULL) {
		Glib::RefPtr<Gnome::Glade::Xml> xml = app->new_glade_reference("node_control_win");
		xml->get_widget_derived("node_control_win", m_control_window);
		m_control_window->init(m_patch_controller, m_patch_controller->model());
	}
	m_control_window->show();
	m_control_window->raise();
}


/** React to a piece of metadata for a port on this patch.
 *
 * This really doesn't conceptually belong here, but it's here to facilitate
 * port range metadata updates on top level patches.  The general "finding an
 * object" stuff in this client needs a refactoring.
 */
void
PatchWindow::port_metadata_update(const MetadataModel* const mm)
{
	if (m_control_window != NULL) {
		if (mm->key() == "user-min")
			m_control_window->set_range_min(mm->path(), atof(mm->value().c_str()));
		else if (mm->key() == "user-max")
			m_control_window->set_range_max(mm->path(), atof(mm->value().c_str()));
		//else
		//	cerr << "[PatchWindow::port_metadata_update] Unknown metadata key " << mm->key() << endl;
	} else {
		string port_name = ClientPathParser::name(mm->path());
		PortModel* pm = m_patch_controller->model()->port_model(port_name);
		if (pm != NULL) {
			if (mm->key() == "user-min")
				pm->user_min(atof(mm->value().c_str()));
			else if (mm->key() == "user-max")
				pm->user_max(atof(mm->value().c_str()));
		//	else
		//		cerr << "[PatchWindow::port_metadata_update] Unknown metadata key " << mm->key() << endl;
		//} else {
		//	cerr << "[PatchWindow::port_metadata_update] Can not find port " << mm->path() << endl;
		}
	}
}


void
PatchWindow::event_save_patch()
{
	if (m_patch_controller->model()->filename() == "") {
		event_save_patch_as();
	} else {
		controller->save_patch(m_patch_controller->model(), m_patch_controller->model()->filename());
	}
}


void
PatchWindow::event_save_patch_as()
{
	Gtk::FileChooserDialog dialog(*this, "Save Patch", Gtk::FILE_CHOOSER_ACTION_SAVE);
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);	
	
	int result = dialog.run();

	assert(result == Gtk::RESPONSE_OK || result == Gtk::RESPONSE_CANCEL || result == Gtk::RESPONSE_NONE);
	
	if (result == Gtk::RESPONSE_OK) {	
		string filename = dialog.get_filename();
		if (filename.length() < 4 || filename.substr(filename.length()-3) != ".om")
			filename += ".om";
			
		bool confirm = false;
		std::fstream fin;
		fin.open(filename.c_str(), std::ios::in);
		if (fin.is_open()) {  // File exists
			string msg = "File already exists!  Are you sure you want to overwrite ";
			msg += filename + "?";
			Gtk::MessageDialog confirm_dialog(*this,
				msg, false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_YES_NO, true);
			if (confirm_dialog.run() == Gtk::RESPONSE_YES)
				confirm = true;
			else
				confirm = false;
		} else {  // File doesn't exist
			confirm = true;
		}
		fin.close();
		
		if (confirm) {
			controller->save_patch(m_patch_controller->model(), filename);
			m_patch_controller->model()->filename(filename);
		}
	}
}


void
PatchWindow::event_close()
{
	hide();
}


void
PatchWindow::event_show_add_plugin_win()
{
	m_load_plugin_window->show();
	m_load_plugin_window->raise();
}


void
PatchWindow::event_show_new_subpatch_win()
{
	m_new_subpatch_window->show();
	m_new_subpatch_window->raise();
}


void
PatchWindow::event_show_load_subpatch_win()
{
	m_load_subpatch_window->show();
	m_load_subpatch_window->raise();
}


} // namespace OmGtk
