/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <libgnomecanvasmm.h>
#include "Port.h"
#include "Module.h"
#include "PatchBayArea.h"

namespace PatchBay {
	

Port::Port(Module* module, const string& name, bool is_input, int color)
: Gnome::Canvas::Group(*module, 0, 0),
  m_patch_bay(module->patch_bay()),
  m_module(module),
  m_name(name),
  m_is_input(is_input),
  m_color(color),
  m_label(*this, 1, 1, m_name),
  m_rect(*this, 0, 0, 0, 0)
{
	m_menu.items().push_back(Gtk::Menu_Helpers::MenuElem(
		"Disconnect All", sigc::mem_fun(m_patch_bay, &PatchBayArea::port_menu_disconnect_all)));

	m_rect.property_fill_color_rgba() = color;
	
	// Make rectangle pretty
	m_rect.property_outline_color_rgba() = 0x8899AAFF;
	m_rect.property_join_style() = Gdk::JOIN_MITER;
	m_rect.property_width_units() = 0.5;
	
	// Make label pretty
	m_label.property_size() = PORT_LABEL_SIZE;
	m_label.property_fill_color_rgba() = 0xFFFFFFFF;
	m_label.property_weight() = 200;
	
	m_width = m_label.property_text_width() + 4.0;
	m_height = m_label.property_text_height();

	// Place everything
	m_rect.property_x1() = 0;
	m_rect.property_y1() = 0;
	m_rect.property_x2() = m_width;	
	m_rect.property_y2() = m_height;
	m_label.property_x() = m_label.property_text_width() / 2 + 1;
	m_label.property_y() = m_height / 2;

	m_label.raise_to_top();

}


void
Port::zoom(float z)
{
	m_label.property_size() = static_cast<int>(8000 * z);
}


		
void
Port::move_connections()
{
	for (list<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); i++) {
		((Connection*)(*i))->update_location();
	}
}


void
Port::remove_connection(Connection* c)
{
	m_connections.erase(
		find(m_connections.begin(), m_connections.end(), c)
	);
}


void
Port::hilite_connections(bool b)
{
	for (list<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		(*i)->hilite(b);
		(*i)->raise_to_top();
	}
}
	

void
Port::raise_connections()
{
	for (list<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		(*i)->raise_to_top();
	}
}


// Returns the world-relative coordinates of where a connection line
// should attach
Gnome::Art::Point Port::connection_coords()
{
	double x = (is_input()) ? m_rect.property_x1() : m_rect.property_x2();
	double y = m_rect.property_y1() + m_height / 2;
	
	i2w(x, y); // convert to world-relative coords
	
	return Gnome::Art::Point(x, y);
}


void
Port::width(double w)
{
	double diff = w - m_width;
	m_rect.property_x2() = m_rect.property_x2() + diff;
	m_width = w;
}


} // namespace PatchBay
