/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "AddNodeEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "PluginInfo.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Om.h"
#include "OmApp.h"
#include "Patch.h"
#include "PluginFactory.h"
#include "JackDriver.h"
#include "OSCSender.h"
#include "SlowEventQueue.h"

namespace Om {


AddNodeEvent::AddNodeEvent(Request request, const string& path, PluginInfo* pi, bool poly)
: SlowEvent(request),
  m_path(path),
  m_plugin_info(pi),
  m_poly(poly),
  m_patch(NULL),
  m_node(NULL),
  m_tree_node(NULL),
  m_node_already_exists(false)
{
}


void
AddNodeEvent::prepare()
{
	Node* existing = om->path_parser()->find_node(m_path);
	if (existing != NULL) {
		m_node_already_exists = true;
		SlowEvent::prepare();
		return;
	}

	m_patch = om->path_parser()->find_patch(om->path_parser()->parent(m_path));

	if (m_patch != NULL) {
		if (m_poly)
			m_node = om->plugin_factory()->load_plugin(m_plugin_info, m_path, m_patch->internal_poly(), m_patch);
		else
			m_node = om->plugin_factory()->load_plugin(m_plugin_info, m_path, 1, m_patch);
		
		if (m_node != NULL) {
			m_node->parent(m_patch);
			m_node->activate();
		
			m_tree_node = new TreeNode(m_node);

			m_patch->node_remove_mutex().soft_lock();

			// This can be done here because the audio thread doesn't touch the
			// node tree - just the process order array
			m_patch->add_node(m_tree_node);
		}
	}
	SlowEvent::prepare();
}


void
AddNodeEvent::execute(uint sample_offset)
{
	if (m_patch == NULL || m_node == NULL) return;

	SlowEvent::execute(sample_offset);

	if (m_tree_node != NULL) {
		m_tree_node->node()->add_to_patch(m_patch);
		//m_patch->add_node(m_tree_node);
	}
}


void
AddNodeEvent::post_process()
{
	om->jack_driver()->slow_event_queue()->signal_blocking_event_finished();
	string msg;
	if (m_node_already_exists) {
		msg = "Could not create node - ";
		msg += m_path + " already exists.";
		m_request.respond_error(msg);
	} else if (m_patch == NULL) {
		m_request.respond_error("Could not find patch in /patch/add_node");
	} else if (m_node == NULL) {
		msg = "Unable to load node ";
		msg.append(m_path);
		m_request.respond_error(msg);
	} else {
		m_patch->node_remove_mutex().soft_unlock();
		m_request.respond_ok();
		om->osc_sender()->send_node_creation_messages(m_node);
	}
}


} // namespace Om

