/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "DestroyPatchEvent.h"
#include "Patch.h"
#include "Node.h"
#include "NodeTree.h"
#include "PluginInfo.h"
#include "Om.h"
#include "Array.h"
#include "OSCSender.h"
#include "OmApp.h"
#include "JackDriver.h"
#include "Maid.h"
#include "PathParser.h"
#include "RemoveNodeEvent.h"
#include "SlowEventQueue.h"

namespace Om {


DestroyPatchEvent::DestroyPatchEvent(Request request, const string& path)
: SlowEvent(request, true),
  m_path(path),
  m_patch(NULL),
  m_remove_node_event(NULL)
{
	//cout << "Creating destroy patch event.." << endl;
}


/** No lookup version to be used internally */
DestroyPatchEvent::DestroyPatchEvent(Patch* patch)
: SlowEvent(NIL_REQUEST),
  m_path(patch->path()),
  m_patch(patch),
  m_remove_node_event(NULL)
{
}


DestroyPatchEvent::~DestroyPatchEvent()
{
	for (List<DestroyPatchEvent*>::iterator i = m_destroy_subpatch_events.begin();
			i != m_destroy_subpatch_events.end(); ++i)
		delete (*i);
}


void
DestroyPatchEvent::prepare()
{
	if (m_patch == NULL)
		m_patch = om->path_parser()->find_patch(m_path);
	
	if (m_patch == NULL) {
		SlowEvent::prepare();
		return;
	}
	
	if (m_patch->parent() != NULL) {
		// FIXME: slow, looks up the Patch all over again
		m_remove_node_event = new RemoveNodeEvent(NIL_REQUEST, m_path, false);
		m_remove_node_event->prepare();
	}
	
	for (List<Patch*>::iterator i = om->patches().begin(); i != om->patches().end(); ++i)
		if ((*i)->parent() == m_patch) {
			DestroyPatchEvent* ev = new DestroyPatchEvent(*i);
			ev->prepare();
			m_destroy_subpatch_events.push_back(new ListNode<DestroyPatchEvent*>(ev));
		}
				
	SlowEvent::prepare();
}


void
DestroyPatchEvent::execute(uint sample_offset)
{
	SlowEvent::execute(sample_offset);

	//std::cerr << "Executing destroy event...";

	if (m_remove_node_event != NULL)
		m_remove_node_event->execute(sample_offset);
	
	if (m_patch != NULL) {
		// Assures patch will not be run anymore, so it can be destroyed
		// without race issues
		om->remove_patch(m_patch);

		for (List<DestroyPatchEvent*>::iterator i = m_destroy_subpatch_events.begin();
				i != m_destroy_subpatch_events.end(); ++i)
			(*i)->execute(sample_offset);
	}
	
	//std::cerr << "done." << std::endl;
}


void
DestroyPatchEvent::post_process()
{
	om->jack_driver()->slow_event_queue()->signal_blocking_event_finished();
	if (m_patch != NULL) {
		m_request.respond_ok();

		for (List<DestroyPatchEvent*>::iterator i = m_destroy_subpatch_events.begin();
				i != m_destroy_subpatch_events.end(); ++i)
			(*i)->post_process();

		if (m_remove_node_event != NULL)
			m_remove_node_event->post_process();  // this will call...
		else
			om->osc_sender()->send_node_deletion_messages(m_patch);  // ... this ...
		//om->jack_driver()->remove_patch_ports(m_patch);
		om->alsa_driver()->remove_port(m_patch);
		if (m_remove_node_event == NULL)
			om->maid()->push(m_patch);  // ... and this
		
	} else {
		string msg = "Could not find patch '";
		msg.append(m_path).append("' to destroy.");
		m_request.respond_error(msg);
	}
}


} // namespace Om

