/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "DisconnectionEvent.h"
#include <iostream>
#include "Om.h"
#include "OmApp.h"
#include "Maid.h"
#include "List.h"
#include "Node.h"
#include "Connection.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Patch.h"
#include "OSCSender.h"
#include "util.h"


using std::cerr; using std::endl;

namespace Om {


DisconnectionEvent::DisconnectionEvent(Request request, const string& src_port_path, const string& dst_port_path)
: SlowEvent(request),
  m_src_port_path(src_port_path),
  m_dst_port_path(dst_port_path),
  m_patch(NULL),
  m_src_node(NULL),
  m_dst_node(NULL),
  m_src_port(NULL),
  m_dst_port(NULL),
  m_process_order(NULL),
  m_succeeded(true),
  m_lookup(true)
{
}


DisconnectionEvent::DisconnectionEvent(OutputPort* src_port, InputPort* dst_port)
: SlowEvent(NIL_REQUEST),
  m_src_port_path(src_port->path()),
  m_dst_port_path(dst_port->path()),
  m_patch(NULL),
  m_src_node(src_port->node()),
  m_dst_node(dst_port->node()),
  m_src_port(src_port),
  m_dst_port(dst_port),
  m_process_order(NULL),
  m_succeeded(true),
  m_lookup(false)
{
}


void
DisconnectionEvent::prepare()
{
	// cerr << "Preparing disconnection event...\n";

	if (m_lookup) {  // lookup version
		m_patch = om->path_parser()->find_patch(
			om->path_parser()->parent(om->path_parser()->parent(m_src_port_path)));
	
		if (m_patch == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		
		m_patch->node_remove_mutex().soft_lock();
		
		Port* port1 = om->path_parser()->find_port(m_src_port_path);
		Port* port2 = om->path_parser()->find_port(m_dst_port_path);
		
		if (port1 == NULL || port2 == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
	
		if (port1->is_audio() != port2->is_audio()) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		
		if (port1->is_output() && port2->is_input()) {
			m_src_port = (OutputPort*)port1;
			m_dst_port = (InputPort*)port2;
		} else if (port2->is_output() && port1->is_input()) {
			m_src_port = (OutputPort*)port2;
			m_dst_port = (InputPort*)port1;
		} else {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
	
		m_src_node = m_src_port->node();
		m_dst_node = m_dst_port->node();
		
		if (m_src_node == NULL || m_dst_node == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		
		if (m_src_node->parent() != m_patch || m_dst_node->parent() != m_patch) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
	} else { // Internal version (no lookups, ports passed directly)
		if (m_src_node->parent() != m_dst_node->parent() || m_src_node->parent() == NULL) {
			m_succeeded = false;
			SlowEvent::prepare();
			return;
		}
		m_patch = m_src_node->parent();
	}
	
	bool removed = false;
	
	for (List<Node*>::iterator i = m_dst_node->providers()->begin(); i != m_dst_node->providers()->end(); ++i)
		if ((*i) == m_src_node) {
			m_dst_node->providers()->remove(i);
			removed = true;
			break;
		}
	assert(removed);
	removed = false;

	for (List<Node*>::iterator i = m_src_node->dependants()->begin(); i != m_src_node->dependants()->end(); ++i)
		if ((*i) == m_dst_node) {
			m_src_node->dependants()->remove(i);
			removed = true;
			break;
		}
	assert(removed);
	
	if (m_patch->process())
		m_process_order = Om::find_process_order(m_patch);
	
	m_succeeded = true;
	SlowEvent::prepare();	
}


void
DisconnectionEvent::execute(uint sample_offset)
{
	//cerr << "Executing disconnection event...\n";	
	//cerr << "Disconnecting " << m_src_port->path() << " -> " << m_dst_port->path() << endl;

	if (m_succeeded) {

		ListNode<Connection*>* const port_connection = m_dst_port->remove_connection(m_src_port);
		
		if (port_connection != NULL) {
			ListNode<Connection*>* const patch_connection =
				m_patch->remove_connection(m_src_port_path, m_dst_port_path);
	
			assert(port_connection->elem() == patch_connection->elem());
			
			// Clean up both the list node and the connection itself...
			om->maid()->push(port_connection);
			om->maid()->push(patch_connection);
			om->maid()->push(port_connection->elem());
	
			if (m_patch->process_order() != NULL)
				om->maid()->push(m_patch->process_order());
			m_patch->process_order(m_process_order);
		} else {
			m_succeeded = false;  // Ports weren't connected
		}
	}
	SlowEvent::execute(sample_offset);
}


void
DisconnectionEvent::post_process()
{
	if (m_lookup)
		m_patch->node_remove_mutex().soft_unlock();

	if (m_succeeded) {
		m_request.respond_ok();
		om->osc_sender()->send_disconnection(m_src_port_path, m_dst_port_path);

	} else {
		m_request.respond_error("Unable to disconnect ports.");
	}
}


} // namespace Om

