/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "JackDriver.h"
#include <iostream>
#include "Om.h"
#include "OmApp.h"
#include "util.h"
#include "Event.h"
#include "SlowEvent.h"
#include "Node.h"
#include "Patch.h"
#include "Port.h"
#include "InputNode.h"
#include "OutputNode.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "AlsaDriver.h"

using std::cout; using std::cerr; using std::endl;


namespace Om {


JackDriver::JackDriver()
: m_client(NULL),
  m_buffer_size(0),
  m_sample_rate(0),
  m_is_activated(false),
  m_is_enabled(false),
  m_event_queue(1024),      // FIXME: figure out a size
  m_slow_event_queue(1024), // FIXME: ditto
  m_post_processor(2048),   // FIXME: ditto
  m_start_of_current_cycle(0),
  m_start_of_last_cycle(0)
{
	m_client = jack_client_new("Om");
	if (m_client == NULL) {
		cerr << "Unable to connect to Jack.  Exiting." << endl;
		throw;
	}

	jack_on_shutdown(m_client, shutdown_cb, this);

	m_buffer_size = jack_get_buffer_size(m_client);
	m_sample_rate = jack_get_sample_rate(m_client);

	jack_set_sample_rate_callback(m_client, sample_rate_cb, this);
	jack_set_buffer_size_callback(m_client, buffer_size_cb, this);

	pthread_mutex_init(&m_event_queue_mutex, NULL);
	pthread_mutex_init(&m_slow_event_queue_mutex, NULL);

	m_slow_event_queue.start();
	m_post_processor.start();
}


JackDriver::~JackDriver()
{
	deactivate();
}


void
JackDriver::activate()
{	
	if (m_is_activated) {
		cerr << "Jack driver already activated." << endl;
		return;
	}

	jack_set_process_callback(m_client, process_cb, this);

	m_is_activated = true;

	if (jack_activate(m_client)) {
		cerr << "Could not activate Jack client, aborting." << endl;
		throw;
	} else {
		cout << "Activated jack client." << endl;
	}
}


void
JackDriver::deactivate()
{
	jack_deactivate(m_client);
	m_is_activated = false;

	for (List<JackPort>::iterator i = m_out_ports.begin(); i != m_out_ports.end(); ++i)
		jack_port_unregister(m_client, (*i).jack_port());

	m_in_ports.clear();
	m_out_ports.clear();
}



/** Add a jack input port.  Not realtime safe.
 */
ListNode<JackPort>*
JackDriver::add_input(Patch* const patch, InputNode* const node)
{
	jack_port_t* p = NULL;
	jack_default_audio_sample_t* b = 0;
	
	// Register input port and get buffer
	p = jack_port_register(m_client,
		(patch->name() + string("/") + node->name()).c_str(),
		JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
	b = (jack_default_audio_sample_t*)jack_port_get_buffer(p, m_buffer_size);
	ListNode<JackPort>* ln = new ListNode<JackPort>(JackPort(p, b, patch, node->external_port()));
	m_in_ports.push_back(ln);
	return ln;
	//cout << "Added input port.\n";
}


/** Add a jack output port.  Not realtime safe.
 */
ListNode<JackPort>*
JackDriver::add_output(Patch* const patch, OutputNode* const node)
{
	jack_port_t* p = NULL;
	jack_default_audio_sample_t* b = 0;

	// Register output port and get buffer
	p = jack_port_register(m_client,
		(patch->name() + string("/") + node->name()).c_str(),
		JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
	b = (jack_default_audio_sample_t*)jack_port_get_buffer(p, m_buffer_size);
	ListNode<JackPort>* ln = new ListNode<JackPort>(JackPort(p, b, patch, node->external_port()));
	m_out_ports.push_back(ln);
	return ln;
	//cout << "Added output port.\n";
}


void
JackDriver::remove_input(Patch* const patch, InputNode* const node)
{
	for (List<JackPort>::iterator i = m_in_ports.begin(); i != m_in_ports.end(); ++i) {
		if ((*i).patch() == patch && (*i).patch_port() == node->external_port()) {
			remove_port(*i);
			return;
		}
	}
	cerr << "[JackDriver::remove_input] WARNING: Failed to find jack port to remove!" << endl;
}


void
JackDriver::remove_output(Patch* const patch, OutputNode* const node)
{
	for (List<JackPort>::iterator i = m_out_ports.begin(); i != m_out_ports.end(); ++i) {
		if ((*i).patch() == patch && (*i).patch_port() == node->external_port()) {
			remove_port(*i);
			return;
		}
	}
	cerr << "[JackDriver::remove_output] WARNING: Failed to find jack port to remove!" << endl;
}


/** Remove a jack port.  Not realtime safe.
 */
void
JackDriver::remove_port(JackPort& port)
{
	port.is_registered(false);
	jack_port_unregister(m_client, port.jack_port());
}


void
JackDriver::enable() 
{
	assert(m_is_activated);
	
	cout << "[JackDriver] Enabling." << endl;
	m_is_enabled = true;
}


void
JackDriver::disable() 
{
	m_is_enabled = false;
	
	// Testing for > 1 instead of 0 because the patch will still contain a
	// ProcessOrder which has a count
	if (MaidObject::object_count() > 1) {
		cout << "All objects cleaned up.\n";
	} else {
		cerr << "ERROR: " << MaidObject::object_count()-1 <<
			" objects have been leaked.  Aborting.\n";
	}

	// Write output buffers to 0
	for (List<JackPort>::iterator i = m_out_ports.begin(); i != m_out_ports.end(); ++i)
		Om::write_buffer((*i).jack_buffer(), 0.0f, 0, m_buffer_size);
}


/** Push an Event onto the event queue.
 *
 * If the driver isn't running, the event is immediately run and deleted.
 * If the driver is running, the event is pushed on to the event queue.
 * 
 * Multiple threads may use this function to push events - it locks.  The
 * audio thread must NOT call this function.
 */
void
JackDriver::push_event(Event* const ev)
{
	pthread_mutex_lock(&m_event_queue_mutex);
	
	m_event_queue.push(ev);
	
	pthread_mutex_unlock(&m_event_queue_mutex);
}


/** Push a slow event onto the slow event queue.
 *
 * If the driver isn't running, the event is immediately run and deleted.
 * If the driver is running, the event is pushed on to the slow event
 * queue.
 *
 * Multiple threads may use this function to push events - it locks.  The
 * audio thread must NOT call this function.
 */
void
JackDriver::push_slow_event(SlowEvent* const ev)
{
	pthread_mutex_lock(&m_slow_event_queue_mutex);
	
	m_slow_event_queue.push(ev);
	
	pthread_mutex_unlock(&m_slow_event_queue_mutex);
}


/** Process all the pending events for this cycle.
 *
 * Called from the realtime thread once every process cycle.
 */
void
JackDriver::process_events()
{
	Event* ev = NULL;

	int offset = 0;
	
	// Process the "slow" events first, because it's possible some of the
	// RT events depend on them
	
	while ((ev = m_slow_event_queue.pop_before(m_start_of_current_cycle)) != NULL) {
		ev->execute(0);  // SlowEvents are not sample accurate
		m_post_processor.push(ev);
	}
	
	//while ((ev = m_event_queue.pop_before(m_start_of_current_cycle)) != NULL) {
	while (!m_event_queue.empty()
			&& m_event_queue.front()->time_stamp() < m_start_of_current_cycle) {
		ev = m_event_queue.pop();
		offset = ev->time_stamp() - m_start_of_last_cycle;
		if (offset < 0) offset = 0; // this can happen if we miss a cycle
		ev->execute(offset);
		m_post_processor.push(ev);
	}
}



/**** Jack Callbacks ****/



/** Jack process callback, drives entire audio thread.
 *
 * \callgraph
 */
int
JackDriver::m_process_cb(jack_nframes_t nframes) 
{
	// FIXME: support nframes != buffer_size, even though that never damn well happens
	assert(nframes == m_buffer_size);

	// Note that jack can elect to not call this function for a cycle, if things aren't
	// keeping up
	m_start_of_current_cycle = jack_last_frame_time(m_client);
	m_start_of_last_cycle = m_start_of_current_cycle - nframes;

	assert(m_start_of_current_cycle - m_start_of_last_cycle == nframes);

	jack_transport_query(m_client, &m_position);
	
	process_events();
	
	if (m_is_enabled) {
		
		om->alsa_driver()->build_dssi_events_arrays(m_start_of_last_cycle, m_start_of_last_cycle + nframes);

		// Copy jack input audio to patch's input ports 
		for (List<JackPort>::iterator i = m_in_ports.begin(); i != m_in_ports.end(); ++i) {
			// Input buffers can't be cached, so have to do this every time:
			if ((*i).is_registered()) {
				(*i).jack_buffer((jack_default_audio_sample_t*)jack_port_get_buffer((*i).jack_port(), m_buffer_size));
				memcpy((*i).patch_port()->buffer(0), (*i).jack_buffer(),
				       sizeof(jack_default_audio_sample_t) * nframes);
			}
		}
		
		for (List<Patch*>::iterator i = om->patches().begin(); i != om->patches().end(); ++i)
			if ((*i)->parent() == NULL && (*i)->process())
				(*i)->run(nframes);
		
		
		// Copy patch's output to the jack output ports
		for (List<JackPort>::iterator i = m_out_ports.begin(); i != m_out_ports.end(); ++i) {
			if ((*i).is_registered()) {
				memcpy((*i).jack_buffer(), (*i).patch_port()->buffer(0),
				       sizeof(jack_default_audio_sample_t) * nframes);
			}
		}
	}

	return 0;
}


void 
JackDriver::m_shutdown_cb() 
{
	cout << "Jack shutdown.  Exiting." << endl;
	om->set_quit_flag();
}


int
JackDriver::m_sample_rate_cb(jack_nframes_t nframes) 
{
	if (m_is_activated) {
		cerr << "Om does not support changing sample rate on the fly (yet).  Aborting." << endl;
		throw;
	} else {
		m_sample_rate = nframes;
	}
	return 0;
}


int
JackDriver::m_buffer_size_cb(jack_nframes_t nframes) 
{
	if (m_is_activated) {
		cerr << "Om does not support chanding buffer size on the fly (yet).  Aborting." << endl;
		throw;
	} else {
		m_buffer_size = nframes;
	}
	return 0;
}


} // namespace Om

