/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "OSCSender.h"
#include <cassert>
#include "unistd.h"
#include "Om.h"
#include "OmApp.h"
#include "PluginFactory.h"
#include "OSCReceiver.h"
#include "util.h"
#include "Patch.h"
#include "Node.h"
#include "Port.h"
#include "PortInfo.h"
#include "PluginInfo.h"
#include "Port.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Connection.h"
#include "InputNode.h"
#include "OutputNode.h"
#include "JackDriver.h"

namespace Om {

	
/*! \page client_osc_namespace Client OSC Namespace Documentation
 *
 * \p These are the commands sent from the engine to the client.  A client must
 * recognize these commands if it wants to know what is happening with the
 * engine (new nodes, patches, control changes, etc) \n \n
 */
	

OSCSender::OSCSender(const OSCReceiver* osc_receiver)
: m_osc_receiver(osc_receiver)
{}


void
OSCSender::send_response_ok(lo_address addr, int request_id)
{
	assert(addr != NULL);

	lo_send(addr, "/om/response/ok", "i", request_id);
}


void
OSCSender::send_response_error(lo_address addr, int request_id, const string& msg)
{
	assert(addr != NULL);
	lo_send(addr, "/om/response/error", "is", request_id, msg.c_str());
}


/** Notify the client of an error.
 *
 * This is for errors that aren't the direct result of a client command.  The
 * 'response' stuff is for that.
 */
void
OSCSender::send_error(const string& msg)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i) {
		//std::cerr << "Sending error." << std::endl;
		lo_send((*i)->addr(), "/om/error", "s", msg.c_str());
	}
}


void
OSCSender::send_plugins(lo_address addr)
{
	om->plugin_factory()->lock_plugin_list();
	
	const list<PluginInfo>& plugs = om->plugin_factory()->plugins();
	const PluginInfo* info;

	for (list<PluginInfo>::const_iterator j = plugs.begin(); j != plugs.end(); ++j) {
		info = &(*j);
		lo_send(addr, "/om/plugin", "ssss", 
			info->lib_name().c_str(),
			info->plug_label().c_str(),
			info->name().c_str(),
			info->type_string());
		usleep(10000);  // plugins get lost in the ether without this :/  (?) 
	}
	
	om->plugin_factory()->unlock_plugin_list();
}


void
OSCSender::send_node(const Node* const node)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_node_to((*i)->addr(), node);
}


void
OSCSender::send_node_removal(const string& path)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_node_removal_to((*i)->addr(), path);
}


void
OSCSender::send_new_port(const Port* port)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_new_port_to((*i)->addr(), port);
}


void
OSCSender::send_port_removal(const Port* port)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_port_removal_to((*i)->addr(), port);
}


void
OSCSender::send_patch_destruction(const string& patch_path)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_patch_destruction_to((*i)->addr(), patch_path);
}


void
OSCSender::send_connection(const Connection* const c)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_connection_to((*i)->addr(), c);
}


void
OSCSender::send_disconnection(const string& src_port_path, const string& dst_port_path)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_disconnection_to((*i)->addr(), src_port_path, dst_port_path);
}


void
OSCSender::send_engine_enable(Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_engine_enable_to((*i)->addr());
}


void
OSCSender::send_engine_disable(Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_engine_disable_to((*i)->addr());
}


void
OSCSender::send_patch_enable(const string& patch_path, Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_patch_enable_to((*i)->addr(), patch_path);
}


void
OSCSender::send_patch_disable(const string& patch_path, Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_patch_disable_to((*i)->addr(), patch_path);
}


/** Send notification of a metadata update.
 *
 * Like control changes, does not send update to client that set the metadata, if applicable.
 */
void
OSCSender::send_metadata_update(const string& node_path, const string& key, const string& value, Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_metadata_update_to((*i)->addr(), node_path, key, value);
}


/** Send notification of a control change.
 *
 * If request is specified, the notification will not be send to the address of
 * that request (to avoid sending redundant information back to clients and
 * forcing clients to ignore things to avoid feedback loops etc).
 */
void
OSCSender::send_control_change(const string& port_path, float value, Request request)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		if ((*i)->addr() != request.source())
			send_control_change_to((*i)->addr(), port_path, value);
}


/** Send a patch.
 *
 * Sends all objects underneath Patch - contained Nodes, etc.
 */
void
OSCSender::send_patch(const Patch* const p)
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_patch_to((*i)->addr(), p);
}


/** Sends all OmObjects known to the engine.
 */
void
OSCSender::send_all_objects()
{
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		send_all_objects_to((*i)->addr());
}


void
OSCSender::send_node_creation_messages(const Node* const node) {
	// This is pretty stupid :/  in and out and back again!
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		node->send_creation_messages((*i)->addr());
}
	

void
OSCSender::send_node_deletion_messages(const Node* const node) {
	// This is pretty stupid :/  in and out and back again!
	for (list<ClientRecord*>::const_iterator i = m_osc_receiver->clients().begin(); i != m_osc_receiver->clients().end(); ++i)
		node->send_deletion_messages((*i)->addr());
}



//
//  SINGLE DESTINATION VERSIONS
//


/** \page client_osc_namespace
 * <p> \b /om/engine_enabled - Notification engine's DSP has been enabled. </p> \n \n
 */
void
OSCSender::send_engine_enable_to(lo_address addr)
{
	lo_send(addr, "/om/engine_enabled", NULL);
}


/** \page client_osc_namespace
 * <p> \b /om/engine_disabled - Notification engine's DSP has been disabled. </p> \n \n
 */
void
OSCSender::send_engine_disable_to(lo_address addr)
{
	lo_send(addr, "/om/engine_disabled", NULL);
}


/** \page client_osc_namespace
 * <p> \b /om/new_node - Notification of a new node's creation.
 * \arg \b path (string) - Path of the new node
 * \arg \b polyphonic (integer-boolean) - Node is polyphonic (1 for yes, 0 for no)
 * \arg \b type (string) - Type of plugin (LADSPA, DSSI, Internal, Patch)
 * \arg \b lib-name (string) - Name of library if a plugin (ie cmt.so)
 * \arg \b plug-label (string) - Label of plugin in library (ie adsr_env)
 * 
 * \li New nodes are sent as a blob.  The first message in the blob will be this
 * one (/om/new_node), followed by a series of /om/new_port commands, followed
 * by /om/new_node_end. </p> \n \n
 */
void
OSCSender::send_node_to(lo_address addr, const Node* const node)
{
	lo_timetag tt;
	lo_timetag_now(&tt);
	lo_bundle b = lo_bundle_new(tt);
	lo_message m = lo_message_new();
	list<lo_message> msgs;

	lo_message_add_string(m, node->path().c_str());
	lo_message_add_int32(m, (node->poly() == node->parent()->internal_poly() ? 1 : 0));
	lo_message_add_string(m, node->plugin_info().type_string());
	lo_message_add_string(m, node->plugin_info().lib_name().c_str());
	lo_message_add_string(m, node->plugin_info().plug_label().c_str());
	lo_bundle_add_message(b, "/om/new_node", m);

	const Array<Port*>& ports = node->ports();
	Port*     port;
	PortInfo* info;
	for (uint j=0; j < ports.size(); ++j) {
		port = ports.at(j);
		info = port->port_info();

		assert(port != NULL);
		assert(info != NULL);

		m = lo_message_new();
		lo_message_add_string(m, port->path().c_str());
		lo_message_add_string(m, info->type_string().c_str());
		lo_message_add_string(m, info->direction_string().c_str());
		lo_message_add_string(m, info->hint_string().c_str());
		lo_message_add_float(m, info->default_val());
		lo_message_add_float(m, info->min_val());
		lo_message_add_float(m, info->max_val());
		lo_bundle_add_message(b, "/om/new_port", m);
		msgs.push_back(m);
	}

	m = lo_message_new();
	lo_bundle_add_message(b, "/om/new_node_end", m);

	lo_send_bundle(addr, b);
	lo_bundle_free(b);

	for (list<lo_bundle>::const_iterator i = msgs.begin(); i != msgs.end(); ++i)
		lo_message_free(*i);

	usleep(10000);
	
	const map<string, string>& data = node->metadata();
	// Send node metadata
	for (map<string, string>::const_iterator i = data.begin(); i != data.end(); ++i)
		send_metadata_update_to(addr, node->path(), (*i).first, (*i).second);

	usleep(1000);
	
	// Send port metadata
	for (uint j=0; j < ports.size(); ++j) {
		port = ports.at(j);
		const map<string, string>& data = port->metadata();
		for (map<string, string>::const_iterator i = data.begin(); i != data.end(); ++i)
			send_metadata_update_to(addr, port->path(), (*i).first, (*i).second);
	}

	usleep(1000);

	// Send control values
	for (uint i=0; i < node->ports().size(); ++i) {
		Port* port = node->ports().at(i);
		if (port->is_control())
			send_control_change_to(addr, port->path(), port->get_value(0, 0));
	}
}


/** \page client_osc_namespace
 * <p> \b /om/node_removal - Notification of a node's destruction.
 * \arg \b path (string) - Path of node (which no longer exists) </p> \n \n
 */
void
OSCSender::send_node_removal_to(lo_address addr, const string& path)
{
	lo_send(addr, "/om/node_removal", "s", path.c_str());
}



/** \page client_osc_namespace
 * <p> \b /om/new_port - Notification of a node's destruction.
 * \arg \b path (string) - Path of new port
 * \arg \b type (string) - Type of port (CONTROL or AUDIO)
 * \arg \b direction (string) - Direction of data flow (INPUT or OUTPUT)
 * \arg \b hint (string) - Hint (INTEGER, LOGARITHMIC, TOGGLE, or NONE)
 * \arg \b default-value (float) - Default (initial) value
 * \arg \b min-value (float) - Suggested minimum value
 * \arg \b min-value (float) - Suggested maximum value
 *
 * \li Note that in the event of loading a patch, this message could be
 * followed immediately by a control change, meaning the default-value is
 * not actually the current value of the port (ahem, Lachlan).
 * \li The minimum and maximum values are suggestions only, they are not
 * enforced in any way, and going outside them is perfectly fine.  Also note
 * that the port ranges in om_gtk are not these ones!  Those ranges are set
 * as metadata.</p> \n \n
 */
void
OSCSender::send_new_port_to(lo_address addr, const Port* port)
{
	PortInfo* info = port->port_info();
	
	lo_send(addr, "/om/new_port", "ssssfff",
	        port->path().c_str(), info->type_string().c_str(), info->direction_string().c_str(),
			info->hint_string().c_str(), info->default_val(), info->min_val(), info->max_val());
	
	// Send metadata
	const map<string, string>& data = port->metadata();
	for (map<string, string>::const_iterator i = data.begin(); i != data.end(); ++i)
		send_metadata_update_to(addr, port->path(), (*i).first, (*i).second);
}


/** \page client_osc_namespace
 * <p> \b /om/port_removal - Notification of a port's destruction.
 * \arg \b path (string) - Path of port (which no longer exists) </p> \n \n
 */
void
OSCSender::send_port_removal_to(lo_address addr, const Port* port)
{
	lo_send(addr, "/om/port_removal", "s", port->path().c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/patch_destruction - Notification of a patch's destruction.
 * \arg \b path (string) - Path of patch (which no longer exists) </p> \n \n
 */
void
OSCSender::send_patch_destruction_to(lo_address addr, const string& patch_path)
{
	lo_send(addr, "/om/patch_destruction", "s", patch_path.c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/patch_enabled - Notification a patch's DSP processing has been enabled.
 * \arg \b path (string) - Path of enabled patch</p> \n \n
 */
void
OSCSender::send_patch_enable_to(lo_address addr, const string& patch_path)
{
	lo_send(addr, "/om/patch_enabled", "s", patch_path.c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/patch_disabled - Notification a patch's DSP processing has been disabled.
 * \arg \b path (string) - Path of disabled patch</p> \n \n
 */
void
OSCSender::send_patch_disable_to(lo_address addr, const string& patch_path)
{
	lo_send(addr, "/om/patch_disabled", "s", patch_path.c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/new_connection - Notification a new connection has been made.
 * \arg \b src-path (string) - Path of the source port
 * \arg \b dst-path (string) - Path of the destination port</p> \n \n
 */
void
OSCSender::send_connection_to(lo_address addr, const Connection* const c)
{
	lo_send(addr, "/om/new_connection", "ss", c->src_port()->path().c_str(), c->dst_port()->path().c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/disconnection - Notification a connection has been unmade.
 * \arg \b src-path (string) - Path of the source port
 * \arg \b dst-path (string) - Path of the destination port</p> \n \n
 */
void
OSCSender::send_disconnection_to(lo_address addr, const string& src_port_path, const string& dst_port_path)
{
	lo_send(addr, "/om/disconnection", "ss", src_port_path.c_str(), dst_port_path.c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/metadata/update - Notification of a piece of metadata.
 * \arg \b path (string) - Path of the object associated with metadata (can be a node, patch, or port)
 * \arg \b key (string)
 * \arg \b value (string)</p> \n \n
 */
void
OSCSender::send_metadata_update_to(lo_address addr, const string& path, const string& key, const string& value)
{
	lo_send(addr, "/om/metadata/update", "sss", path.c_str(), key.c_str(), value.c_str());
}


/** \page client_osc_namespace
 * <p> \b /om/control_change - Notification the value of a port has changed
 * \arg \b path (string) - Path of port
 * \arg \b value (float) - New value of port
 *
 * \li This will only send updates for values set by clients of course - not values
 * changing because of connections to other ports!</p> \n \n
 */
void
OSCSender::send_control_change_to(lo_address addr, const string& port_path, float value)
{
	lo_send(addr, "/om/control_change", "sf", port_path.c_str(), value);
}


/** \page client_osc_namespace
 * <p> \b /om/new_patch - Notification of a new patch
 * \arg \b path (string) - Path of new patch
 * \arg \b poly (int) - Polyphony of new patch (\em not a boolean like new_node)
 */
void
OSCSender::send_patch_to(lo_address addr, const Patch* const p)
{
	// FIXME: this should really lock this, since it accesses the node tree and all,
	// but then p would have to be non-const, which ricochets through the whole damn
	// codebase, argh.
	
	//p->node_remove_mutex().soft_lock();

	lo_send(addr, "/om/new_patch", "si", p->path().c_str(), p->internal_poly());
	
	usleep(10000);

	// FIXME: This is less than ideal...
	for (NodeTree::iterator j = p->nodes().begin(); j != p->nodes().end(); ++j) {
		usleep(1000);
		(*j)->send_creation_messages(addr);

		// If this is an input/output node, send the patch control value as well
		const PluginInfo& pi = (*j)->plugin_info();
		if (pi.type() == PluginInfo::Internal) {
			Port* parent_port = NULL;
			if (pi.plug_label() == "control_input" || pi.plug_label() == "audio_input")
				parent_port = ((InputNode*)(*j))->external_port();
			else if (pi.plug_label() == "control_output" || pi.plug_label() == "audio_output")
				parent_port = ((OutputNode*)(*j))->external_port();
			if (parent_port != NULL) {
				send_control_change_to(addr, parent_port->path(), parent_port->get_value(0, 0));
			}
		}
	}
	
	for (List<Connection*>::const_iterator j = p->connections().begin(); j != p->connections().end(); ++j) {
		usleep(1000);
		send_connection_to(addr, *j);
	}

	// Send control values
	for (uint i=0; i < p->ports().size(); ++i) {
		Port* port = p->ports().at(i);
		if (port->is_control())
			send_control_change_to(addr, port->path(), port->get_value(0, 0));
	}

	// Send metadata
	const map<string, string>& data = p->metadata();
	for (map<string, string>::const_iterator i = data.begin(); i != data.end(); ++i) {
		usleep(1000);
		send_metadata_update_to(addr, p->path(), (*i).first, (*i).second);
	}

	if (p->process())
		send_patch_enable_to(addr, p->path());

	//p->node_remove_mutex.soft_unlock();
}


/** Sends all OmObjects known to the engine.
 */
void
OSCSender::send_all_objects_to(lo_address addr)
{
	for (List<Patch*>::iterator i = om->patches().begin(); i != om->patches().end(); ++i)
		if ((*i)->parent() == NULL)
			(*i)->send_creation_messages(addr);
	
	if (om->jack_driver()->is_enabled())
		send_engine_enable_to(addr);
	else
		send_engine_disable_to(addr);
}


} // namespace Om
