/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PluginFactory.h"
#include <pthread.h>
#include <dirent.h>
#include <float.h>
#include <cmath>
#include <dlfcn.h>
#include "JackDriver.h"
#include "LADSPAPlugin.h"
#include "DSSIPlugin.h"
#include "MidiNoteNode.h"
#include "MidiTriggerNode.h"
#include "MidiControlNode.h"
#include "AudioInputNode.h"
#include "ControlInputNode.h"
#include "AudioOutputNode.h"
#include "ControlOutputNode.h"
#include "TransportNode.h"
#include "PluginInfo.h"
#include "Patch.h"
#include "Om.h"
#include "OmApp.h"

using std::string;

namespace Om {


/* I am perfectly aware that the vast majority of this class is a 
 * vomit inducing nightmare at the moment ;)
 */



PluginFactory::PluginFactory()
: m_has_loaded(false)
{
	pthread_mutex_init(&m_plugin_list_mutex, NULL);
	
	// Add builtin plugin types to m_internal_plugins list
	// FIXME: ewwww, definitely a better way to do this!
	//PluginInfo* pi = NULL;

	Patch* parent = new Patch("/dummy", 1, NULL, 1, 1, 1);

	Node* n = NULL;
	n = new AudioInputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new ControlInputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new AudioOutputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new ControlOutputNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new MidiNoteNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new MidiTriggerNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new MidiControlNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
	n = new TransportNode("foo", 1, parent, 1, 1);
	m_internal_plugins.push_back(n->plugin_info());
	delete n;
}


PluginFactory::~PluginFactory()
{
	//for (list<PluginInfo*>::iterator i = m_internal_plugins.begin(); i != m_internal_plugins.end(); ++i)
	//	delete (*i);
}


/** Loads a plugin.
 *
 * Calls the load_*_plugin functions to actually do things, just a wrapper.
 */
Node*
PluginFactory::load_plugin(const PluginInfo* info, const string& path, uint poly, Patch* parent)
{
	pthread_mutex_lock(&m_plugin_list_mutex);
	
	Node* r = NULL;
	
	switch (info->type()) {
	case PluginInfo::LADSPA:
		r = load_ladspa_plugin(info->lib_name(), info->plug_label(), path, poly, parent);
		break;
	case PluginInfo::DSSI:
		r = load_dssi_plugin(info->lib_name(), info->plug_label(), path, poly, parent);
		break;
	case PluginInfo::Internal:
		r = load_internal_plugin(info->plug_label(), path, poly, parent);
		break;
	default:
		cerr << "[PluginFactory] WARNING: Unknown plugin type." << endl;
	}

	pthread_mutex_unlock(&m_plugin_list_mutex);

	return r;
}


/** Loads an internal plugin.
 */
Node*
PluginFactory::load_internal_plugin(const string& plug_label, const string& path, uint poly, Patch* parent)
{
	// FIXME: this hardcoded plugin label stuff is nasty.  remove
	
	assert(poly == 1 || poly == parent->internal_poly());

	// Poly to use for input/output nodes
	// A port on a patch (represented by an InputNode or OutputNode) will only
	// appear polyphonic if the patch's polyphony is the same as it's patch's parent's
	// polyphony
	uint io_poly = 1;
	if (parent->parent() != NULL
			&& parent->parent()->internal_poly() == parent->internal_poly())
		io_poly = parent->internal_poly();
	
	if (plug_label == "input" || plug_label == "audio_input") {  // FIXME: remove temp compat fix(es)
		AudioInputNode* in = new AudioInputNode(path, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return in;
	} else if (plug_label == "control_input") {
		ControlInputNode* in = new ControlInputNode(path, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return in;
	} else if (plug_label == "output" || plug_label == "audio_output") {
		AudioOutputNode* on = new AudioOutputNode(path, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return on;
	} else if (plug_label == "control_output") {
		ControlOutputNode* on = new ControlOutputNode(path, io_poly, parent,
			om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return on;
	} else if (plug_label == "note_in" || plug_label == "midi_note_in") {
		MidiNoteNode* mn = new MidiNoteNode(path, poly, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "trigger_in" || plug_label == "midi_trigger_in") {
		MidiTriggerNode* mn = new MidiTriggerNode(path, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "midi_control_in") {
		MidiControlNode* mn = new MidiControlNode(path, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return mn;
	} else if (plug_label == "transport") {
		TransportNode* tn = new TransportNode(path, 1, parent, om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
		return tn;
	} else {
		std::cerr << "Unknown internal plugin type '" << plug_label << "'" << std::endl;
		throw;
	}

	return NULL;
}

			
/** Loads a LADSPA plugin.
 * Returns 'poly' independant plugins as a Node*
 */
// FIXME: make this take a PluginInfo as a param
Node*
PluginFactory::load_ladspa_plugin(const string& lib_name, const string& plug_name, const string& path, uint poly, Patch* parent)
{
	assert(plug_name != "");
	assert(lib_name != "");
	assert(path != "");
	assert(poly > 0);

	LADSPA_Descriptor_Function df = NULL;
	const PluginInfo* plugin_info = NULL;
	Node* n = NULL;
	void* plugin_lib = NULL;
	
	// Attempt to find the lib
	list<PluginInfo>::iterator i;
	for (i = m_plugins.begin(); i != m_plugins.end(); ++i) {
		plugin_info = &(*i);
		if (plugin_info->lib_name() == lib_name && plugin_info->plug_label() == plug_name) break;
	}

	if (i == m_plugins.end()) {
		std::cerr << "Did not find ladspa plugin " << lib_name << ":" << plug_name << " in database." << std::endl;
		return NULL;
	} else {
		string full_lib_name = plugin_info->lib_path();
		full_lib_name += "/";
		full_lib_name += plugin_info->lib_name();
			
		// Load descriptor function
		dlerror();
		plugin_lib = dlopen(full_lib_name.c_str(), RTLD_NOW);
		if (plugin_lib == NULL || dlerror() != NULL) {
			std::cerr << "Unable to load LADSPA plugin library." << std::endl;
			dlclose(plugin_lib);
			throw;
		}
			
		dlerror();
		df = (LADSPA_Descriptor_Function)dlsym(plugin_lib, "ladspa_descriptor");
		if (df == NULL || dlerror() != NULL) {
			std::cerr << "Looks like this isn't a LADSPA plugin." << std::endl;
			dlclose(plugin_lib);
			throw;
		}
	}

	if (df == NULL) {
		std::cerr << "Could not find LADSPA library \"" << lib_name << "\"." << std::endl;
		return NULL;
	}
	
	// Attempt to find the plugin in lib
	LADSPA_Descriptor* descriptor = NULL;
	for (uint i=0; (descriptor = (LADSPA_Descriptor*)df(i)) != NULL; ++i) {
		if (descriptor->Label == plug_name) {
			break;
		}
	}
	
	if (descriptor == NULL) {
		std::cerr << "Could not find plugin \"" << plug_name << "\" in " << lib_name << std::endl;
		return NULL;
	}

	n = new LADSPAPlugin(path, poly, parent, descriptor,
		om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
	
	n->plugin_info(*plugin_info);

	return n;
}


Node*
PluginFactory::load_dssi_plugin(const string& lib_name, const string& plug_name, const string& path, uint poly, Patch* parent)
{
	// FIXME: awful code duplication here
	
	assert(plug_name != "");
	assert(lib_name != "");
	assert(path != "");
	assert(poly > 0);

	DSSI_Descriptor_Function df = NULL;
	const PluginInfo* plugin_info = NULL;
	Node* n = NULL;
	void* plugin_lib = NULL;
	
	// Attempt to find the lib
	list<PluginInfo>::iterator i;
	for (i = m_plugins.begin(); i != m_plugins.end(); ++i) {
		plugin_info = &(*i);
		if (plugin_info->lib_name() == lib_name && plugin_info->plug_label() == plug_name) break;
	}

	if (i == m_plugins.end()) {
		std::cerr << "Did not find dssi plugin " << lib_name << ":" << plug_name << " in database." << std::endl;
		return NULL;
	} else {
		string full_lib_name = plugin_info->lib_path();
		full_lib_name += "/";
		full_lib_name += plugin_info->lib_name();
			
		// Load descriptor function
		dlerror();
		plugin_lib = dlopen(full_lib_name.c_str(), RTLD_NOW);
		if (plugin_lib == NULL || dlerror() != NULL) {
			std::cerr << "Unable to load DSSI plugin library." << std::endl;
			dlclose(plugin_lib);
			throw;
		}
			
		dlerror();
		df = (DSSI_Descriptor_Function)dlsym(plugin_lib, "dssi_descriptor");
		if (df == NULL || dlerror() != NULL) {
			std::cerr << "Looks like this isn't a DSSI plugin." << std::endl;
			dlclose(plugin_lib);
			throw;
		}
	}

	if (df == NULL) {
		std::cerr << "Could not find DSSI library \"" << lib_name << "\"." << std::endl;
		return NULL;
	}
	
	// Attempt to find the plugin in lib
	DSSI_Descriptor* descriptor = NULL;
	for (uint i=0; (descriptor = (DSSI_Descriptor*)df(i)) != NULL; ++i) {
		if (descriptor->LADSPA_Plugin != NULL && descriptor->LADSPA_Plugin->Label == plug_name) {
			break;
		}
	}
	
	if (descriptor == NULL) {
		std::cerr << "Could not find plugin \"" << plug_name << "\" in " << lib_name << std::endl;
		return NULL;
	}

	n = new DSSIPlugin(path, poly, parent, descriptor,
		om->jack_driver()->sample_rate(), om->jack_driver()->buffer_size());
	
	n->plugin_info(*plugin_info);

	return n;
}


void
PluginFactory::load_plugins()
{
	// Only load if we havn't already, so every client connecting doesn't cause
	// this (expensive!) stuff to happen.  Not the best solution - would be nice
	// if clients could refresh plugins list for whatever reason :/
	if (!m_has_loaded) {
		pthread_mutex_lock(&m_plugin_list_mutex);
		
		m_plugins.clear();
		m_plugins = m_internal_plugins;
	
		load_dssi_plugins();
		load_ladspa_plugins();
		
		m_has_loaded = true;
		
		pthread_mutex_unlock(&m_plugin_list_mutex);
	}
}


/** Loads information about all LADSPA plugins into internal plugin database.
 */
void
PluginFactory::load_ladspa_plugins()
{
	char* env_ladspa_path = getenv("LADSPA_PATH");
	string ladspa_path;
	if (!env_ladspa_path) {
	 	std::cerr << "[PluginFactory] LADSPA_PATH is empty.  Assuming /usr/lib/ladspa:/usr/local/lib/ladspa." << std::endl;
		ladspa_path = "/usr/lib/ladspa:/usr/local/lib/ladspa";
	} else {
		ladspa_path = env_ladspa_path;
	}

	LADSPA_Descriptor_Function df         = NULL;
	LADSPA_Descriptor*         descriptor = NULL;
	
	string dir;

	// Yep, this should use an sstream alright..
	while (ladspa_path != "") {
		string dir = ladspa_path.substr(0, ladspa_path.find(':'));
		if (ladspa_path.find(':') != string::npos)
			ladspa_path = ladspa_path.substr(ladspa_path.find(':')+1);
		else
			ladspa_path = "";
	
		DIR* pdir = opendir(dir.c_str());
		if (pdir == NULL) {
			//std::cerr << "[PluginFactory] Unreadable directory in LADSPA_PATH: " << dir.c_str() << std::endl;
			continue;
		}

		struct dirent* pfile;
		while ((pfile = readdir(pdir))) {
			string full_lib_name = dir;
			full_lib_name += "/";
			full_lib_name += pfile->d_name;
			
			// Load descriptor function
			dlerror();
			void* plugin = dlopen(full_lib_name.c_str(), RTLD_NOW);
			if (plugin == NULL || dlerror() != NULL) {
				//cerr << "[PluginFactory] WARNING: Unable to load LADSPA plugin library \""
				//	<< pfile->d_name << "\"" << endl;
				continue;
			}
			dlerror();
			df = (LADSPA_Descriptor_Function)dlsym(plugin, "ladspa_descriptor");
			if (df == NULL || dlerror() != NULL) {
				//std::cerr << "Looks like this isn't a LADSPA plugin." << std::endl;
				continue;
			}	

			for (uint i=0; (descriptor = (LADSPA_Descriptor*)df(i)) != NULL; ++i) {
				PluginInfo info;
				info.lib_path(dir);
				info.lib_name(pfile->d_name);
				info.plug_label(descriptor->Label);
				info.name(descriptor->Name);
				//info.id(descriptor->UniqueID);
				info.type(PluginInfo::LADSPA);
				
				bool found = false;
				for (list<PluginInfo>::iterator i = m_plugins.begin(); i != m_plugins.end(); ++i) {
					if ((*i).lib_name() == info.lib_name() && (*i).plug_label() == info.plug_label()) {
						cerr << "Warning: Duplicate LADSPA plugin (" << info.lib_name() << ":"
							<< info.plug_label() << ")" << " found.  Using " << (*i).lib_path()
							<< " version." << endl;
						found = true;
						break;
					}
				}
				if (!found)
					m_plugins.push_back(info);
				//cout << "Loaded " << info.plug_label() << " (" << m_plugins.size() << " plugins)" << endl;
			}
			
			df = NULL;
			descriptor = NULL;
			dlclose(plugin);
		}
		closedir(pdir);
	}
}


/** Loads information about all DSSI plugins into internal plugin database.
 */
void
PluginFactory::load_dssi_plugins()
{
	// FIXME: this is _awful_ code duplication!
	
	char* env_dssi_path = getenv("DSSI_PATH");
	string dssi_path;
	if (!env_dssi_path) {
	 	std::cerr << "[PluginFactory] DSSI_PATH is empty.  Assuming /usr/lib/dssi:/usr/local/lib/dssi." << std::endl;
		dssi_path = "/usr/lib/dssi:/usr/local/lib/dssi";
	} else {
		dssi_path = env_dssi_path;
	}

	DSSI_Descriptor_Function df         = NULL;
	DSSI_Descriptor*         descriptor = NULL;
	
	string dir;

	// Yep, this should use an sstream alright..
	while (dssi_path != "") {
		string dir = dssi_path.substr(0, dssi_path.find(':'));
		if (dssi_path.find(':') != string::npos)
			dssi_path = dssi_path.substr(dssi_path.find(':')+1);
		else
			dssi_path = "";
	
		DIR* pdir = opendir(dir.c_str());
		if (pdir == NULL) {
			//std::cerr << "[PluginFactory] Unreadable directory in DSSI_PATH: " << dir.c_str() << std::endl;
			continue;
		}

		struct dirent* pfile;
		while ((pfile = readdir(pdir))) {
			string full_lib_name = dir;
			full_lib_name += "/";
			full_lib_name += pfile->d_name;
			
			// Load descriptor function
			dlerror();
			void* plugin = dlopen(full_lib_name.c_str(), RTLD_NOW);
			if (plugin == NULL || dlerror() != NULL)
				continue;
			
			dlerror();
			df = (DSSI_Descriptor_Function)dlsym(plugin, "dssi_descriptor");
			if (df == NULL || dlerror() != NULL) {
				//std::cerr << "Looks like this isn't a DSSI plugin." << std::endl;
				continue;
			}	

			const LADSPA_Descriptor* ld = NULL;
			
			for (uint i=0; (descriptor = (DSSI_Descriptor*)df(i)) != NULL; ++i) {
				ld = descriptor->LADSPA_Plugin;
				assert(ld != NULL);
				PluginInfo info;
				info.lib_path(dir);
				info.lib_name(pfile->d_name);
				info.plug_label(ld->Label);
				info.name(ld->Name);
				//info.id(descriptor->UniqueID);
				info.type(PluginInfo::DSSI);

				bool found = false;
				for (list<PluginInfo>::iterator i = m_plugins.begin(); i != m_plugins.end(); ++i) {
					if ((*i).lib_name() == info.lib_name() && (*i).plug_label() == info.plug_label()) {
						cerr << "Warning: Duplicate DSSI plugin (" << info.lib_name() << ":"
							<< info.plug_label() << ")" << " found.  Using " << (*i).lib_path()
							<< " version." << endl;
						found = true;
						break;
					}
				}
				if (!found)
					m_plugins.push_back(info);
				//std::cerr << "Loaded DSSI plugin " << info.name << std::endl;
			}
			
			df = NULL;
			descriptor = NULL;
			dlclose(plugin);
		}
		closedir(pdir);
	}
}

} // namespace Om
