/* This file is part of Om.  Copyright (C) 2004 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "Port.h"
#include <cmath>
#include <iostream>
#include <cassert>
#include "util.h"
#include "Node.h"
#include "PortInfo.h"

namespace Om {


/** Constructor for a Port.
 */
Port::Port(Node* node, const string& path, uint index, uint poly, PortInfo* port_info, size_t buffer_size)
: OmObject(path),
  m_node(node),
  m_poly(poly),
  m_index(index),
  m_buffer_size(buffer_size),
  m_port_info(port_info),
  m_is_local_buffer(false),
  m_buffer_state(OK)
{
	assert(port_info != NULL);

	assert(m_poly > 0);
	allocate_buffers();
}


Port::~Port()
{
	// FIXME: This is causing crashes for some reason... double free I would
	// assume, but I don't understand how
	//free(m_local_buffer);
	for (uint i=0; i < m_poly; ++i)
		free(m_local_buffers.at(i));
}


sample
Port::get_value(uint voice, size_t offset)
{
	assert(offset < m_buffer_size);
	return (m_buffers.at(voice))[offset];
}


/** Set the port's value for a specific voice.
 */
void
Port::set_value(uint voice, size_t offset, sample val)
{
	assert(voice < m_poly);
	
	if (is_control()) {
		*(m_buffers.at(voice)) = val;
	} else {
		Om::write_buffer(m_buffers.at(voice), val, offset, m_buffer_size-1);
		if (offset > 0)
			m_buffer_state = HALF_SET_BLOCK_1;
	}
}


/** Set the port's value for all voices.
 */
void
Port::set_value(size_t offset, sample val)
{
	if (is_control()) {	
		for (uint v=0; v < m_poly; ++v)
			*(m_buffers.at(v)) = val;
	} else {
		for (uint v=0; v < m_poly; ++v)
			Om::write_buffer(m_buffers.at(v), val, offset, m_buffer_size-1);
		if (offset > 0)
			m_buffer_state = HALF_SET_BLOCK_1;
	}
}


void
Port::allocate_buffers()
{
	sample* buf = NULL;
	
	m_local_buffers.alloc(m_poly);
	m_buffers.alloc(m_poly);

	//std::cerr << ". . . ." << m_name << " allocating buffer." << std::endl;
	if (is_control()) {
		for (uint i=0; i < m_poly; ++i) {
			buf = (sample*)malloc(sizeof(sample));
			*buf = 0.0f;
			m_local_buffers.at(i) = buf;
			m_buffers.at(i) = buf;
		}
	} else if (is_audio()) {
		for (uint i=0; i < m_poly; ++i) {
			buf = (sample*)calloc(m_buffer_size, sizeof(sample));
			Om::write_buffer(buf, 0.0f, 0, m_buffer_size-1);
			m_local_buffers.at(i) = buf;
			m_buffers.at(i) = buf;
		}
	} else {
		throw;
	}
	m_is_local_buffer = true;
}


void
Port::prepare_buffers()
{
	switch (m_buffer_state) {
	case HALF_SET_BLOCK_1:
		m_buffer_state = HALF_SET_BLOCK_2;
		break;
	case HALF_SET_BLOCK_2:
		for (uint i=0; i < m_poly; ++i)
			Om::write_buffer(m_buffers.at(i), m_buffers.at(i)[m_buffer_size-1], 0, m_buffer_size-1);
		m_buffer_state = OK;
		break;
	default:
		break;
	}
}


} // namespace Om

