/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "SubpatchModule.h"
#include <cassert>
#include <iostream>
#include "OmModule.h"
#include "NodeControlWindow.h"
#include "PatchModel.h"
#include "PatchWindow.h"
#include "OmPatchBayArea.h"
#include "PatchController.h"
#include "OmGtk.h"
#include "OmPort.h"
#include "MetadataModel.h"
#include "Controller.h"
using std::cerr; using std::cout; using std::endl;

namespace OmGtk {


SubpatchModule::SubpatchModule(OmPatchBayArea* patch_bay, PatchController* patch_controller)
: OmModule(patch_bay, patch_controller->model(), patch_controller->parent()),
  m_patch_model(patch_controller->model()),
  m_patch_controller(patch_controller)
{
	Gtk::Menu::MenuList& items = m_menu.items();
	items.push_front(Gtk::Menu_Helpers::MenuElem("Show patch window",
		sigc::mem_fun(this, &SubpatchModule::show_patch_window)));

	// Disable "Disconnect All" menuitem
	if (m_patch_model->parent() == NULL)
		items[4].property_sensitive() = false;
	
	m_dialog = patch_controller->control_window();
}


void
SubpatchModule::add_om_port(PortModel* pm, bool resize_to_fit)
{
	OmPort* port = new OmPort(this, pm);

	port->signal_event().connect(
		sigc::bind<Port*>(sigc::mem_fun(m_patch_bay, &OmPatchBayArea::port_event), port));
	
	Module::add_port(port, resize_to_fit);

	if (pm->is_control() && pm->is_input()) {
		m_menu.items()[1].property_sensitive() = true;

		if (m_dialog != NULL) {
			m_dialog->control_panel()->add_port(pm);
			m_dialog->resize();
			port->add_control_panel(m_dialog->control_panel());
			if (m_dialog->control_panel()->mirror() != NULL)
				port->add_control_panel(m_dialog->control_panel()->mirror());
		}
	}
}


void
SubpatchModule::metadata_update(const MetadataModel* const mm)
{
	assert(mm->path() == m_patch_model->path());
	
	// Only store the new filename if this client doesn't already have one
	// (which is more likely to be local and correct)
	if (mm->key() == "filename" && m_patch_model->filename() == "") {
		m_patch_model->filename(mm->value());
	}
	OmModule::metadata_update(mm);
}


void
SubpatchModule::on_double_click()
{
	if (m_patch_controller->control_window()->control_panel()->num_controls() > 0)
		m_patch_controller->show_control_window();
	else
		show_patch_window();
}


void
SubpatchModule::show_patch_window()
{
	PatchWindow* const pw = m_patch_controller->window();
	if (pw != NULL) {
		pw->show();
		pw->raise();
	} else {
		cerr << "** Unable to find PatchWindow for subpatch!" << endl;
	}
}


void
SubpatchModule::show_dialog()
{
	m_patch_controller->show_control_window();
}


void
SubpatchModule::menu_remove()
{
	controller->destroy_patch(m_patch_model->path());
}


} // namespace OmGtk
