/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "Connection.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Node.h"
#include "Om.h"
#include "Port.h"

namespace Om {


/** Constructor for a connection from a node's output port.
 *
 * This handles both polyphonic and monophonic nodes, transparently to the 
 * user (InputPort).
 */
Connection::Connection(OutputPort* const src_port, InputPort* const dst_port)
: m_src_port(src_port),
  m_dst_port(dst_port),
  m_local_buffer(NULL),
  m_is_poly_to_mono( (src_port->parent_node()->poly() > dst_port->parent_node()->poly()) ),
  m_buffer_size(0),
  m_pending_disconnection(false)
{
	assert((src_port->parent_node()->poly() == dst_port->parent_node()->poly())
		|| (src_port->parent_node()->poly() == 1 || dst_port->parent_node()->poly() == 1));

	if (m_is_poly_to_mono) {  // Poly -> Mono connection, need a buffer to mix in to
		m_buffer_size = m_src_port->buffer_size();
		m_local_buffer = (sample*)calloc(m_buffer_size, sizeof(sample));
	}
}


Connection::~Connection()
{
	free(m_local_buffer);
}


void
Connection::prepare_buffers()
{
	if (m_is_poly_to_mono) {
		const uint num_ins = m_src_port->poly();
		
		sample* src;
		sample* dst;
		uint    buffer_size;
	
		memcpy(m_local_buffer, om->zero_buffer(), sizeof(sample) * m_buffer_size);

		/* Thought:  A poly output port can be connected to multiple mono input
		 * ports, which means this mix down would have to happen many times.
		 * Adding a method to OutputPort that mixes down all it's outputs into
		 * a buffer (if it hasn't been done already this cycle) and returns that
		 * would avoid having to mix multiple times.  Probably not a very common
		 * case, but it would be faster anyway. */
	
		// Mix all the source's voices down into local buffer
		for (uint j=0; j < m_src_port->poly(); ++j) {
			src = m_src_port->buffer(j);
			dst = m_local_buffer;
			buffer_size = m_buffer_size;

			while (buffer_size--)
				*dst++ += *src++;
		}

		// Scale the buffer down.
		// (Should this even be done here?  It prevents clipping, but expensive..)
		if (num_ins > 1) {
			float mult = 1.0f/((float)num_ins);
			dst = m_local_buffer;
			buffer_size = m_buffer_size;
			while (buffer_size--)
				*dst++ *= mult;
		}
	}
}


} // namespace Om

