/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef NODETREE_H
#define NODETREE_H

#include <cstdlib>
#include <string>
#include "Node.h"
#include "MaidObject.h"

using std::string;

namespace Om {


/** A node in a Tree.
 */
class TreeNode : public MaidObject
{
public:
	TreeNode()
	: m_parent(NULL), m_left_child(NULL), m_right_child(NULL),
	  m_node(NULL) {}

	TreeNode(Node* n)
	: m_parent(NULL), m_left_child(NULL), m_right_child(NULL),
	  m_node(n) {}

	~TreeNode() { m_parent = NULL; m_left_child = NULL; m_right_child = NULL;}// m_node = NULL; }
	
	TreeNode* parent() const           { return m_parent; }
	void      parent(TreeNode* n)      { m_parent = n; }
	TreeNode* left_child() const       { return m_left_child; }
	void      left_child(TreeNode* n)  { m_left_child = n; }
	TreeNode* right_child() const      { return m_right_child; }
	void      right_child(TreeNode* n) { m_right_child = n; }
	
	bool is_leaf()        { return (m_left_child == NULL && m_right_child == NULL); }
	bool is_left_child()  { return (m_parent != NULL && m_parent->left_child() == this); }
	bool is_right_child() { return (m_parent != NULL && m_parent->right_child() == this); }

	Node* const node() { return m_node; }
	
	friend class NodeTree;
	
protected:
	// Prevent copies
	TreeNode(const TreeNode& copy)            { exit(EXIT_FAILURE); }
	TreeNode& operator=(const TreeNode& copy) { exit(EXIT_FAILURE); }
	
	TreeNode* m_parent;
	TreeNode* m_left_child;
	TreeNode* m_right_child;
	Node*     m_node;
};


/** The tree nodes are stored in (in a patch).
 *
 * Textbook binary search tree (with node names as keys).  Slightly different
 * from a usual BST implementation in that the "Node" classes (TreeNode) are
 * exposed to the user.  This is so SlowEvent's can create the TreeNode in
 * another thread, and the realtime jack thread can insert them (without having
 * to allocating a TreeNode which is a no-no).
 *
 * It's also a more annoying implementation because there's no leaf type (since
 * a leaf object would have to be deleted on insert)
 */
class NodeTree
{
public:
	NodeTree() : m_root(0), m_size(0) {}
	~NodeTree();

	void      insert(TreeNode* const n);
	TreeNode* remove(TreeNode* const n);
	Node*     find(const string& name) const;
	TreeNode* find_treenode(const string& name) const;

	uint size() const { return m_size; }
	

	class iterator
	{
	public:
		iterator(const NodeTree* tree, uint size);
		~iterator();
	
		Node*     operator*() const;
		iterator& operator++();
		bool      operator!=(const iterator& iter) const;
	
		friend class NodeTree;

		iterator(const iterator& copy);
		iterator& operator=(const iterator& copy);

	private:
		int             m_depth;
		uint            m_size;
		TreeNode**      m_stack;
		const NodeTree* m_tree;
	};

	iterator begin() const;
	iterator end() const;

private:
	// Prevent copies
	NodeTree(const NodeTree& copy)            { exit(EXIT_FAILURE); }
	NodeTree& operator=(const NodeTree& copy) { exit(EXIT_FAILURE); }
	
	void m_set_all_traversed_recursive(TreeNode* root, bool b);
	
	TreeNode* m_find_smallest(TreeNode* root);
	TreeNode* m_find_largest(TreeNode* root);

	TreeNode* m_root;
	uint      m_size;
};


} // namespace Om

#endif // NODETREE_H
