/*
Copyright (C) 2000  Groupe Opale (http://www.opale.ovh.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/
package opale.ode;

import opale.mathtools.*;
import opale.tools.*;

/**
* Classe qui modlise un problme d'quations diffrentielles ordinaires. Une instance de cette classe doit tre associ (sens UML)  une instance de TimeScheme (un schma en temps) et une instance de Equation (une quation) afin de pouvoir appeler la mthode solve() charge de rsoudre le problme ainsi form.
* @version 0.1
* @author O.C.
*/

final public class Problem implements ObjectODE
{
private TimeScheme sch;
private Equation eqn;
private DVect cdinit;
private DVect[] inco;
private double[] times;
private int dim;


/**
* Constructeur par dfaut. Initialise un problme de dimension 1.
* @since Opale 0.11
*/
public Problem()
	{
	dim = 1;
	cdinit = new DVect(dim);
	}	

/**
* Constructeur pour initialiser un problme avec une dimension donne.
* @param int dim, la dimension du problme.
* @since Opale 0.11
*/
public Problem(int dim)
	{
	this.dim = Math.max(dim,1);
	cdinit = new DVect(this.dim);
	}	
	
/**
* Methode principale de la classe. Elle est charge de lancer la rsolution du problme d'ODE aprs avoir vrifi que tous les lments sont bien prsents (schma en temps, quation...).
* @since Opale 0.11
*/
public void solve()
	{
	//Vrification des composants du problme.
	System.err.println("*** Vrification des composants du problme ***");
	
	if (eqn == null)
		{
		System.err.println("Vous n'avez pas associ d'quation au problme.\nArret anormal du calcul.");
		System.exit(-1);
		}
	if ( dim!= eqn.dim())
		{
		System.err.println("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		System.exit(-1);
		}
		
	System.err.println(" --> Equation OK");

	if (sch == null)
		{
		System.err.println("Vous n'avez pas associ de schma en temps au problme.\nArret anormal du calcul.");
		System.exit(-1);
		}
	System.err.println(" --> Schma utilis : "+sch);
		
	
	System.err.println(" --> Conditions initiales : "+cdinit);
		
	System.err.println("*** Problme OK ***\n");
	
	
	//Rservation mmoire pour l'inconnue
	int i;
	
	System.err.println("*** Rservation mmoire pour l'inconnu ***\n");
	inco = new DVect[sch.getNstep()];
	for (i=0;i<sch.getNstep();i++)
		inco[i] = new DVect(dim);
	times = new double[sch.getNstep()];	

	System.err.println("*** Initialisation du schma en temps ***\n");
	sch.init();
	inco[0].set(sch.present());
	times[0] = sch.tmin();
	
	System.err.println("*** Dbut du calcul ***");
	java.util.Date tdeb = new java.util.Date();
	
	for(i=1;i<sch.getNstep();i++)
		{
		sch.forward(eqn);
		sch.update();
		inco[i].set(sch.present());
		times[i] = sch.time();
		//System.err.println("t = "+sch.time()+"\n"+sch.present());
		}
	java.util.Date tfin = new java.util.Date();
	System.err.println("*** Fin du calcul ***");
	System.err.println("***Temps pass pour ce calcul : "+(tfin.getTime()-tdeb.getTime())+" ms\n");
	}

/**
* Renvoie la solution calcule sous forme de tableau dans une chaine String : la premire colonne reprsente le temps, puis les suivantes les inconnues.
* @return String, la chaine contenant la solution.
* @since Opale 0.11
*/
public String printSol()
	{
	StringBuffer sol=new StringBuffer();
	if (inco!=null)
	{
	int i,j;
	for (i=0;i<sch.getNstep();i++)
		{
		sol.append( times[i]+" ");
		for (j=0;j<dim;j++)
			sol.append(inco[i].get(j)+" "); 
		sol.append("\n");
		}
	}
	else sol.append("Problme pas encore rsolu\n");
	return sol.toString();	
	}
	
public void writeSol(WFile wf)
	{
	wf.write(printSol());
	}
	
/**
* Renvoie la dimension du problme.
* @return int, la dimension.
* @since Opale 0.11
*/
public int dim()
	{
	return dim;
	}

/**
* Fixe un vecteur pour la condition initiale. Ce vecteur doit avoir la dimension du problme pour taille.
* @param DVect, le vecteur des conditions initiales.
* @since Opale 0.11
*/	
public void setInit(DVect init)
	{
	cdinit.set(init);
	}

/**
* Renvoie le vecteur des conidtions initiales.
* @return DVect, le vecteur des conditions initiales.
* @since Opale 0.11
*/
public DVect getInit()
	{
	return cdinit;
	}

/**
* Associe une quation au problme. La mthode vrifie aussi que l'quation est compatible avec le problme (mme dimension).
* @param Equation, l'quation.
* @since Opale 0.11
*/
public void setEqn(Equation eq)
	{
	if ( dim!= eq.dim())
		{
		System.err.println("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		System.exit(-1);
		}
	eqn = eq;
	}
	
/**
* Associe un schma en temps au problme. 
* @param TimeScheme, le schma en temps.
* @since Opale 0.11
*/
public void setTS(TimeScheme sch)
	{
	this.sch = sch;
	sch.setPb(this);
	}		

public void readFile(RFile f,ODE ode) {}
public void writeFile(WFile f, ODE ode) throws MalformedFileException
	{
	f.writeln("{");
	if (!ode.contains(sch)) throw new MalformedFileException("Erreur  l'criture du fichier : le schma en temps "+sch+" n'existe pas !!");
	f.writeln("dim "+dim);
	f.writeln("sch "+ode.getId(sch));
	f.writeln("}");
	}

public String toString()
	{
	return "Problem of ODE";
	}

public static void main(String[] arg)
	{
	
	Problem pb = new Problem();
	ExplicitEuler ts = new ExplicitEuler();
	pb.setEqn(
		new Equation()
			{
			public int dim() { return 1; }
			public double[] deriv(double t, double[] x)
				{
				double[] y = new double[1];
				y[0]=2;
				return y;
				}
			}
		);

	ts.setNstep(5000);
	pb.setTS(ts);
	pb.solve();
	System.err.println(pb.printSol());
	
	try
	{
	WFile wf = new WFile("essai.txt");
	pb.writeSol(wf);
	wf.writeln("");
	ODE ode = new ODE();
	ode.add(pb,"pb");
	ode.add(ts,"schema");
	ode.writeFile(wf);
	wf.close();
	}
	catch(java.io.IOException e)
	{}	
	catch(MalformedFileException e)
	{System.err.println(e);}	
	}

}


	
